//-*-c++-*-
/**
 Authors: David Auber, Romain Bourqui, Patrick Mary
 from the LaBRI Visualization Team
 Email : auber@tulip-software.org
 Last modification : 13/07/2007 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/

#include <list>
#include <cmath>
#include <typeinfo>
#include <string>

#if (__GNUC__ < 3)
#include <hash_map>
#else
#include <ext/hash_map>
#endif

#include <time.h>
#include <stdio.h>

#include <GL/glu.h>

#include <tulip/LayoutProperty.h>
#include <tulip/DoubleProperty.h>
#include <tulip/StringProperty.h>
#include <tulip/BooleanProperty.h>
#include <tulip/SizeProperty.h>
#include <tulip/IntegerProperty.h>
#include <tulip/ColorProperty.h>
#include <tulip/GraphProperty.h>
#include <tulip/Graph.h>
#include <tulip/TemplateFactory.h>
#include <tulip/DrawingTools.h>
#include <tulip/StlIterator.h>
#include <tulip/ForEach.h>

#include "tulip/GlGraph.h"
#include "tulip/GlTools.h"
#include "tulip/Color.h"
#include "tulip/Coord.h"
#include "tulip/Size.h"
#include "tulip/EpsFunction.h"
#include "tulip/TlpTools.h"
#include "tulip/Glyph.h"
#include "tulip/PluginLoader.h"
#include "tulip/TextRenderer.h"
#include "tulip/OcclusionTest.h"

using namespace std;

//====================================================
#ifdef _WIN32 
#ifdef DLL_EXPORT
tlp::TemplateFactory<tlp::GlyphFactory,tlp::Glyph, tlp::GlyphContext *> *tlp::GlyphFactory::factory;
#endif
#else
tlp::TemplateFactory<tlp::GlyphFactory, tlp::Glyph, tlp::GlyphContext *> *tlp::GlyphFactory::factory;
#endif

static stdext::hash_map<int,std::string>   glyphIdToName;
static stdext::hash_map<std::string, int>  nameToGlyphId;

namespace tlp {
  //====================================================
  std::string GlGraph::glyphName(int id) {
    if (glyphIdToName.find(id)!=glyphIdToName.end())
      return glyphIdToName[id];
    else {
      cerr << __PRETTY_FUNCTION__ << endl;
      cerr << "Invalid glyph id" << endl;
      return string("invalid");
    }
  }
  //====================================================
  int GlGraph::glyphId(std::string name) {
    if (nameToGlyphId.find(name)!=nameToGlyphId.end())
      return nameToGlyphId[name];
    else {
      cerr << __PRETTY_FUNCTION__ << endl;
      cerr << "Invalid glyph name" << endl;
      return 0;
    }
  }
  //====================================================
  const int GlGraph::edgeShapesCount = 3;
  int GlGraph::edgeShapeIds[GlGraph::edgeShapesCount] =
    {BEZIERSHAPE, POLYLINESHAPE, SPLINESHAPE};
  std::string GlGraph::edgeShapeName(int id) {
    switch(id) {
    case POLYLINESHAPE:
      return std::string("Polyline");
    case BEZIERSHAPE:
      return std::string("Bezier Curve");
    case SPLINESHAPE:
      return std::string("Spline Curve");
    default:
      cerr << __PRETTY_FUNCTION__ << endl;
      cerr << "Invalid edge shape id" << endl;
      return std::string("invalid shape id");
    }
  }
  //====================================================
  int GlGraph::edgeShapeId(std::string name) {
    if (name == edgeShapeName(POLYLINESHAPE))
      return POLYLINESHAPE;
    if (name == edgeShapeName(BEZIERSHAPE))
      return BEZIERSHAPE;
    if (name == edgeShapeName(SPLINESHAPE))
      return SPLINESHAPE;
    cerr << __PRETTY_FUNCTION__ << endl;
    cerr << "Invalid edge shape name" << endl;
    return -1;
  }
  //====================================================
  // known label positions
  static std::string labelPositionNames[] = {
    std::string("Center"), std::string("Top"), std::string("Bottom"),
    std::string("Left"), std::string("Right") };

  std::string GlGraph::labelPositionName(int id) {
    if (id > -1 && id < 5)
      return labelPositionNames[id];
    return std::string("invalid label position id");
  }
  //====================================================
  int GlGraph::labelPositionId(std::string name) {
    for (int i = 0; i < 5; i++) {
      if (name == labelPositionNames[i])
	return i;
    }
    cerr << __PRETTY_FUNCTION__ << endl;
    cerr << "Invalid label position name" << endl;
    return -1;
  }
  //====================================================
  void GlGraph::loadGlyphPlugins() {
    Iterator<string> *itS=GlyphFactory::factory->availablePlugins();
    while (itS->hasNext()) {
      string pluginName=itS->next();
      int pluginId=GlyphFactory::factory->objMap[pluginName]->getId();
      glyphIdToName[pluginId]=pluginName;
      nameToGlyphId[pluginName]=pluginId;
    } delete itS;
  }
  //====================================================
  void GlGraph::loadPlugins(PluginLoader *plug) {
    GlyphFactory::initFactory();
    string::const_iterator begin=tlp::TulipPluginsPath.begin();
    string::const_iterator end=begin;
    glyphIdToName.clear();
    nameToGlyphId.clear();
    while (end!=tlp::TulipPluginsPath.end())
      if ((*end)==tlp::PATH_DELIMITER) {
	if (begin!=end)
	  tlp::loadPluginsFromDir(string(begin,end)+"/glyphs", "Glyph", plug);
	++end;
	begin=end;
      } else
	++end;
    if (begin!=end) {
      tlp::loadPluginsFromDir(string(begin,end)+"/glyphs", "Glyph", plug);
    }
    loadGlyphPlugins();
  }
  //====================================================
  GlGraph::GlGraph():
    selectBuf(0),
    elementColor(0), elementLabelColor(0), elementSize(0), elementLabelPosition(0),
    elementShape(0), elementRotation(0), elementSelected(0), elementLabel(0),
    elementLayout(0), elementGraph(0), elementTexture(0),
    _graph(0),
    selectionDL(0),
    arrowDL(0),
    drawNodesIterator(0), drawLabelsIterator(0), drawSelectedLabelsIterator(0),
    drawEdgesIterator(0), drawEdgeLabelsIterator(0), drawEdgeSelectedLabelsIterator(0) 
  {
    GlyphFactory::initFactory();
    initProxies();
    TRACE_EXEC();
    fontRenderer  = new TextRenderer();
    occlusionTest = new OcclusionTest();
  }
  //====================================================
  //define just to forbid copy of the object
  GlGraph::GlGraph(const GlGraph &g) {}
  //====================================================
  GlGraph::~GlGraph() {
    //  cerr << __PRETTY_FUNCTION__ << endl;
    notifyDestroy(this);
    removeObservers();
    if(_graph != 0) _graph->removeObserver(this);
    deleteIterators();
    deleteDisplayLists();
    delete fontRenderer;
    delete occlusionTest;
  }
  //====================================================
  tlp::Graph* GlGraph::getGraph() const {
    return _graph;
  }
  //====================================================
  void GlGraph::centerScene() {
    if (_graph==0) return;
    GlGraphRenderingParameters parameters = getRenderingParameters();
    initProxies();
    Camera camera = parameters.getCamera();
    pair<Coord, Coord> bboxes = tlp::computeBoundingBox(_graph, elementLayout, elementSize, elementRotation);
    Coord maxC = bboxes.first;
    Coord minC = bboxes.second;
  
    double dx = maxC[0] - minC[0];
    double dy = maxC[1] - minC[1];
    double dz = maxC[2] - minC[2];
    camera.center = (maxC + minC) / 2.0;

    if ((dx==0) && (dy==0) && (dz==0))
      dx = dy = dz = 10.0;
  
    camera.sceneRadius = sqrt(dx*dx+dy*dy+dz*dz)/2.0; //radius of the sphere hull of the layout bounding box
    if (camera.sceneRadius < 1E-4) { //default value if there is no node or node on 0 with size 0
      camera = Camera();
    }
    else {
      camera.eyes.set(0, 0, camera.sceneRadius);
      camera.eyes += camera.center;
      camera.up.set(0, 1., 0);
      camera.zoomFactor = 0.5;
    }
    parameters.setCamera(camera);
    setRenderingParameters(parameters);
    //  cerr << __PRETTY_FUNCTION__ << endl;
  }
  //====================================================
  void GlGraph::draw() {
    //cerr << __PRETTY_FUNCTION__ << endl;
    if (_graph==0) {
#ifndef NDEBUG
      cerr << __PRETTY_FUNCTION__ << "===> no graph" << endl;
#endif
      initGlParameter();
      Color background = _renderingParameters.getBackgroundColor();
      glClearColor(background.getRGL(), background.getGGL(), background.getBGL(), 1.0);
      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
      return;
    }
    if (!glIsList(selectionDL))
      buildDisplayLists();

    Observable::holdObservers();
    nodesRenderedAsPoint.setAll(false);
    deleteIterators();
    occlusionTest->reset();
    initGlParameter();
    initLights();
    initProjection();
    initModelView();
    initIterators();

    drawNodes(_graph->numberOfNodes(),drawNodesIterator);
    if (_renderingParameters.isDisplayEdges()) 
      drawEdges(_graph->numberOfEdges(),drawEdgesIterator);
    if (_renderingParameters.isViewNodeLabel())
      drawNodeLabels(_graph->numberOfNodes(),drawSelectedLabelsIterator,true);
    if (_renderingParameters.isViewEdgeLabel())
      drawEdgeLabels(_graph->numberOfEdges(),drawEdgeSelectedLabelsIterator,true);
    if (_renderingParameters.isViewNodeLabel())
      drawNodeLabels(_graph->numberOfNodes(),drawLabelsIterator,false);
    if (_renderingParameters.isViewEdgeLabel())
      drawEdgeLabels(_graph->numberOfEdges(),drawEdgeLabelsIterator,false);


    glTest(__PRETTY_FUNCTION__);

    notifyDraw(this);
    Observable::unholdObservers();
  }
  //====================================================
  bool GlGraph::outputEPS(int size, int doSort, const char *filename) {
    TRACE_EXEC();
    GLfloat *feedbackBuffer;
    GLint returned;
    FILE *file;
    //backup unsuported rendering setting
    GlGraphRenderingParameters backupParam = getRenderingParameters();
    if (_renderingParameters.getFontsType()!=0) {
      _renderingParameters.setViewNodeLabel(false);
      _renderingParameters.setViewEdgeLabel(false);
    }
    feedbackBuffer = (GLfloat *)calloc(size, sizeof(GLfloat));
    glFeedbackBuffer(size, GL_3D_COLOR, feedbackBuffer);
    glRenderMode(GL_FEEDBACK);
    initGlParameter();
    initLights();
    initProjection();
    initModelView();
    Iterator<node> *drawNodesIterator=_graph->getNodes();
    drawNodes(_graph->numberOfNodes(), drawNodesIterator);  
    delete drawNodesIterator;
    Iterator<edge> *drawEdgesIterator=_graph->getEdges();
    if (_renderingParameters.isDisplayEdges())
      drawEdges(_graph->numberOfEdges(), drawEdgesIterator);  
    delete drawEdgesIterator;
    drawNodesIterator=_graph->getNodes();
    if (_renderingParameters.isViewNodeLabel())
      drawNodeLabels(_graph->numberOfNodes(), drawNodesIterator, true);
    delete drawNodesIterator;
    drawNodesIterator=_graph->getNodes();
    if (_renderingParameters.isViewNodeLabel())
      drawNodeLabels(_graph->numberOfNodes(), drawNodesIterator, false);
    delete drawNodesIterator;
    glFlush();
    glFinish();
    returned = glRenderMode(GL_RENDER);
    if (filename) {
      file = fopen(filename, "w");
      if (file)
	{spewWireFrameEPS(file, doSort, returned, feedbackBuffer, "rendereps");} 
      else 
	perror(filename);
    } 
    else 
      {printBuffer(returned, feedbackBuffer);}
    free(feedbackBuffer);
    setRenderingParameters(backupParam);
    return file != 0;
  }
  //====================================================
  unsigned char * GlGraph::getImage() {
    TRACE_EXEC();
    GlGraphRenderingParameters backupParam = getRenderingParameters();
    GlGraphRenderingParameters newParam = getRenderingParameters();
    newParam.setIncrementalRendering(false);
    setRenderingParameters(newParam);
    Vector<int, 4> viewport = _renderingParameters.getViewport();
    unsigned int w = viewport[2];
    unsigned int h = viewport[3];
    unsigned char *image = (unsigned char *)malloc(w*h*3*sizeof(unsigned char));
    draw();
    glFlush();
    glFinish();
    glPixelStorei(GL_PACK_ALIGNMENT,1);
    glReadPixels(viewport[0],viewport[1],viewport[2],viewport[3],GL_RGB,GL_UNSIGNED_BYTE,image);
    glTest(__PRETTY_FUNCTION__);
    setRenderingParameters(backupParam);
    return image;
  }
  //====================================================
  Coord GlGraph::screenTo3DWorld(const Coord &point) {
    Vector<int, 4> viewport = getRenderingParameters().getViewport();
    //Try to find a good z-coordinate for reverse projection
    Coord pScr = projectPoint(Coord(0,0,0), transformMatrix, viewport);
    pScr[0] = (float)viewport[2] - point[0];
    pScr[1] = (float)viewport[3] - point[1] - 1.0;
    MatrixGL tmp(transformMatrix);
    tmp.inverse();
    return unprojectPoint(pScr, tmp, viewport);
  }
  //====================================================
  Coord GlGraph::worldTo2DScreen(const Coord &obj) {
    Vector<int, 4> viewport = getRenderingParameters().getViewport();
    return projectPoint(obj, transformMatrix, viewport);
  }
  //====================================================
  class LessThanNode {
  public:
    DoubleProperty *metric;
    bool operator() (node n1,node n2)  {
      return (metric->getNodeValue(n1) > metric->getNodeValue(n2));
    } 
  };
  //====================================================
  class LessThanEdge {
  public:
    DoubleProperty *metric;
    Graph *sp;
    bool operator() (edge e1,edge e2) {
      return (metric->getNodeValue(sp->target(e1))>metric->getNodeValue(sp->target(e2)));
    } 
  };
  
  //====================================================
  void GlGraph::buildOrderedList() {
    //  cerr << __PRETTY_FUNCTION__ << endl;
    orderedNode.clear();
    orderedEdge.clear();
    if (!_renderingParameters.isElementOrdered()) return;
    DoubleProperty *metric = _graph->getProperty<DoubleProperty>("viewMetric");
    node n;
    forEach(n, _graph->getNodes())
      orderedNode.push_back(n);
    LessThanNode comp;
    comp.metric=metric;
    orderedNode.sort(comp);
    edge e;
    forEach(e, _graph->getEdges()) 
      orderedEdge.push_back(e);
    LessThanEdge comp2;
    comp2.metric = metric;
    comp2.sp = _graph;
    orderedEdge.sort(comp2);
  }
  //====================================================
  /**
   * Management of the graph modification, necessary when we store ordered list
   * of graph elements
   */
  //====================================================
  void GlGraph::addNode (Graph *, const node n) {
    if (_renderingParameters.isElementOrdered())
      orderedNode.push_back(n);
  }
  //====================================================
  void GlGraph::addEdge (Graph *, const edge e) {
    if (_renderingParameters.isElementOrdered())
      orderedEdge.push_back(e);
  }
  //====================================================
  void GlGraph::delNode (Graph *, const node n) {
    if (_renderingParameters.isElementOrdered())
      orderedNode.remove(n);
  }
  //====================================================
  void GlGraph::delEdge (Graph *, const edge e) {
    if (_renderingParameters.isElementOrdered())
      orderedEdge.remove(e);
  }
  //====================================================
  void GlGraph::destroy (Graph *graph) {
    GlGraphRenderingParameters newParam = getRenderingParameters();
    newParam.setGraph(0);
    setRenderingParameters(newParam);
  }
  //====================================================
  /**
   *Management of the iterators on the graph elements
   */
  //====================================================
  void GlGraph::deleteIterators() {
    //  cerr << __PRETTY_FUNCTION__ << endl;
    if (drawNodesIterator != 0) {
      delete drawNodesIterator;
      drawNodesIterator = 0;
    }
    if (drawEdgesIterator != 0) {
      delete drawEdgesIterator;
      drawEdgesIterator = 0;
    }
    if (drawLabelsIterator != 0) {
      delete drawLabelsIterator;
      drawLabelsIterator = 0;
    }
    if (drawSelectedLabelsIterator != 0) {
      delete drawSelectedLabelsIterator;
      drawSelectedLabelsIterator = 0;
    }
    if (drawEdgeLabelsIterator != 0) {
      delete drawEdgeLabelsIterator;
      drawEdgeLabelsIterator = 0;
    }
    if (drawEdgeSelectedLabelsIterator != 0) {
      delete drawEdgeSelectedLabelsIterator;
      drawEdgeSelectedLabelsIterator = 0;
    }
  }
  //====================================================
  void GlGraph::initProxies() {
    if (_graph != NULL) {
      elementRotation = _graph->getProperty<DoubleProperty>("viewRotation");
      elementSelected = _graph->getProperty<BooleanProperty>("viewSelection");
      elementLabel = _graph->getProperty<StringProperty>("viewLabel");
      elementLabelColor = _graph->getProperty<ColorProperty>("viewLabelColor");
      elementLabelPosition = _graph->getProperty<IntegerProperty>("viewLabelPosition");
      elementColor = _graph->getProperty<ColorProperty>("viewColor");
      elementShape = _graph->getProperty<IntegerProperty>("viewShape");
      elementSize = _graph->getProperty<SizeProperty>("viewSize");
      elementLayout = _graph->getProperty<LayoutProperty>(_renderingParameters.getInputLayout() );
      elementGraph = _graph->getProperty<GraphProperty>("viewMetaGraph");
      elementTexture = _graph->getProperty<StringProperty>("viewTexture");
      elementBorderColor = _graph->getProperty<ColorProperty>("viewBorderColor");
      elementBorderWidth = _graph->getProperty<DoubleProperty>("viewBorderWidth");
    }
  }
  //====================================================
  void GlGraph::initIterators() {
    //  cerr << __PRETTY_FUNCTION__ << endl;
    deleteIterators();
    if (_renderingParameters.isElementOrdered()) {
      typedef std::list<node>::const_iterator  ListNodeIt;
      typedef std::list<edge>::const_iterator  ListEdgeIt;
      drawNodesIterator = new tlp::StlIterator<node, ListNodeIt > (orderedNode.begin(), orderedNode.end());
      drawEdgesIterator = new tlp::StlIterator<edge, ListEdgeIt > (orderedEdge.begin(), orderedEdge.end());
      drawLabelsIterator = new StlIterator<node, ListNodeIt > (orderedNode.begin(),orderedNode.end());
      drawSelectedLabelsIterator = new StlIterator<node, ListNodeIt > (orderedNode.begin(),orderedNode.end());
      drawEdgeLabelsIterator = new StlIterator<edge, ListEdgeIt > (orderedEdge.begin(),orderedEdge.end());
      drawEdgeSelectedLabelsIterator = new StlIterator<edge, ListEdgeIt > (orderedEdge.begin(),orderedEdge.end());
    }
    else {
      drawNodesIterator = _graph->getNodes();
      drawEdgesIterator = _graph->getEdges();
      drawLabelsIterator = _graph->getNodes();
      drawSelectedLabelsIterator = _graph->getNodes();
      drawEdgeLabelsIterator = _graph->getEdges();
      drawEdgeSelectedLabelsIterator = _graph->getEdges();
    }
  }
  //====================================================
  // SVG export structures
  struct GraphExporter
  {
    float lineWidth, pointSize;
    float width, height;
    std::map<int,int> nodein, edgein;
    Color fillColor, strokeColor, textColor;
  };

  struct Feedback3DColor
  {
    float x, y, z, r, g, b, a;
  };


  // 
  void GlGraph::initMapsSVG(Graph *g, GraphExporter *ge) {
    register unsigned int gid = g->getId();
    node n;
    forEach(n, g->getNodes()) {
      ge->nodein[n.id] = gid;
    }
    edge e;
    forEach(e, g->getEdges()) {
      ge->edgein[e.id] = gid;
    }
    Graph * sg;
    forEach(sg, g->getSubGraphs()) {
      initMapsSVG(sg, ge);
    }
  }

  void GlGraph::exportHeaderSVG(FILE *file, GraphExporter *ge) {
    initProjection();
    glScalef(1.0f, -1.0f, 1.0f);
    initModelView();
    initGlParameter();
	
    GLfloat clearColor[4], viewport[4];
	
    /* Read back a bunch of OpenGL state to help make the SVG
       consistent with the OpenGL clear color, line width, point
       size, and viewport. */
    glGetFloatv(GL_VIEWPORT, viewport);
    glGetFloatv(GL_COLOR_CLEAR_VALUE, clearColor);
    glGetFloatv(GL_LINE_WIDTH, &ge->lineWidth);
    glGetFloatv(GL_POINT_SIZE, &ge->pointSize);
	
    ge->width = viewport[2] - viewport[0];
    ge->height = viewport[3] - viewport[1];
	
    fprintf(file, "<?xml version=\"1.0\" standalone=\"no\" ?>\n");
    fprintf(file, "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 20010904//EN\" \"http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd\">\n");
    fprintf(file, "<svg width=\"%fpx\" height=\"%fpx\" xmlns=\"http://www.w3.org/2000/svg\">\n",
	    ge->width, ge->height);
    fprintf(file, "\t<!-- Exported from Tulip - plugin made by %s (using OpenGL feedback) -->\n",
	    "OF-JD-NL-SH");
	
    /* Clear the background like OpenGL had it. */
    fprintf(file, "\t<rect x=\"%f\" y=\"%f\" widht=\"%f\" height=\"%f\" fill=\"rgb(%f,%f,%f)\"/>\n",
	    viewport[0], viewport[1], viewport[2], viewport[3], clearColor[0], clearColor[1],
	    clearColor[2]);
  }

  void GlGraph::exportGraphSVG(FILE *file, GLfloat *buffer, Graph *graph,
			       GraphExporter *ge, string indentation) {
    ColorProperty *fillColors = graph->getProperty<ColorProperty>("viewColor");
    ColorProperty *strokeColors = graph->getProperty<ColorProperty>("viewBorderColor");
    ColorProperty *textColors = graph->getProperty<ColorProperty>("viewLabelColor");
    LayoutProperty *layout = graph->getProperty<LayoutProperty>("viewLayout");
    StringProperty *label = graph->getProperty<StringProperty>("viewLabel");
    //GraphProperty *metaGraph = graph->getProperty<GraphProperty>("viewMetaGraph");
    float projectionMatrix[16], modelviewMatrix[16], transformMatrix[16];
	
    fprintf(file, "%s<g id=\"g%d\"><!-- Graph %d -->\n", indentation.c_str(), graph->getId(),
	    graph->getId());

    /* no need because subgraphs is disabled 
       int gid = graph->getId(); */

    if(_renderingParameters.isDisplayEdges()) {
      edge e;
      Iterator<edge> *itE = graph->getEdges();
      forEach(e, graph->getEdges()) {
	/* subgraphs drawing disabled
	   if(ge->edgein[e.id] != gid)
	   continue; */
			
	ge->fillColor = fillColors->getEdgeValue(e);
	ge->strokeColor = strokeColors->getEdgeValue(e);
	ge->textColor = textColors->getEdgeValue(e);
			
	glRenderMode(GL_FEEDBACK);
	initProjection();
	glScalef(1.0f, -1.0f, 1.0f);
	initModelView();
	initGlParameter();
			
	if(_renderingParameters.isViewEdgeLabel()) {
	  glMatrixMode(GL_PROJECTION);
	  glGetFloatv(GL_PROJECTION_MATRIX, projectionMatrix);
	  glMatrixMode(GL_MODELVIEW);
	  glGetFloatv(GL_MODELVIEW_MATRIX, modelviewMatrix);
				
	  glPushMatrix();
	  glLoadIdentity();
	  glMultMatrixf(projectionMatrix);
	  glMultMatrixf(modelviewMatrix);
	  glGetFloatv(GL_MODELVIEW_MATRIX, transformMatrix);
	  glPopMatrix();
	}
			
	drawEdges(1, itE);
			
	// if(_viewLabel)
	// {
	// Iterator<edge> *itE2;
	// itE2 = graph->getEdges();
	// drawEdgeLabels(1, itE2, true);
	// itE2 = graph->getEdges();
	// drawEdgeLabels(1, itE2, false);
	// }
			
	glFlush();
	glFinish();
	GLint count = glRenderMode(GL_RENDER);
			
	fprintf(file, "%s\t<g id=\"e%d\"><!-- Edge %d -->\n", indentation.c_str(), e.id, e.id);
	exportBufferSVG(file, buffer, count, ge, indentation + "\t\t");
	if(_renderingParameters.isViewEdgeLabel()) {
	  Coord coord1 = layout->getNodeValue(graph->source(e));
	  Coord coord2 = layout->getNodeValue(graph->target(e));
	  float x = (coord1.getX() + coord2.getX()) / 2. * transformMatrix[0] +
	    (coord1.getY() + coord2.getY()) / 2. * transformMatrix[4] +
	    (coord1.getZ() + coord2.getZ()) / 2. * transformMatrix[8] + transformMatrix[12];
	  float y = (coord1.getX() + coord2.getX()) / 2. * transformMatrix[1] +
	    (coord1.getY() + coord2.getY()) / 2. * transformMatrix[5] +
	    (coord1.getZ() + coord2.getZ()) / 2. * transformMatrix[9] + transformMatrix[13];
	  // float z = (coord1.getX() + coord2.getX()) / 2. * transformMatrix[2] + (coord1.getY() + coord2.getY()) / 2. * transformMatrix[6] + (coord1.getZ() + coord2.getZ()) / 2. * transformMatrix[10] + transformMatrix[14];
	  x = (x + 1.0f) / (1.0f / _renderingParameters.getCamera().zoomFactor) * ge->width;
	  y = (y + 1.0f) / (1.0f / _renderingParameters.getCamera().zoomFactor) * ge->height;
	  const char *eLabel = label->getEdgeValue(e).c_str();
	  if(eLabel[0] && x >= 0 && x <= ge->width && y >= 0 && y <= ge->height)
	    fprintf(file, "%s\t\t<text x=\"%f\" y=\"%f\" text-anchor=\"middle\" alignment-baseline=\"middle\">%s</text>\n",
		    indentation.c_str(), x, y, eLabel);
	}
	fprintf(file, "%s\t</g>\n", indentation.c_str());
      }
      delete itE;
    }
	
    node n;
    Iterator<node> *itN = graph->getNodes();
    forEach(n, graph->getNodes()){ 
      /* subgraphs drawing disabled
	 if(ge->nodein[n.id] != gid)
	 continue; */

      ge->fillColor = fillColors->getNodeValue(n);
      ge->strokeColor = strokeColors->getNodeValue(n);
      ge->textColor = textColors->getNodeValue(n);
		
      glRenderMode(GL_FEEDBACK);
      initProjection();
      glScalef(1.0f, -1.0f, 1.0f);
      initModelView();
      initGlParameter();
		
      if(_renderingParameters.isViewNodeLabel()) {
	glMatrixMode(GL_PROJECTION);
	glGetFloatv(GL_PROJECTION_MATRIX, projectionMatrix);
	glMatrixMode(GL_MODELVIEW);
	glGetFloatv(GL_MODELVIEW_MATRIX, modelviewMatrix);
			
	glPushMatrix();
	glLoadIdentity();
	glMultMatrixf(projectionMatrix);
	glMultMatrixf(modelviewMatrix);
	glGetFloatv(GL_MODELVIEW_MATRIX, transformMatrix);
	glPopMatrix();
      }
		
      // if(metaGraph->getNodeValue(n) == 0)
      drawNodes(1, itN);
      // else
      // itNDraw->next();
		
      // if(_viewLabel)
      // {
      // Iterator<node> *itN2;
      // itN2 = graph->getNodes();
      // drawNodeLabels(1, itN2, true);
      // itN2 = graph->getNodes();
      // drawNodeLabels(1, itN2, false);
      // }
		
      glFlush();
      glFinish();
      GLint count = glRenderMode(GL_RENDER);
		
      fprintf(file, "%s\t<g id=\"n%d\"><!-- Node %d -->\n", indentation.c_str(), n.id, n.id);
      exportBufferSVG(file, buffer, count, ge, indentation + "\t\t");
      if(_renderingParameters.isViewNodeLabel()) {
	Coord coord = layout->getNodeValue(n);
	float x = coord.getX() * transformMatrix[0] + coord.getY() * transformMatrix[4] +
	  coord.getZ() * transformMatrix[8] + transformMatrix[12];
	float y = coord.getX() * transformMatrix[1] + coord.getY() * transformMatrix[5] +
	  coord.getZ() * transformMatrix[9] + transformMatrix[13];
	// float z = coord.getX() * transformMatrix[2] + coord.getY() * transformMatrix[6] + coord.getZ() * transformMatrix[10] + transformMatrix[14];
	x = (x + 1.0f) / (1.0f / _renderingParameters.getCamera().zoomFactor) * ge->width;
	y = (y + 1.0f) / (1.0f / _renderingParameters.getCamera().zoomFactor) * ge->height;
	const char *nLabel = label->getNodeValue(n).c_str();
	if(nLabel[0] && x >= 0 && x <= ge->width && y >= 0 && y <= ge->height)
	  fprintf(file, "%s\t\t<text x=\"%f\" y=\"%f\" text-anchor=\"middle\" alignment-baseline=\"middle\">%s</text>\n",
		  indentation.c_str(), x, y, nLabel);
      }
      fprintf(file, "%s\t</g>\n", indentation.c_str());
    }
    delete itN;
	
    /* subgraphs drawing disabled
       Graph *sg;
    forEach(sg, graph->getSubGraphs())
    exportGraphSVG(file, buffer, sg, ge, indentation + string("\t")); */
	
    fprintf(file, "%s</g>\n", indentation.c_str());
  }

  void GlGraph::exportBufferSVG(FILE *file, GLfloat *buffer, GLint count,
				GraphExporter *ge, string indentation) {
    int pos = 0;
    while(pos < count) {
      unsigned int token = (unsigned int)(buffer[pos++]);
      switch(token) {
      case GL_PASS_THROUGH_TOKEN:
	// fprintf(file, "%s<!-- Token %f -->\n", indentation.c_str(), buffer[pos]);
	pos++;
	break;
      case GL_POINT_TOKEN: {
	Feedback3DColor *vertex = (Feedback3DColor *)&buffer[pos]; pos += 7;
	fprintf(file, "%s<circle cx=\"%f\" cy=\"%f\" r=\"%f\" fill=\"rgb(%d, %d, %d)\" stroke=\"rgb(%d, %d, %d)\"/>\n",
		indentation.c_str(), vertex->x, vertex->y, ge->pointSize,
		ge->strokeColor.getR(), ge->strokeColor.getG(), ge->strokeColor.getB(),
		ge->strokeColor.getR(), ge->strokeColor.getG(), ge->strokeColor.getB());
	break;
      }
      case GL_LINE_TOKEN:
      case GL_LINE_RESET_TOKEN: {
	Feedback3DColor *vertex1 = (Feedback3DColor *)&buffer[pos]; pos += 7;
	Feedback3DColor *vertex2 = (Feedback3DColor *)&buffer[pos]; pos += 7;
	fprintf(file, "%s<line x1=\"%f\" y1=\"%f\" x2=\"%f\" y2=\"%f\" fill=\"none\" stroke=\"rgb(%d, %d, %d)\"/>\n",
		indentation.c_str(), vertex1->x, vertex1->y, vertex2->x,
		vertex2->y, ge->strokeColor.getR(), ge->strokeColor.getG(), ge->strokeColor.getB());
	break;
      }
      case GL_POLYGON_TOKEN: {
	fprintf(file, "%s<polygon points=\"", indentation.c_str());
	unsigned int nbvertices = (unsigned int)(buffer[pos++]);
	for(unsigned int i = 0; i < nbvertices; i++) {
	  Feedback3DColor *vertex = (Feedback3DColor *)&buffer[pos]; pos += 7;
	  fprintf(file, "%s%f,%f", ((i == 0) ? "" : " "), vertex->x, vertex->y);
	}
	fprintf(file, "\" fill=\"rgb(%d, %d, %d)\" stroke=\"rgb(%d, %d, %d)\"/>\n",
		ge->fillColor.getR(), ge->fillColor.getG(), ge->fillColor.getB(),
		ge->fillColor.getR(), ge->fillColor.getG(), ge->fillColor.getB());
	break;
      }
      case GL_BITMAP_TOKEN:
      case GL_DRAW_PIXEL_TOKEN:
      case GL_COPY_PIXEL_TOKEN:
	// fprintf(file, "%s<!-- Unsupported feature -->\n", indentation.c_str());
	pos += 7;
	break;
      default:
	// fprintf(file, "%s<!-- Error -->\n", indentation.c_str());
	break;
      }
    }
  }

  bool GlGraph::outputSVG(int size, const char *filename) {
    TRACE_EXEC();
    GLfloat *buffer;
    GraphExporter ge;
    FILE *file; // _file = file;
	
    // backup unsuported rendering setting
    // bool saveLabelState = _viewLabel;
 
    buffer = (GLfloat *)calloc(size, sizeof(GLfloat));
    glFeedbackBuffer(size, GL_3D_COLOR, buffer);
    if(filename) {
      /* subgraphs drawing disabled
	 initMapsSVG(_renderingParameters.getGraph(), &ge); */
      file = fopen(filename, "w");
      if (file) {
	exportHeaderSVG(file, &ge);
	exportGraphSVG(file, buffer, _renderingParameters.getGraph(), &ge, string("\t"));
	fprintf(file, "</svg>\n");
	fclose(file);
      } else
	perror(filename);
    } 
    free(buffer);
    
    // restore unsuported rendering setting
    // _viewLabel = saveLabelState;
    return file != 0;
  }
}
