//-*-c++-*-
/**
 Authors: David Auber, Romain Bourqui, Patrick Mary
 from the LaBRI Visualization Team
 Email : auber@tulip-software.org
 Last modification : 13/07/2007 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <GL/glu.h>
#include <cmath>

#if (__GNUC__ < 3)
#include <hash_map>
#else
#include <ext/hash_map>
#endif

#include "tulip/GlGraph.h"
#include "tulip/Glyph.h"
#include "tulip/GlTools.h"

using namespace std;
using namespace tlp;

const Color colorSelect2= Color(255, 102, 255, 255);
const Color colorMeta= Color(79, 36, 80, 255);
//====================================================
void SetMat(const Color &C) {
  float color[4];
  color[0]=((float)C.getR())/255.0;
  color[1]=((float)C.getG())/255.0;
  color[2]=((float)C.getB())/255.0;
  color[3]=1.0;
  glMaterialfv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color );
}
//=======================================================
//Calcul 3D,
//=======================================================
//Calcul de la matrice de transformation pour 
//le positionnement des flches sur les artes
MatrixGL GlGraph::makeArrowMatrix(const Coord &A, const Coord &B) {
  MatrixGL matrix;

  //Vecteur AB
  Vector<float, 3> vAB;
  //Vecteur V
  Vector<float, 3> vV;
  //Vecteur W
  Vector<float, 3> vW;
  
  vAB = B - A;
  float nAB; //|AB|
  nAB = vAB.norm();
  if (fabs(nAB) > 1E-6)
    vAB /= nAB;

  //vAB * vV = xAB * xV + yAB*yV + zAB * zV = |AB| * |V| * cos(alpha) = 0;
  if (fabs(vAB[2]) < 1E-6) {
    vV[0] = 0; vV[1] = 0; vV[2] = 1.0;
  }
  else 
    if (fabs(vAB[1]) < 1E-6) {
      vV[0] = 0; vV[1] = 1.0; vV[2] = 0;
    }
    else {
      vV[0] = 0;
      vV[1] = 1./vAB[1];
      vV[2] = -1./vAB[2];
      vV /= vV.norm();
    }

  vW = vAB ^ vV;
  float nW = vW.norm();
  if (fabs(nW) > 1E-6)
    vW /= nW;
 
  for (unsigned int i = 0; i < 3; ++i) {
    matrix[0][i] = vW[i];
    matrix[1][i] = vV[i];
    matrix[2][i] = vAB[i];
    matrix[3][i] = B[i];
  }
  matrix[0][3]=0; 
  matrix[1][3]=0;
  matrix[2][3]=0;
  matrix[3][3]=1;

  return matrix;
}
//========================================================
//Common display list
//========================================================
void delList(GLuint dl) {
  if (glIsList(dl))  
    glDeleteLists(dl,1);
  glTest(__PRETTY_FUNCTION__);
}
//====================================================
void GlGraph::deleteDisplayLists() {
  if (glIsList(arrowDL))
    delList(arrowDL);
  if (glIsList(selectionDL))
    delList(selectionDL);
  glTest(__PRETTY_FUNCTION__);
}
//==================================================
///Build the open gl display list
void GlGraph::buildDisplayLists() {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << " (" << (int)this << ")" << endl;
#endif

  glNewList(arrowDL = glGenLists(1)    , GL_COMPILE ); {
    tlp::solidCone();
  } glEndList();  
  glNewList(selectionDL = glGenLists(1), GL_COMPILE); {
    glPushAttrib(GL_ALL_ATTRIB_BITS);
    glDisable(GL_LIGHTING);
    glDepthFunc(GL_LEQUAL);
    glLineWidth(3);
    glColor4ub(colorSelect2[0], colorSelect2[1],  colorSelect2[2], colorSelect2[3]); 
    tlp::cube(GL_LINE_LOOP);
    glPopAttrib();
  } glEndList();

  glTest(__PRETTY_FUNCTION__);

  GlyphContext gc = GlyphContext(&_graph, this);
  glyphs.setAll(0);
  Iterator<string> *itS = GlyphFactory::factory->availablePlugins();
  while (itS->hasNext()) {
    string glyphName=itS->next();
    //    cerr << "loading : " << glyphName << endl;
    Glyph *newGlyph = GlyphFactory::factory->getPluginObject(glyphName, &gc);
    //    cerr << newGlyph << endl;
    //if (GlyphFactory::factory->objMap[glyphName]->getId() == 0) 
    //  glyphs.setAll(newGlyph);
    //    cerr << " id:" << GlyphFactory::factory->objMap[glyphName]->getId() << endl;
    glyphs.set(GlyphFactory::factory->objMap[glyphName]->getId(), newGlyph);
    //    cerr << glyphs.get(GlyphFactory::factory->objMap[glyphName]->getId()) << endl;
  } delete itS;

  glTest(__PRETTY_FUNCTION__);
}
