//-*-c++-*-
/**
 Authors: David Auber, Romain Bourqui, Patrick Mary
 from the LaBRI Visualization Team
 Email : auber@tulip-software.org
 Last modification : 13/07/2007 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/

/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 06/09/2005
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/

#ifndef Tulip_GLGRAPH_WIDGET_H
#define Tulip_GLGRAPH_WIDGET_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <tulip/GlADComposite.h>
#include <tulip/Graph.h>
#include <tulip/GlGraph.h>

#if (QT_REL == 3)
#include <qgl.h>
#include <qpoint.h>
#else
#include <QtOpenGL/qgl.h>
#include <QtCore/qpoint.h>
#include "tulip/Qt3ForTulip.h"
#endif

#include "tulip/GWInteractor.h"

//class QTextView;

/** \addtogroup Tulip_Widgets */ 
/*@{*/
class TLP_QT_SIMPLE_SCOPE GlGraphWidget : public QGLWidget, public tlp::GlGraph {
  Q_OBJECT;
  
public:
  GlGraphWidget(QWidget *parent=NULL, const char *name=NULL);
  ~GlGraphWidget();
  
  
  /**************************************
   * inherited methods overloading
   **************************************/
  /** \brief select nodes and edges in a region of the screen
   *
   *  select all nodes and edges lying in the area of the screen of given width and height,
   *  and with its upper-left corner at (x,y)
   *  \param sNode filled by the method with the nodes found in the region
   *  \param sEdge filled by the method with the edges found in the region
   */
  void doSelect(const int x, const int y, 
		const int width, const int height, 
		std::vector<node> &sNode, std::vector<edge> &sEdge);
  /** \brief select a node or edge at a point
   *  select either a node or edge at point (x,y)
   *  \param type tells what has been found: NODE, EDGE
   *  \return true if something has been found, false otherwise
   */
  bool doSelect(const int x, const int y, 
		tlp::ElementType &type, 
		node &, edge &);
  /** 
   *  Take a snapshot of the Widget
   *  \return an array of dimension width*height*3 char (8bits per color RGB).
   *   The pointer has to be freed after (with free, not delete)
   **/
  unsigned char* getImage();
  // EPS output of a GlGraph
  bool outputEPS(int size, int doSort, const char *filename);
  // SVG output of a GlGraph
  bool outputSVG(int size, const char* filename);

  /**************************************
   * Management of threads
   **************************************/
  bool  isDrawing();
  void  stopDrawing();
  void  waitDrawing();

  /*******************************
   * Augmented display management
   *******************************/
  void  addGlAugmentedDisplay(tlp::GlAugmentedDisplay *augmentedDisplay, const string &name);
  void  removeGlAugmentedDisplay(tlp::GlAugmentedDisplay *augmentedDisplay);
  void  removeGlAugmentedDisplay(const string &name);
  tlp::GlAugmentedDisplay* findGlAugmentedDisplay(const std::string &name);
  /**
   * Function to do picking on augmented displays.  It just calls
   * doSelect on the GlADComposite instance.
   */
  bool doSelectAugmentedDisplay (const int x, const int y, const int width, const int height,
				 std::vector <std::pair <string, tlp::GlAugmentedDisplay *> >  &pickedAugmentedDisplays);
  /**
   * Function to do picking on augmented displays.  It just calls
   * doSelect on the GlADComposite instance with a small window of 
   * twelve pixels.
   */
  bool doSelectAugmentedDisplay (const int x, const int y,
				 std::vector <std::pair <string, tlp::GlAugmentedDisplay *> > &pickedAugmentedDisplays);

  /**
   * Gets the list of augmented displays.
   */
  const stdext::hash_map<std::string, GlAugmentedDisplay*> * getAugmentedDisplays();
  // allows to iterate on installed interactors
  Iterator<GWInteractor *> *getInteractors() const;
  virtual QImage grabFrameBuffer(bool withAlpha = false);


public slots:
  /** 
   * Draw the graph, the augmented dispaly and the interactors
   */
  void draw();
  /**
   * That function is given for optimisation purpose. If the hardware enable it,
   * it enables to redraw only the Augmented display and the interactors and not the graph
   * it is really usefull for interactors such as zoom box etc..
   * Warning, if you change the graph or the porperties of element (Colors, size, etc...)
   * applying that fonction will not display the change, in that case, use the draw function.
   */
  void redraw();

  void closeEvent(QCloseEvent *e);
  /// install a clone of the interactor as event filter and assign the returned id
  GWInteractor::ID pushInteractor(tlp::GWInteractor *interactor);
  /// remove the last added interactor from the event filters list and delete it
  void popInteractor();
  /// remove the interactor with id from the event filters list and delete it
  void removeInteractor(tlp::GWInteractor::ID id);
  ///  remove all interactors and delete them, push a new one if any
  GWInteractor::ID resetInteractors(tlp::GWInteractor *interactor = NULL);
  /// remove all iteractors and delete them, then install clones of the interactors
  vector<GWInteractor::ID> resetInteractors(const std::vector<tlp::GWInteractor *>&interactors);

signals:
  void closing(GlGraphWidget *, QCloseEvent *);
  void graphRedrawn(GlGraphWidget *glWidget);

protected slots:
  void paintEvent( QPaintEvent* );

private:
  void  updateGL();
  void  glInit();
  void  glDraw();
  void  paintGL();
  void  resizeGL(int w, int h);
  void  setupOpenGlContext();
  void  initializeGL();
  void  drawInteractors();

  tlp::GlADComposite *_composite;
  bool _firstStepOfIncrementalRendering;
  QRegion _visibleArea;
  GWInteractor::ID _id;
  std::vector<GWInteractor *> _interactors;

};
/*@}*/
#endif
