<?php
/**
 * @file
 * Tweeper - a Twitter to RSS web scraper.
 *
 * Copyright (C) 2013-2015  Antonio Ospite <ao2@ao2.it>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

require_once 'Symfony/Component/Serializer/autoload.php';

use Symfony\Component\Serializer\Serializer;
use Symfony\Component\Serializer\Encoder\XmlEncoder;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;

date_default_timezone_set('UTC');

/**
 * Scrape supported websites and perform conversion to RSS.
 */
class Tweeper {

  private static $userAgent = "Mozilla/5.0 (Windows NT 6.1; rv:22.0) Gecko/20130405 Firefox/22.0";

  /**
   * Constructor sets up {@link $generate_enclosure}.
   */
  public function __construct($generate_enclosure = FALSE) {
    $this->generate_enclosure = $generate_enclosure;
  }

  /**
   * Convert numeric Epoch to the date format expected in a RSS document.
   */
  public static function epochToRssDate($timestamp) {
    if (!is_numeric($timestamp) || is_nan($timestamp)) {
      $timestamp = 0;
    }

    return gmdate(DATE_RSS, $timestamp);
  }

  /**
   * Convert generic date string to the date format expected in a RSS document.
   */
  public static function strToRssDate($date) {
    $timestamp = strtotime($date);
    if (FALSE === $timestamp) {
      $timestamp = 0;
    }

    return Tweeper::epochToRssDate($timestamp);
  }

  /**
   * Convert string to UpperCamelCase.
   */
  public static function toUpperCamelCase($str, $delim = ' ') {
    $str_upper = ucwords($str, $delim);
    $str_camel_case = str_replace($delim, '', $str_upper);
    return $str_camel_case;
  }

  /**
   * Get the contents from a URL.
   */
  private static function getUrlContents($url) {
    $ch = curl_init($url);
    curl_setopt_array($ch, array(
      CURLOPT_HEADER => FALSE,
      // Follow http redirects to get the real URL.
      CURLOPT_FOLLOWLOCATION => TRUE,
      CURLOPT_RETURNTRANSFER => TRUE,
      CURLOPT_SSL_VERIFYHOST => FALSE,
      CURLOPT_SSL_VERIFYPEER => FALSE,
      CURLOPT_HTTPHEADER => array('Accept-language: en'),
      CURLOPT_USERAGENT => Tweeper::$userAgent,
    ));
    $contents = curl_exec($ch);
    curl_close($ch);

    return $contents;
  }

  /**
   * Get the headers from a URL.
   */
  private static function getUrlInfo($url) {
    $ch = curl_init($url);
    curl_setopt_array($ch, array(
      CURLOPT_HEADER => TRUE,
      CURLOPT_NOBODY => TRUE,
      // Follow http redirects to get the real URL.
      CURLOPT_FOLLOWLOCATION => TRUE,
      CURLOPT_RETURNTRANSFER => TRUE,
      CURLOPT_SSL_VERIFYHOST => FALSE,
      CURLOPT_SSL_VERIFYPEER => FALSE,
      CURLOPT_USERAGENT => Tweeper::$userAgent,
    ));
    curl_exec($ch);
    $url_info = curl_getinfo($ch);
    curl_close($ch);

    return $url_info;
  }

  /**
   * Generate an RSS <enclosure/> element.
   */
  public static function generateEnclosure($url) {
    $supported_content_types = array(
      "application/ogg",
      "audio/aac",
      "audio/mp4",
      "audio/mpeg",
      "audio/ogg",
      "audio/vorbis",
      "audio/wav",
      "audio/webm",
      "audio/x-midi",
      "image/gif",
      "image/jpeg",
      "image/png",
      "video/avi",
      "video/mp4",
      "video/mpeg",
      "video/ogg",
    );

    $url_info = Tweeper::getUrlInfo($url);

    $supported = in_array($url_info['content_type'], $supported_content_types);
    if (!$supported) {
      error_log("Unsupported enclosure content type \"" . $url_info['content_type'] . "\" for URL: " . $url_info['url']);
      return '';
    }

    // The RSS specification says that the enclosure element URL must be http.
    // See http://sourceforge.net/p/feedvalidator/bugs/72/
    $http_url = preg_replace("/^https/", "http", $url_info['url']);

    $dom = new DOMDocument();
    $enc = $dom->createElement('enclosure');
    $enc->setAttribute('url', $http_url);
    $enc->setAttribute('length', $url_info['download_content_length']);
    $enc->setAttribute('type', $url_info['content_type']);

    return $enc;
  }

  /**
   * Mimic the message from libxml.c::php_libxml_ctx_error_level()
   */
  private static function logXmlError($error) {
    $output = "";

    switch ($error->level) {
      case LIBXML_ERR_WARNING:
        $output .= "Warning $error->code: ";
        break;

      case LIBXML_ERR_ERROR:
        $output .= "Error $error->code: ";
        break;

      case LIBXML_ERR_FATAL:
        $output .= "Fatal Error $error->code: ";
        break;
    }

    $output .= trim($error->message);

    if ($error->file) {
      $output .= " in $error->file";
    }
    else {
      $output .= " in Entity,";
    }

    $output .= " line $error->line";

    error_log($output);
  }

  /**
   * Convert json to XML.
   */
  private static function jsonToXml($json, $root_node_name) {
    // Apparently the ObjectNormalizer used afterwards is not able to handle
    // the stdClass object created by json_decode() with the default setting
    // $assoc = false; so use $assoc = true.
    $data = json_decode($json, $assoc = TRUE);
    if (!$data) {
      return NULL;
    }

    $encoder = new XmlEncoder();
    $normalizer = new ObjectNormalizer();
    $serializer = new Serializer(array($normalizer), array($encoder));

    $serializer_options = array(
      'xml_encoding' => "UTF-8",
      'xml_format_output' => TRUE,
      'xml_root_node_name' => $root_node_name,
    );

    $xml_data = $serializer->serialize($data, 'xml', $serializer_options);
    if (!$xml_data) {
      trigger_error("Cannot serialize data", E_USER_ERROR);
      return NULL;
    }

    return $xml_data;
  }

  /**
   * Convert the Instagram content to XML.
   */
  private function getXmlInstagramCom($html) {
    // Extract the json data from the html code.
    $json_match_expr = '/window._sharedData = (.*);/';
    $ret = preg_match($json_match_expr, $html, $matches);
    if ($ret !== 1) {
      trigger_error("Cannot match expression: $json_match_expr\n", E_USER_ERROR);
      return NULL;
    }

    return Tweeper::jsonToXml($matches[1], 'instagram');
  }

  /**
   * Make the Facebook HTML processable.
   */
  private function preprocessHtmlFacebookCom($html) {
    $html = str_replace('<!--', '', $html);
    $html = str_replace('-->', '', $html);
    return $html;
  }

  /**
   * Convert the HTML retrieved from the site to XML.
   */
  private function htmlToXml($html, $host) {
    $xmlDoc = new DOMDocument();

    // Handle warnings and errors when loading invalid HTML.
    $xml_errors_value = libxml_use_internal_errors(TRUE);

    // If there is a host-specific method to get the XML data, use it!
    $get_xml_host_method = 'getXml' . Tweeper::toUpperCamelCase($host, '.');
    if (method_exists($this, $get_xml_host_method)) {
      $xml_data = call_user_func_array(array($this, $get_xml_host_method), array($html));
      $xmlDoc->loadXML($xml_data);
    }
    else {
      $xmlDoc->loadHTML($html);
    }

    foreach (libxml_get_errors() as $xml_error) {
      Tweeper::logXmlError($xml_error);
    }
    libxml_clear_errors();
    libxml_use_internal_errors($xml_errors_value);

    return $xmlDoc;
  }

  /**
   * Load a stylesheet if the web site is supported.
   */
  private function loadStylesheet($host) {
    $stylesheet = "file://" . __DIR__ . "/rss_converter_" . $host . ".xsl";
    if (FALSE === file_exists($stylesheet)) {
      trigger_error("Conversion to RSS not supported for $host ($stylesheet not found)", E_USER_ERROR);
      return NULL;
    }

    $stylesheet_contents = Tweeper::getUrlContents($stylesheet);

    $xslDoc = new DOMDocument();
    $xslDoc->loadXML($stylesheet_contents);

    $xsltProcessor = new XSLTProcessor();
    $xsltProcessor->registerPHPFunctions();
    $xsltProcessor->setParameter('', 'generate-enclosure', $this->generate_enclosure);
    $xsltProcessor->importStylesheet($xslDoc);

    return $xsltProcessor;
  }

  /**
   * Convert the site content to RSS.
   */
  public function tweep($src_url) {
    $url = parse_url($src_url);
    if (FALSE === $url || empty($url["host"])) {
      trigger_error("Invalid URL: $src_url", E_USER_ERROR);
      return NULL;
    }

    // Strip the leading www. to be more forgiving on input URLs.
    $host = preg_replace('/^www\./', '', $url["host"]);

    $xsltProcessor = $this->loadStylesheet($host);
    if (NULL === $xsltProcessor) {
      return NULL;
    }

    $html = Tweeper::getUrlContents($src_url);
    if (FALSE === $html) {
      return NULL;
    }

    $preprocess_html_host_method = 'preprocessHtml' . Tweeper::toUpperCamelCase($host, '.');
    if (method_exists($this, $preprocess_html_host_method)) {
      $html = call_user_func_array(array($this, $preprocess_html_host_method), array($html));
    }

    $xmlDoc = $this->htmlToXml($html, $host);
    if (NULL === $xmlDoc) {
      return NULL;
    }

    $output = $xsltProcessor->transformToXML($xmlDoc);

    if (FALSE === $output) {
      trigger_error('XSL transformation failed.', E_USER_ERROR);
      return NULL;
    }
    return $output;
  }

}

/**
 * Check if the script is being run from the command line.
 */
function is_cli() {
  return (php_sapi_name() === "cli");
}

/**
 * Show the script usage.
 */
function usage($argv) {
  if (is_cli()) {
    $usage = "{$argv[0]} [-e|-h|--help] <src_url>\n";
  }
  else {
    $usage = htmlentities("{$_SERVER['SCRIPT_NAME']}?src_url=<src_url>&generate_enclosure=<0|1>");
  }

  return "usage: $usage";
}

/**
 * Parse command line options.
 */
function parse_options_cli($argv, $argc) {
  $options = array(
    'generate_enclosure' => FALSE,
  );

  if ($argc < 2) {
    return $options;
  }

  $cli_options = getopt("eh", array("help"));
  foreach ($cli_options as $opt => $val) {
    switch ($opt) {
      case 'e':
        $options['generate_enclosure'] = TRUE;
        break;

      case 'h':
      case 'help':
        echo usage($argv);
        exit(0);

      default:
        fwrite(STDERR, usage($argv));
        exit(1);
    }
  }

  $options['src_url'] = $argv[count($cli_options) + 1];

  return $options;
}

/**
 * Parse options passed from a query string.
 */
function parse_options_query_string() {
  $options = array(
    'generate_enclosure' => FALSE,
  );

  if (isset($_GET['src_url'])) {
    $options['src_url'] = $_GET['src_url'];
  }

  if (isset($_GET['generate_enclosure'])) {
    $options['generate_enclosure'] = $_GET['generate_enclosure'] == 1;
  }

  return $options;
}

if (is_cli()) {
  $options = parse_options_cli($argv, $argc);
  $error_stream = fopen('php://stderr', 'w');
}
else {
  $options = parse_options_query_string();
  $error_stream = fopen('php://output', 'w');
}

if (!isset($options['src_url'])) {
  fwrite($error_stream, usage(is_cli() ? $argv : NULL));
  exit(1);
}

$tweeper = new Tweeper($options['generate_enclosure']);
echo $tweeper->tweep($options['src_url']);
