/* twpsk  - PSK31 for Linux with a Motif interface
 * Copyright (C) 1999-2014 Ted Williams WA0EIR 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
 * Version: 4.1 - Feb 2014
 *
 * Portions derived from:
 * Soundcard-based implementation of the PSK31 HF keyboard-to-keyboard mode
 * Copyright (C) 1998  Hansi Reiser, DL9RDZ
 */

#include "GUI.h"
#include "server/server.h"
#include "twpskWids.h"
#include "decoderWids.h"
#include "twpskCB.h"
#include "twpskScope.h"
#include "twpskWF.h"
#include "twpskHelp.h"
#include "decoderWids.h"
#include "callbox.h"

#include <grp.h>
#include <signal.h>
#include <dirent.h>

#include <../config.h>

#if (MAKE_ICON == 1) && (HAVE_X11_XPM_H == 1) && (HAVE_LIBXPM == 1)
   #include <X11/xpm.h>
   #include <icons/twpsk.xpm>
   Pixmap pixmap, mask;
   XpmAttributes pix_attributes;
   XpmColorSymbol transparentColor[1] = {{NULL, (char *)"none", 0}};
#endif

int OPTnew = 0;   // todo: rausdamit (-> filter and sync alg selection for RX)

/* Globals */
int use_stereo;
char dispStr[200] = "";
XtAppContext ac;
AppRes appRes;
Widget shell;
Wids pskwids;
DecoderWid decoderWids[MAX_DECODERS];
Position winlocs[MAX_DECODERS][2];
Scope scope;
Disp disp;
Help help;
TwpskCB twpskCB;

/* DL9RDZ */
#if 0
#include "catWids.h"
Cat catwids;
#endif

#define DO_SPEAK 0                 /* comment */
#if DO_SPEAK == 1
void mkWord (char ch)
{
   static char word[20];
//   char cmd[200] = "espeak -v en-us+f4 -s 150 \"";
   static int  i;

   word[i] = ch;

   if (ch == ' ' || ch == '\n')
   {
      word[i] = '\0';
      strcat (cmd, word);
      strcat (cmd, "\"");
      strcat (cmd, " &");
      system (cmd);
//fprintf (stdout, "%s\n", word);
//printf ("%s\n", cmd);
      i = 0;
      return;
   }
   i++;
}
#endif

/*
 * main - main function for twpsk
 */
int main (int argc, char *argv[])
{
   int height = 0, width = 0, i = 0;
   int rtn;

   XtResource appRes_desc[] =
   {
      {
      (char *) XmNbuttonNames,                    /* Resource Name */ 
      (char *) XmCButtonNames,                    /* Resource Class */
      XmRStringTable,                    /* Resource Data Type */
      sizeof (String *),                 /* Size of Resource */
      XtOffsetOf (AppRes, buttonNames),  /* Offset into Struct */
      XmRString,                         /* Default Type */
      (XtPointer) "-1"                   /* Default Value */
      },

      {
      (char *) XmNxmitHighlight,                  /* Same as above */
      (char *) XmCXmitHighlight,
      XmRInt,
      sizeof (int),
      XtOffsetOf (AppRes, xmitHighlight),
      XtRInt,
      (XtPointer) 0
      },

      {
      (char *) XmNrxFreq,                         /* Same as above */
      (char *) XmCRxFreq,
      XmRFloat,
      sizeof (float),
      XtOffsetOf (AppRes, rxFreq),
      XtRFloat,
      (XtPointer) 0
      },

      {
      (char *) XmNtxFreq,                         /* Same as above */
      (char *) XmCTxFreq,
      XmRFloat,
      sizeof (float),
      XtOffsetOf (AppRes, txFreq),
      XtRFloat,
      (XtPointer) 0
      },

      {
      (char *) XmNafc,                            /* Same as above */
      (char *) XmCAfc,
      XmRInt,
      sizeof (int),
      XtOffsetOf (AppRes, afc),
      XtRInt,
      (XtPointer) 0
      },

      {
      (char *) XmNnet,                            /* Same as above */
      (char *) XmCNet,
      XmRInt,
      sizeof (int),
      XtOffsetOf (AppRes, net),
      XtRInt,
      (XtPointer) 0
      },

      {
      (char *) XmNcall,                           /* Same as above */
      (char *) XmCCall,
      XmRString,
      sizeof (char),
      XtOffsetOf (AppRes, call),
      XtRImmediate,
      (XtPointer) " ",
      },

      {
      (char *) XmNzero,                           /* Same as above */
      (char *) XmCZero,
      XmRInt,
      sizeof (int),
      XtOffsetOf (AppRes, zero),
      XtRImmediate,
      (XtPointer) 0
      },

      {
      (char *) XmNserDev,                         /* Same as above */
      (char *) XmCSerDev,
      XmRString,
      sizeof (char),
      XtOffsetOf (AppRes, serDev),
      XtRImmediate,
      (XtPointer) "/dev/ttyS1",
      },

      {
      (char *) XmNptt,                            /* Same as above */
      (char *) XmCPtt,
      XmRInt,
      sizeof (int),
      XtOffsetOf (AppRes, ptt),
      XtRImmediate,
      (XtPointer) 0
      },

      {
      (char *) XmNcallBox,                        /* Same as above */
      (char *) XmCCallBox,
      XmRInt,
      sizeof (int),
      XtOffsetOf (AppRes, callBox),
      XtRImmediate,
      (XtPointer) 0
      },

      {
      (char *) XmNpostamble,                      /* Same as above */
      (char *) XmCPostamble,
      XmRInt,
      sizeof (int),
      XtOffsetOf (AppRes, postamble),
      XtRImmediate,
      (XtPointer) 1
      },

      {
      (char *) XmNsampsize,                      /* Same as above */
      (char *) XmCSampsize,
      XmRInt,
      sizeof (int),
      XtOffsetOf (AppRes, sampsize),
      XtRImmediate,
      (XtPointer) 8000
      }
   };

   /*
    * Create the shell, register a callback for the WM Close button,
    * catch the INT, QUIT, and TERM signals, and read the resource file
    * if Twpsk was found.
    */
   shell = XtVaAppInitialize (&ac, "Twpsk", NULL, 0, &argc, argv, NULL,
      XmNmwmFunctions, MWM_FUNC_ALL | MWM_FUNC_CLOSE,
      XmNtitle, "TWPSK",
      NULL);

#if (MAKE_ICON == 1) && (HAVE_X11_XPM_H == 1) && (HAVE_LIBXPM == 1)
   /*
    * then create pixmap for applicaton Icon
    */
   pix_attributes.closeness = 40000;
   pix_attributes.valuemask = XpmColorSymbols | XpmCloseness;
   pix_attributes.colorsymbols = transparentColor;
   pix_attributes.numsymbols = 1;

   XpmCreatePixmapFromData (XtDisplay(shell),
      DefaultRootWindow (XtDisplay (shell)),
      (char **)twpsk_xpm, &pixmap, &mask, &pix_attributes);

   XtVaSetValues (shell,
      XmNiconPixmap, pixmap,
      NULL);
#endif

   /*
    * Get the app resources from the resource file
    */
   XtGetApplicationResources (shell, &appRes, appRes_desc,
      XtNumber(appRes_desc), NULL, 0);

   /*
    * Check to see if we are running on the default value for
    * buttonNames ("-1").  If so, Twpsk was probably not found,
    * so exit gracefully.
    */
   if (strcmp (appRes.buttonNames[0], "-1") == 0)
   {
      XtRealizeWidget (shell);
      errorDiag (shell, (char *) "The resource file, Twpsk, is missing\n"
                        "This is an installation problem.  Did\n"
                        "you install twpsk as root?\n\n"
                        "Please send bug reports to \n"
                         PACKAGE_BUGREPORT, !CANCEL_BTN, ERR_DIAG);
      fprintf (stderr, "twpsk: Twpsk resource file missing\n");
      XtAppMainLoop(ac);
      exit (1);        /* Never gets here */
   }

   /*
    * Read parameters from ~/.twpskDir/.twpsk.dat 
    * Don't go into the AppLoop, just continue on blindly
    */
   rtn = iniProc ('r');  /* read window locations/channels */
   if (rtn == FALSE)
   {
      XtRealizeWidget (shell);
      errorDiag (shell, (char *) "iniProc failed.  Run twpsk from a terminal\n"
                        "to get more error information.\n\n"\
                        "Please send bug reports to\n"
                         PACKAGE_BUGREPORT, !CANCEL_BTN, ERR_DIAG);
      fprintf (stderr, "iniProc failed. Run twpsk from a terminal\n");
   }
   /*
    * initialize server
    */
   rtn = server_main ( appRes.ptt ? appRes.serDev : NULL, (char *)PKG_DATA_DIR);
   if (rtn == FALSE)
   {
      XtRealizeWidget (shell);
      errorDiag (shell, (char *) "The server fail to initialize.\n"
                        "Run twpsk from a terminal to get\n"
                        "more error information.\n\n"\
                        "Please send bug reports to\n" PACKAGE_BUGREPORT,
                         !CANCEL_BTN, ERR_DIAG);
      fprintf (stderr, "server initialization failed\n");
      XtAppMainLoop (ac);
      exit (1);        /* Never gets here */
   }

   /*
    * Create widgets
    * Output lesstif/openMotif info and
    * make icon
    */
   pskwids.buildWidgets (shell, &appRes);  

   /*
    * Any info messages have been added to dispStr and can be printed now
    */
   pskwids.addText (dispStr);

   #ifdef XmVERSION_STRING
   extern char _XmVersionString[];
   sprintf (dispStr, (char *) "Compiled with %s\n", XmVERSION_STRING + 4);
   pskwids.addText (dispStr);

   sprintf (dispStr, (char *) "Running  with %s\n", _XmVersionString + 4); 
   pskwids.addText (dispStr);
   #endif

   /* print some resources to the tx Text window */
   pskwids.addText ( (char *) "Twpsk Application Resources\n   Call:  ");
   pskwids.addText (appRes.call);

   pskwids.addText ((char *) "\n   File Names:\n");
   i = 0;
   while (appRes.buttonNames[i] != NULL)
   {
      sprintf (dispStr, "      %s%s\n", appRes.buttonNames[i],
               appRes.buttonNames[i+1] == NULL ? "":",");
      pskwids.addText (dispStr);
      i++;
   }

   sprintf (dispStr, "   PTT Device:  %s\n",
            appRes.ptt == 1 ? appRes.serDev : "disabled");
   pskwids.addText (dispStr);

   /*
    * trap some signals
    */  
   //wm_delete_win = XmInternAtom (XtDisplay(shell), "WM_DELETE_WINDOW", False);
   //XmAddWMProtocolCallback (shell, wm_delete_win, quitCB, NULL);

   signal (SIGINT,  gotSig);
   signal (SIGQUIT, gotSig);
   signal (SIGTERM, gotSig);

   /*
    * Tell the help class about the shell.
    */
   help.setShell (shell);  

#if 0
   /* 
    * Hansi's cat window
    */
    catwids.buildWidgets(shell, &appRes);
#endif

   /*
    * setup secondary rx windows
    */
   DecoderWid::shell = shell;
   for(int i=0; i<MAX_DECODERS; i++)
   {
       decoderWids[i].visible = -1;
   }

   /*
    * Add callbox if desired
    */
   if ( appRes.callBox == 1 )
   {
      fprintf (stderr,"starting callbox\n");
      Callbox a_callbox (shell);
      a_callbox.quiet();
   }

   /*
    * setup and initialize the scope and display
    */
   scope.setup (shell, pskwids.getScope());
   disp.setup(pskwids.getWF());

   /*
    * Set min size - set min sizes to current size.
    */
   XtVaGetValues (shell,
      XmNwidth, &width,
      XmNheight, &height,
      NULL);

   XtVaSetValues (shell,
      XmNminWidth, width,
      XmNminHeight, height,
      NULL);

   /*
    * chk_dialout
    * check to see if user is a member of dialout group
    */
   if (chk_dialout () <= 0)
   {
      errorDiag (shell, (char *) "You are not a member of the dialout group.\n"
                        "You will not be able to use the serial device\n"
                        "for Push to Talk.\n\n"
                        "Please add yourself to the dialout group.\n\n"
                        "Click OK to exit or Cancel to continue",
                         CANCEL_BTN, !ERR_DIAG);
   }

   /*
    * Start the application work proc
    * and enter the main loop
    */
   XtRealizeWidget (shell);
   XtAppAddWorkProc (ac, workProc, (XtPointer) NULL);
   XtAppMainLoop (ac); 
}


/*
 * iniProc - read or write the window locations and stereo bit
 * read - stat to see if $HOME/.twpskDir exists. 
 * If it does, read it. If not, call getQSOfiles to create and populate it
 *
 * write - just over writes the ini file with the current arrays
 * See the quitCB for changes to the array.
 *
 * Put info strings into dispStr.  The main() will print them once we got
 * the rx text widget created.
 */
int iniProc (char mode)
{
   char *filepath;
   char *homepath;
   FILE *fp;
   struct stat file_stat;
   char str[10], mesg[80], *rtn;
   int i;

   homepath = getenv("HOME");
   filepath = (char *)XtMalloc (strlen (homepath)
              + strlen (DAT_DIR)
              + strlen (DAT_FILE) + 4);
   strcpy (filepath, homepath);              /* $HOME */
   strcat (filepath, DAT_DIR);               /* $HOME/.twpskDir */

   /* check if $HOME/.twpskDir exist? */
   if (stat (filepath, &file_stat) == -1)
   {
      getQSOfiles(filepath);                 /* no, so create and populate */
   }

   strcat (filepath, DAT_FILE);              /* $HOME/.twpskDir/.twpsk.dat */

   switch (mode)
   {
   case 'r':
      if (stat (filepath, &file_stat) == -1) /* see if .twpsk.dat exists */
      {
         /* stat failed */
         if (errno == ENOENT)
         {
         /*
          * no twpsk.dat - but this may be the first time we ran, so
          * set winlocs and use_stereo to some nice defaults, then
          * just return. twpsk.dat will be created when twpsk exits.
          */
            /* Put info in dispStr. main() will print it later */
            sprintf (dispStr, "no %s file yet. I'll fix that.\n", filepath);

            for (i=0; i<MAX_DECODERS; i++)
            {
               winlocs[i][0] = 50;
               winlocs[i][1] = 50;
            }
            use_stereo = 1;                  /* default to stereo */
            return TRUE;
         }
         else
         {
            /* all stat errors except ENOENT come here */
            fprintf (stderr, "stat %s failed\n", filepath);
            return FALSE;
         }
      }

      /* .twpsk.dat exists so open and read it */
      if ((fp = fopen (filepath, "r")) == NULL)
      {
         fprintf (stderr, "open %s for read failed\n", filepath);
         return FALSE;
      }
      else
      {
         /* Put info in dispStr. main() will print it later */
         sprintf (mesg, "Reading %s\n", filepath);
         strcat (dispStr, mesg);

         /* now, get the secondary decoder window locations */
         for (i=0; i<MAX_DECODERS; i++)
         {
            rtn = fgets (str, 7, fp);
            if (rtn != str)                  /* no data in file so default */
            {
              strcpy (str, "50\n");
            }
            winlocs[i][0] = atoi(str);               
            rtn = fgets (str, 7, fp);
            if (rtn != str)                  /* no data in file so default */
            {
              strcpy (str, "50\n");
            }
            winlocs[i][1] = atoi(str);               
         }
         /* now get the use_stereo flag */
         rtn = fgets (str, 7, fp);
         if (rtn != str)                     /* if no data in file */
         {
            strcpy (str, "1");               /* then default to stereo */
         }
         use_stereo = atoi(str);
         fclose (fp);
      }
      break;

   case 'w':
      if ((fp = fopen (filepath, "w")) == NULL)
      {
         fprintf (stderr, "Open %s for write failed\n", filepath);
         return FALSE;
      }
      fprintf (stderr, "Writing %s\n", filepath);


      for (i=0; i<MAX_DECODERS; i++)         /* for the secondary locations */
      {
         fprintf (fp, "%d\n%d\n", winlocs[i][0], winlocs[i][1]);
      }
      fprintf (fp, "%d\n", use_stereo); 
      fclose (fp);
      break;

   default:
      fprintf (stderr, "illegal call in iniProc\n");
      return FALSE;
   }
   XtFree (filepath);
   return TRUE;
}


/*
 * workProcFunction
 * Work Proceedure for receive and transmit
 */
Boolean workProc(XtPointer cdata)
{
   PSK31info rxinfo, txinfo;
   char str[8];
   char buf[256];
   static float rxfreqOld;
   int phdelta;  
   int i, ch, l;   
   int rxecho;

   /* handle FFT */
   int N = disp.getsamplecnt ();
   float fftval[N];

   l = commGetData (COMM_FFTCH, (char *)fftval, sizeof(fftval));
   if (l > 0)
   {
      if (l != N)
      {
         fprintf (stderr, "FFT len mismatch: l=%d N=%d\n", l, N);
      }
      disp.getFFT (fftval, N);
   }
   
   /*
    * handle transmit echo.  We could use a different color?
    * Sri, Hansi.  Best I can do is to underline or highlight it.
    */
   rxecho = commGetData (COMM_ECHOCH, buf, sizeof buf);
   if (rxecho > 0)
   {
      for (i=0; i<rxecho; i++)
      {
         appendRXtext (pskwids.getRxText(), buf[i],
                      appRes.zero, (XmHighlightMode)appRes.xmitHighlight);
      }
   }
   commGetInfo (COMM_TXCH, &txinfo, sizeof(txinfo));

   /*
    * handle receive
    */
   l = commGetData (COMM_RXCH, buf, sizeof buf);
   if(l > 0)
   {
      for (i=0; i<l; i++)
      {
         appendRXtext (pskwids.getRxText(), buf[i],
                       appRes.zero, XmHIGHLIGHT_NORMAL);
#if DO_SPEAK == 1
mkWord (buf[i]);
#endif
      }
   }

   commGetInfo (COMM_RXCH, &rxinfo, sizeof(rxinfo));
   appRes.rxFreq = (0.01 * rxinfo.freq);

   phdelta = rxinfo.phdelta;
   
   /* update DCD toggle button */
   XtVaSetValues (pskwids.getDcdTB(), XmNset, rxinfo.dcd, NULL);

   if (rxinfo.ptt == 0)
   {
      /* receiving */
      /* update scope */
      scope.drawline (rxinfo.phdelta, rxinfo.strength, rxinfo.dcd);

      /* update rx freq */
      /* but only if focusFlag == 0 */
      if (pskwids.getRxFreqFocus() == 0 )
      {
         if (fabs (appRes.rxFreq - rxfreqOld) > .05)
         {
            sprintf (str, "%4.1f", appRes.rxFreq);
            XmTextFieldSetString (pskwids.getRxFreqTF(), str);
            /* tell the fft about it */
            disp.offset (appRes.rxFreq);
            rxfreqOld = appRes.rxFreq;
         }
      }
   }
   else
   {
      /* scope display for each tx character put in the echo (rx) window */
      if (rxecho > 0)
         phdelta = 0;
      else
         phdelta = 128;
      scope.drawline (phdelta, 35, GREEN);
   }

   /* handle receive for additional receivers */
   for (ch=0; ch<MAX_DECODERS; ch++)
   {
      if (decoderWids[ch].visible!=1)
      {
         continue;
      }
      l = commGetData (ch+3, buf, sizeof buf);
      if (l > 0)
      {
         for (i=0; i<l; i++)
         {
            appendRXtext (decoderWids[ch].getTextWid(), buf[i],
               appRes.zero, XmHIGHLIGHT_NORMAL);
         }
      }
      l = commGetInfo (decoderWids[ch].commChannel, &rxinfo, sizeof(rxinfo));
      if (l == 0)
      {
         decoderWids[ch].updateDisplay (0.01*rxinfo.freq, rxinfo.dcd, 0);
         decoderWids[ch].getScope().drawline (rxinfo.phdelta, 
            rxinfo.strength, rxinfo.dcd);
      }
   }
   usleep (10000);    // for rx time
   return (False);
}


void gotSig (int stat)
{
   if (stat == SIGINT)
   {
      fprintf (stderr, "INTERRUPT\n");
      signal (SIGINT, SIG_IGN);
      quitCB ((Widget)0, (XtPointer)0, (XtPointer)0);
      exit (0);
   }
   if (stat == SIGQUIT)
   {
      fprintf (stderr, "QUIT\n");
      signal (SIGINT, SIG_IGN);
      quitCB ((Widget)0, (XtPointer)0, (XtPointer)0);
      exit (0);
   }
}


/*
 * getQSOfiles - called if $HOME/.twpskDir does not exist. We must
 * be running for the first time. So, create $HOME/.twpskDir and copy
 * files from PKG_DATA_DIR/twcw/twcwDir to destdir ($HOME/.twpskDir)
 */
int getQSOfiles (char *destdir)
{
   int n, fdin, fdout;
   char dirpath[100];
   DIR *dirptr;
   struct dirent *dirstruct;
   char dname[100];
   char ifile[100], ofile[100];
   char buff[BUFSIZ];

   strcpy (dirpath, getenv ("HOME"));
   strcat (dirpath, "/.twpskDir");

   /* create $HOME/.twcwDir */
   if (mkdir (dirpath, 0755) < 0)
   {
      errorDiag (shell,
                 (char *) "Can't create $HOME/.twcwDir\n"
                 "twcw must exit\n\n"
                 "Please send bug reports to\n" PACKAGE_BUGREPORT,
                  !CANCEL_BTN, ERR_DIAG);
      fprintf (stderr, "twcw: can't create .twcwDir\n");
      return (-1);
   }

   /* open the data dir at PKG_DATA_DIR/twcwDir */
   strcpy (dname, PKG_DATA_DIR);
   strcat (dname, (char *) "/twpskDir");

   if ((dirptr = opendir (dname)) == NULL)
   {
      errorDiag (shell,
                 (char *) "getQSOfiles: Can't open PKG_DATA_DIR/twpskDir\n"
                 "This is an installation error.\n"
                 "twcw must exit.\n\n"
                 "Please send bug reports to\n" PACKAGE_BUGREPORT,
                  !CANCEL_BTN, ERR_DIAG);
      fprintf (stderr, "can't open twcwDir\n");
      return (-1);
   }

   while ((dirstruct = readdir (dirptr)) != NULL)
   {  /* skip the . and .. entries */
      if ((strcmp (dirstruct->d_name, ".") == 0) |
         (strcmp (dirstruct->d_name, "..") == 0))
      {
         continue;
      }
      strcpy (ifile, dname);
      strcat (ifile, "/");
      strcat (ifile, dirstruct->d_name);

      strcpy (ofile, destdir);
      strcat (ofile, "/");
      strcat (ofile, dirstruct->d_name);

      fdin = open (ifile, O_RDONLY, 0655);
      fdout = open (ofile, O_RDWR | O_CREAT, 0655);

      while ((n = read (fdin, buff, BUFSIZ)) > 0)
      {
         n = write (fdout, buff, n);
      }
      close (fdin);
      close (fdout);
   }
   closedir (dirptr);
   return 1;
}


/*
 * chk_dialout - checks to see if the user is a member of the dialout group
 * return 1 if true, otherwise return 0
 */
int chk_dialout (void)
{
   int i=0;
   struct group *ptr;
   char *user;

   user = getenv ("LOGNAME");
   ptr = getgrnam ("dialout");

   if (ptr == NULL)
   {
     perror ("getgrnam failed");
     return -1;
   }
   for (i=0; ptr->gr_mem[i] != NULL; i++)
   {
      if (strcmp (ptr->gr_mem[i], user) == 0)
      {
         return 1;
      }
   }
   return 0;
}
