# Copyright (c) 2007-2008 Andrew Price
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. The name of the author may not be used to endorse or promote products
#    derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

"""Command line parsing and processing for twyt"""

import getpass
import twitter
import user
import data
from info import VERSION
from optparse import OptionParser, make_option

class Commands:

	"""Provides a convenient interface to the twyt commands"""

	def __init__(self):

		self.commands = {}
		self.parser = OptionParser(usage="%prog COMMAND [options]",
				version=VERSION)

		self.usr = user.User()
		self.twitter = twitter.Twitter()
		self.twitter.set_user_agent("Twyt")
		if self.usr.getpassword() and self.usr.getusername():
			self.twitter.set_auth(
				self.usr.getusername(),
				self.usr.getpassword())

	
	def register_command(self, command):
		self.commands[str(command)] = command

	def __lookup_command(self, command):
		try:
			return self.commands[command]
		except KeyError, k:
			return DefaultCommand()

	def __parse_command(self, command, argv):
		self.parser.prog = "%s %s" %\
			(self.parser.get_prog_name(), str(command))
		self.parser.set_usage(command.get_usage())
		optlst = command.get_options()
		for o in optlst:
			self.parser.add_option(o)
		self.parser.set_description(command.get_help())

		if str(command) != "":
			argv = argv[1:]

		return self.parser.parse_args(argv)

	def process_argv(self, argv):
		try:
			command = self.__lookup_command(argv[1])
		except IndexError, i:
			# This exits
			self.parser.error("No command specified. Try --help")

		options, args = self.__parse_command(command, argv[1:])

		try:
			if options.password is not None:
				self.usr.setpassword(options.password)
			elif options.user is not None:
				self.usr.setusername(options.user)
		except AttributeError:
			# Just didn't ask for a user/password
			pass

		command.do_command(options, args, self)

		self.usr.synctimes()


class Command:
	def __init__(self):
		self.options = []

class DefaultCommand(Command):
	def __str__(self):
		return ""

	def get_options(self):
		self.options.append(make_option("-c","--commands",
				action="store_true",
				help="Lists available commands"))
		return self.options

	def get_usage(self):
		return "%prog COMMAND [options] [args]"

	def get_help(self):
		return "Twyt is a command line Twitter client written using, "\
			"and specifically to test, the python-twyt library.\n"\
			"\nSee --commands for possible values of COMMAND"

	def do_command(self, options, args, base):
		if options.commands:
			base.parser.print_usage()
			print "Available commands:"
			keys = base.commands.keys()
			keys.sort()
			for k in keys:
				print " %10s %s" %\
					(k, base.commands[k].get_help())
			print ""
			print "For command-specific help, use "\
				"%sCOMMAND --help" % base.parser.get_prog_name()
		else:
			base.parser.error("Command not found. Try --commands")


class CommandWithAuth:

	def __init__(self):
		self.options = 	[
			make_option("-u", "--user", metavar="USER",
			help="Specifies USER as your username. USER is "
			"then stored as your default username."),
			make_option("-p", "--pass", metavar="PASS",
			dest="password",
			help="Specifies PASS as your password.")
		]

        def passwdprompt(self, usr):
		pwd = ""
		for _ in range(3):
			pwd = getpass.getpass("Enter %s's Twitter password: " % usr)
			if pwd != "":
				break

		return pwd

	def callwithauth(self, func, data, base):
		answer = ""

		while not answer:
			try:
				answer = func(*data)
				break
			except twitter.TwitterAuthException:
				user = base.usr.getusername()
				if not user:
					base.parser.error('No username specified.')
				base.twitter.set_auth(user, self.passwdprompt(user))

		return answer


class TweetCommand(CommandWithAuth):
	def __str__(self):
		return "tweet"

	def get_options(self):
		return self.options

	def get_usage(self):
		return "%prog [options] message..."

	def get_help(self):
		return "Updates the authenticating user's Twitter status"

	def do_command(self, options, args, base):
		if len(args) < 1:
			base.parser.error("No message specified.")

		tweetmsg = " ".join(args)

		answer = self.callwithauth(
			base.twitter.status_update, [tweetmsg], base)

		result = data.Status()
		result.load_json(answer)

		print unicode(result)


class DirectCommand(CommandWithAuth):
	def __str__(self):
		return "direct"

	def get_options(self):
		return self.options

	def get_usage(self):
		return "%prog [options] RECIPIENT MESSAGE..."

	def get_help(self):
		return "Sends a direct message to another user"

	def do_command(self, options, args, base):
		if len(args) < 1:
			base.parser.error("No recipient specified.")

		if len(args) < 2:
			base.parser.error("No message specified.")

		touser = args[0]
		msg = " ".join(args[1:])
		answer = self.callwithauth(
			base.twitter.direct_new, [touser, msg], base)

		result = data.DirectMsg()
		result.load_json(answer)
		
		if not result.id:
			raise TwitterException("Sending direct message failed.")

		print unicode(result)


class DirectTLCommand(CommandWithAuth):
	def __str__(self):
		return "directtl"

	def get_options(self):
		self.options.append(make_option("-s", "--since",
			metavar="SINCE",
			help="The date or ID of a message to list direct "
			"messages from."))
		self.options.append(make_option("-P", "--page",
			metavar="PAGE", default=1, type="int",
			help="Lists the PAGEth page of direct messages "
			"(default 1)"))

		return self.options

	def get_usage(self):
		return "%prog [options]"

	def get_help(self):
		return "Prints the 20 last direct messages sent to you"

	def do_command(self, options, args, base):
		since = ""
		since_id = ""
		# Since can be an ID or date
		if options.since:
			try:
				since_id = int(options.since)
			except ValueError:
				since = arg
	
		answer = self.callwithauth(
			base.twitter.direct_messages, [since, since_id,
				options.page], base)

		results = data.DirectList(answer)
		
		for result in results:
			print unicode(result)


class DirectSentCommand(CommandWithAuth):
	def __str__(self):
		return "directsent"

	def get_options(self):
		self.options.append(make_option("-s", "--since",
			metavar="SINCE",
			help="The date or ID of a message to list direct "
			"messages from."))
		self.options.append(make_option("-P", "--page",
			metavar="PAGE", default=1, type="int",
			help="Lists the PAGEth page of direct messages "
			"(default 1)"))

		return self.options

	def get_usage(self):
		return "%prog [options]"

	def get_help(self):
		return "Prints the 20 last direct messages sent by you"

	def do_command(self, options, args, base):
		since = ""
		since_id = ""
		# since can be an id or a date
		if options.since:
			try:
				since_id = int(options.since)
			except ValueError:
				since = options.since

		answer = self.callwithauth(base.twitter.direct_sent,
				[since, since_id, options.page], base)

		results = data.DirectList(answer)
		
		for result in results:
			print unicode(result)


class DirectDelCommand(CommandWithAuth):
	def __str__(self):
		return "directdel"

	def get_options(self):
		return self.options

	def get_usage(self):
		return "%prog [options] ID"

	def get_help(self):
		return "Delete a direct message which was sent to you"

	def do_command(self, options, args, base):

		sid = -1
		if len(args) > 0:
			try:
				sid = int(args[0])
			except ValueError:
				pass

		if sid < 0:
			base.parser.error("Invalid ID specified")

		answer = self.callwithauth(base.twitter.direct_destroy,
				[sid], base)
		if answer:
			result = data.DirectMsg()
			result.load_json(answer)

			print u'Deleted: ' + unicode(result)
		else:
			print "No messages deleted. Are you authorized?"
	

class PublicTLCommand(Command):
	def __str__(self):
		return "publictl"
	
	def get_options(self):
		self.options.append(
			make_option("-i", "--since-id", metavar="ID",
			type="int",
			help="Returns only public statuses with an ID greater "
			      "than (more recent than) the specified ID."))
		return self.options

	def get_usage(self):
		return "%prog [options]"

	def get_help(self):
		return "Shows the 20 most recent statuses in Twitter's "\
			"public timeline"

	def do_command(self, options, args, base):
		answer = base.twitter.status_public_timeline(
					options.since_id)

		results = data.StatusList(answer)

		if options.since_id:
			print "Since ID %s:" % options.since_id

		for result in results:
			print unicode(result)

		if len(results) > 0:
			base.usr.set_last_read('publictl', 'lastid',
					str(results.get_last_id()))
		else:
			print "(Nothing new)"

		return results


class FriendsTLCommand(CommandWithAuth):
	def __str__(self):
		return "friendstl"

	def get_options(self):
		self.options.append(make_option("-s", "--since",
			metavar="DATE",
			help="Narrows the returned results to just those "
			      "statuses created after DATE"))
		return self.options

	def get_usage(self):
		return "%prog [options] [USERNAME]"

	def get_help(self):
		return "Returns 20 most recent statuses in your (or "\
			"USERNAME's) friends timeline"

	def do_command(self, options, args, base):
		ident = ""
		if len(args) > 0:
			ident = args[0]

		since = base.usr.get_last_read('friendstl', ident)

		if options.since is not None:
			since = options.since

		answer = self.callwithauth(base.twitter.status_friends_timeline,
				[ident, since], base)

		results = data.StatusList(answer)

		if since:
			print "Since %s:" % since
		for result in results:
			print unicode(result)

		if len(results) > 0:
			base.usr.set_last_read('friendstl', ident, results.get_last_time())
		else:
			print "(Nothing new)"


class UserTLCommand(CommandWithAuth):
	def __str__(self):
		return "usertl"

	def get_options(self):
		self.options.append(make_option("-s", "--since",
			metavar="SINCE",
			help="The date to list direct messages from."))
		self.options.append(make_option("-c", "--count",
			metavar="COUNT", default=20, type="int",
			help="The number of statuses to show, max 20"))
		return self.options

	def get_usage(self):
		return "%prog [options] [USERNAME]"

	def get_help(self):
		return "Show your timeline, or USERNAME's timeline"

	def do_command(self, options, args, base):
		count = 20
		if options.count > 0:
			count = options.count
		
		ident = ""
		if len(args) > 0:
			ident = args[0]

		answer = self.callwithauth(base.twitter.status_user_timeline,
				[ident, count, options.since], base)
		results = data.StatusList(answer)

		for result in results:
			print unicode(result)


class ShowCommand(CommandWithAuth):
	def __str__(self):
		return "show"

	def get_options(self):
		return self.options

	def get_usage(self):
		return "%prog ID"

	def get_help(self):
		return "Show a single status message by ID"

	def do_command(self, options, args, base):
		ident = ""
		if len(args) > 0:
			ident = args[0]
		else:
			base.parser.error("No ID specified.")

		answer = base.twitter.status_show(ident)

		if answer:
			result = data.Status()
			result.load_json(answer)
			
			print unicode(result)


class RepliesCommand(CommandWithAuth):
	def __str__(self):
		return "replies"

	def get_options(self):
		self.options.append(make_option("-P", "--page", metavar="PAGE",
			type="int", default=1,
			help="Shows the PAGEth page of replies."))
		return self.options

	def get_usage(self):
		return "%prog [options]"

	def get_help(self):
		return "Show the Nth page of 20 replies (messages with "\
			"@yourusername in them)"

	def do_command(self, options, args, base):
		answer = self.callwithauth(base.twitter.status_replies,
				[options.page], base)

		if answer:
			results = data.StatusList(answer)

			for result in results:
				 print unicode(result)


class DeleteCommand(CommandWithAuth):
	def __str__(self):
		return "delete"

	def get_options(self):
		return self.options

	def get_usage(self):
		return "%prog [options] ID"

	def get_help(self):
		return "Deletes a tweet by ID"

	def do_command(self, options, args, base):
		sid = -1
		if len(args) > 0:
			try:
				sid = int(args[0])
			except ValueError:
				pass

		if sid < 0:
			base.parser.error("Invalid ID specified")

		answer = self.callwithauth(base.twitter.status_destroy,
				[sid], base)
		
		if answer:
			result = data.Status()
			result.load_json(answer)

			print u'Deleted: ' + unicode(result)
		else:
			print u'No messages deleted. Are you authorised?'


class SavePassCommand(CommandWithAuth):
	def __str__(self):
		return "savepass"

	def get_options(self):
		return self.options

	def get_usage(self):
		return "%prog [options]"

	def get_help(self):
		return "Saves your username and password for subsequent use"

	def do_command(self, _, __, base):
		username = base.usr.getusername()
		if username == "":
			base.parser.error("Username not specified.")

		pwd = self.passwdprompt(username)

		base.usr.savepass(username, pwd)
		base.twitter.set_auth(username, pwd)


class ArchiveCommand(CommandWithAuth):
	def __str__(self):
		return "archive"

	def get_options(self):
		self.options.append(make_option("-P", "--page", metavar="PAGE",
			default=0, type="int",
			help="Retrieves page PAGE of your status messages"))
		self.options.append(make_option("-s", "--since",metavar="DATE",
			help="Narrows the returned results to just those "
			      "statuses created after DATE"))
		self.options.append(make_option("-i", "--since-id",
			metavar="ID", type="int",
			help="Returns only statuses with an ID greater "
			      "than (more recent than) the specified ID."))
		return self.options

	def get_usage(self):
		return "%prog [options]"

	def get_help(self):
		return "Shows 80 of your status messages ordered by date of "\
			"posting"

	def do_command(self, options, args, base):
		page = ""
		since = ""
		since_id = ""

		if options.page > 0:
			page = options.page
		if options.since:
			since = options.since
		if options.since_id:
			since_id = options.since_id

		ident = ""
		if len(args) > 0:
			ident = args[0]

		print [page, since, since_id]
		answer = self.callwithauth(base.twitter.account_archive,
				[page, since, since_id], base)
		results = data.StatusList(answer)

		for result in results:
			print unicode(result)
