/* ==================================================== ======== ======= *
 *
 *  uconf.hpp : configuration of the UAppli
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _uconf_hpp_
#define	_uconf_hpp_
//pragma ident	"@(#)uconf.hpp	ubit:03.06.04"

/* ==================================================== ======== ======= */
/** command line options of the UAppli.
 * example:
 * - command line:  myappli -in[put] foo -val[ue] 5 -r[eco]
 *   (chars between the [] are optional; the negation or -reco is -no-reco)
 * - source code for parsing these options:
 * <pre>
 *   char* input_file  = null;
 *   int   value = 0;
 *   bool reco_mode = false;
 *
 *   UOption options[] = {
 *     {"in","put", UOption::Arg(input_file)},
 *     {"val","ue", UOption::Arg(value)},
 *     {"r","eco",  UOption::Arg(reco_mode)},
 *     {null, null, null}
 *   };
 *
 *   // updates 'input_file', 'value' and 'reco_mode' if the appropriate
 *   // options are specified in the command line
 *   UConf conf(argc, argv, options);
 *   UAppli appli(conf);
 *   ...
 * </pre>
 */
struct UOption {
  const char *begname, *endname;
  class UOptionArg* arg;

  static class UOptionArg* Arg(bool& val);
  static class UOptionArg* Arg(int& val);
  static class UOptionArg* Arg(char*& val);
  static class UOptionArg* Arg(const char*& val);
};

/* ==================================================== ======== ======= */
/** configuration of the UAppli.
 */
class UConf {
public:
  enum GraphicsEngine {X11, OGL};

  UConf::UConf(int& _argc, char** _argv, UOption* client_options = null);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  const char* disp_name;
  ///< starts the appli on this X-server display (default is null)

  const char* app_name;
  
  const char* locale;
  ///< locale used by the application (default is null)

  enum GraphicsEngine graphics_engine;
  ///< not used in this version.
  
  int truecolor_depth;
  /**< 
   * number of bits for representing colors in "True Color" mode.
   * Default is 24 bits. A value of 0 means "use server default".
   *
   * Notes:
   * - truecolor_depth is a hint: the visual with the closest depth 
   *   available will be used.
   * - TrueColor mode should be used whenever possible because it is
   *   requested for alpha blending (transparency)
   */

  /// these options must be specified at init and can't be changed.
  bool 
    double_buffering,    ///< avoids flickering: default is false with X and true with GL
    linear_gamma,        ///< gamma correction on Solaris: default is true
    transp_scrollbars,   ///< transparent scrollbars: default is false
    softwins,            ///< soft window mode: true in GL mode, false in X mode
    telepointers,        ///< show telepointers in multi-disp mode
    iconic,              ///< not implemented
    sync,                ///< synchronous mode (for debugging): default is false
    menu_grab,           ///< enables mouse grabs: default is true
    scrollbar_grab,      ///< not implemented
    force_winfocus;      ///< should be true except in special cases

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  class UStyle* defaultStyle;
  UPix* unknown_image;	        ///< used when an image is not found (default= &UPix::question)

  /// logical scale applied to all windows (0 is the default)
  int app_lscale;

  /// maximum size used by the windows of the UAppli (-1, the default, means no constraint)
  u_dim max_app_width, max_app_height; 

  int scrollpane_mode;
  u_dim scrollpane_bar_size;  // not impl

  class UWidth&  filebox_width;         ///< MUST be defined (default= 400)
  class UHeight& filebox_height;        ///< MUST be defined (default= 230)
  int filebox_line_count;               ///< MUST be defined (default= 10)

  u_dim  mouse_click_radius;
  u_time mouse_click_delay;             ///< millisec, for double clicks
  u_time auto_open_menu_delay;          ///< millisec, for opening menus

  class UColor* default_selection_color;    ///< none if pointer is null
  class UColor* default_selection_bgcolor;  ///< none if pointer is null
  class UFont*  default_selection_font;     ///< none if pointer is null

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  float app_xyscale;  /// impl, set by UDisp::UDisp

private:
  friend class UAppli;
  friend class UDisp;

  struct Profile {
    const char* name;
    void (*apply)(UConf&);
    Profile(const char* _name, void (*_apply)(UConf&));
  };

  int    app_argc; 
  char** app_argv;

  static std::vector<UConf::Profile> profiles;
  static bool init;

  UConf(const UConf&);
  UConf& operator=(const UConf&);

  static bool initDefaultProfiles();
  static void addProfile(const char* name, void (*apply)(UConf&));

  bool setProfile(const char* name);
  void saveArgs(int _argc, char** _argv);
  void parseOptions(int& argc, char** argv, UOption* client_options);
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
