#!/usr/bin/env python
# 
# 2009-03-11 
# This will create a report with an assigned gravity to bug reports for a package
# The gravity is a combination of points associated with bug tags and whether or
# not the bug is private.  Additionally, the quantity of duplicates, subscribers
# and users affected is added to the gravity.
#
# Copyright 2009 Canonical, Ltd
# Author: Brian Murray <brian@ubuntu.com>
# Licensed under the GNU General Public License, version 3.

from launchpadlib.launchpad import Launchpad, EDGE_SERVICE_ROOT, STAGING_SERVICE_ROOT
from launchpadlib.errors import HTTPError
from launchpadlib.credentials import Credentials
from operator import itemgetter
from cgi import escape

import os, sys

cachedir = os.path.expanduser("~/.launchpadlib/cache/")

if not os.path.exists(cachedir):
    os.makedirs(cachedir,0700)

script_name = sys.argv[0].split("/")[-1].split('.')[0]

credfile = os.path.expanduser('~/.launchpadlib/%s.cred' % script_name)

try:
    credentials = Credentials()
    credentials.load(open(credfile))
    launchpad = Launchpad(credentials, EDGE_SERVICE_ROOT, cachedir)
except:
    launchpad = Launchpad.get_token_and_login(script_name, EDGE_SERVICE_ROOT, cachedir)
    launchpad.credentials.save(os.fdopen(os.open(credfile, os.O_WRONLY | os.O_CREAT, 0600), "w"))
    
def save_entry(entry):
    try:
        entry.lp_save()
    except HTTPError, error:
        print error.content
    else:
        print "Did not get expected HTTPError!"

# commonly used things
ubuntu = launchpad.distributions['ubuntu']

try:
    package = sys.argv[1]
    target_package = ubuntu.getSourcePackage(name='%s' % package)
    # should Triaged be added in? not based off the Ubuntu foundations team meeting on 2009-03-11 
    tasks = target_package.searchTasks(status=['New','Incomplete','Confirmed'],order_by='-datecreated')
except IndexError:
    package = 'no-package'
    # should Triaged be added in? not based off the Ubuntu foundations team meeting on 2009-03-11 
    tasks = ubuntu.searchTasks(has_no_package=True,status=['New','Incomplete','Confirmed'],order_by='-datecreated')

# write to this file
datafile = open('%s-bug-gravity.html' % package, 'w')


templatefile = open('template.html', 'r')

for line in templatefile:
    if line.strip() == "<!-- *** Title Space *** -->":
        datafile.write("Bug tasks for %s with a high bug gravity\n" % package)
    elif line.strip() == "<!-- *** Header Space *** -->":
        datafile.write("Bug tasks for %s with a high bug gravity\n" % package)
    elif line.strip() == "<!-- *** Paragraph Space *** -->":
        datafile.write("All of the column headers are sortable give them a click!<br>Curious about how gravity is computed? See the legend at the bottom of the page.")
    elif line.strip() == "<!-- *** Table Header Space *** -->":
        table_header = "<th>#</th>"  # bug number in this column
        table_header += "<th>Summary</th>"
        table_header += "<th>In</th>"
        table_header += "<th>Importance</th>"
        table_header += "<th>Status</th>"
        table_header += '<th class="default-sort initial-sort">Gravity</th>'
        table_header += "<th>Users Affected</th>"
        table_header += "<th>Duplicates</th>"
        table_header += "<th>Subscribers</th>"
        table_header += "\n"
        datafile.write(table_header)
    elif line.strip() == "<!-- *** Table Body Space *** -->":
        for task in tasks:
            try:
                weight = 0
                dupe_count = len(task.bug.duplicates)
                sub_count = len(task.bug.subscriptions)
                user_count = task.bug.users_affected_count
                tag_weights = { 'apport-bug':50, 'apport-package':100, 'apport-crash':100, 
                                'regression-potential':150, 'regression-release':200,
                                'regression-proposed':250, 'regression-updates':300 }
                tag_set = set()
                for tag in tag_weights:
                    tag_set.add(tag)
                bug_tag_set = set()
                for tag in task.bug.tags: 
                    bug_tag_set.add(tag)
                if tag_set.intersection(bug_tag_set):
                    for tag in tag_set.intersection(bug_tag_set):
                        weight += tag_weights[tag]
                        #print "adding weight %s for %s" % ( tag_weights[tag], tag )
                # any odd numbered gravity is a private bug
                if task.bug.private:
                    weight += 151
                weight += ( 6*dupe_count + 4*sub_count + 2*user_count )
                if weight >= 12:
                    table_row = "<tr>"
                    table_row += "<td>%s</td>" % task.bug.id
                    table_row += "<td><a href='http://launchpad.net/bugs/%s'>%s</a></td>" % (task.bug.id, escape(task.bug.title.encode('utf-8')))
                    table_row += "<td>%s</td>" % task.bug_target_display_name.encode('utf-8')
                    table_row += "<td>%s</td>" % task.importance.encode('utf-8')
                    table_row += "<td>%s</td>" % task.status.encode('utf-8')
                    table_row += "<td>%s</td>" % ( weight )
                    table_row += "<td>%s</td>" % ( user_count )
                    table_row += "<td>%s</td>" % ( dupe_count )
                    table_row += "<td>%s</td>" % ( sub_count )
                    table_row += "</tr>"
                    table_row += "\n"
                    datafile.write(table_row)
            except HTTPError, error:
                print "There was an error with LP: #%s: %s" % (task.title.split(' ')[1].replace('#',''), error.content)
    elif line.strip() == "<!-- *** Last Paragraph Space *** -->":
        datafile.write("<table border=1>\n")
        datafile.write("<thead><tr><th align=center>Item</th><th align=center>Value</th></tr></thead>\n")
        for k,v in sorted(tag_weights.items(), key=itemgetter(1), reverse=True):
            datafile.write("<tr><td>%s tag</td><td>%s</td>\n" % (k,v))
        datafile.write("<tr><td>private bug</td><td>151</td></tr>\n")
        datafile.write("<tr><td>num duplicates</td><td>x*6</td></tr>\n")
        datafile.write("<tr><td>num subscribers</td><td>x*4</td></tr>\n")
        datafile.write("<tr><td>num users affected</td><td>x*2</td></tr>\n")
        datafile.write("<table>\n")
        datafile.write('<br>')
        datafile.write('Is there a problem with this report? File a bug about the <a href="https://bugs.launchpad.net/ubuntu-qa-tools/+filebug">ubuntu-qa-tools.</a>')
    else:
        datafile.write(line)
datafile.close()
