#!/usr/bin/env python
#
# Copyright 2009 Canonical, Ltd
# Author: Brian Murray <brian@ubuntu.com>
# License: GPLv2
# 
# 2009-01-28 
# This searches for Ubuntu bug reports, that are not needs-packaging bugs,
# with patches attached to them.  It then checks to see if the bug report is 
# part of a workflow and if so skips it.  If it isn't and the patch is recent,
# date added > 2009-12-15, subscribe the ubuntu-reviewers team and add the tag patch.

from launchpadlib.launchpad import Launchpad, EDGE_SERVICE_ROOT, STAGING_SERVICE_ROOT
from launchpadlib.errors import HTTPError
from launchpadlib.credentials import Credentials

import os, sys

cachedir = os.path.expanduser("~/.launchpadlib/cache/")

if not os.path.exists(cachedir):
    os.makedirs(cachedir,0700)

script_name = sys.argv[0].split("/")[-1].split('.')[0]

credfile = os.path.expanduser('~/.launchpadlib/%s.cred' % script_name)

try:
    credentials = Credentials()
    credentials.load(open(credfile))
    launchpad = Launchpad(credentials, EDGE_SERVICE_ROOT, cachedir)
except:
    launchpad = Launchpad.get_token_and_login(script_name, EDGE_SERVICE_ROOT, cachedir)
    launchpad.credentials.save(open(credfile,"w",0600)) 
    
def save_entry(entry):
    try:
        entry.lp_save()
    except HTTPError, error:
        print error.content

# commonly used things
ubuntu = launchpad.distributions['ubuntu']

# excluding needs-packaging since that has a separate process - revu
tasks = ubuntu.searchTasks(order_by='-date_last_updated',tags='-needs-packaging',has_patch=True)

reviewers = launchpad.people['ubuntu-reviewers']
jfo = launchpad.people['jeremyfoshee']
# teams that process bugs with patches
workflow_teams = [ 'ubuntu-universe-sponsors', 'ubuntu-main-sponsors', 'ubuntu-sru', \
                   'ubuntu-mir', 'ubuntu-release', 'ubuntu-reviewers', 'motu-sru', \
                   'ubuntu-security-sponsors', 'ubuntu-sponsors' ]
# tags that indicate a review has happened
reviewed_tags = [ 'patch-needswork', 'patch-forwarded-upstream', 'patch-forwarded-debian', \
                  'indicator-application', 'patch-accepted-upstream', 'patch-accepted-debian', \
                  'patch-rejected-upstream', 'patch-rejected-debian', 'patch-rejected']
# kernel packages are special and subject to change
kernel_packages = ['linux (Ubuntu)', 'linux-fsl-imx51 (Ubuntu)', 'ubuntu-imx (Ubuntu)', \
                   'linux-backports-modules-2.6.32 (Ubuntu)', 'linux-ti-omap (Ubuntu)']

stop = False

for task in tasks:
    if stop:
        print 'found a bug with an old patch so stopping'
        break
    bug = launchpad.bugs[task.bug.id]
    # these bugs have been reviewed and require something
    if set(reviewed_tags).intersection(bug.tags):
        continue
    subscribers = []
    for subscriber in bug.subscriptions:
        subscribers.append(subscriber.person.name)
    # don't act on sync requests with the archive admin team subscribed
    if 'sync' in bug.title.lower() and 'ubuntu-archive' in subscribers:
        continue
    # don't act on bugs that already have a sponsor's team subscribed
    if set(workflow_teams).intersection(subscribers):
        continue
    try:
        for attachment in bug.attachments:
            if attachment.type == 'Patch':
                # depending on the queue's throughput we'll want to start adding historical bugs too
                if attachment.message.date_created.strftime('%Y-%m-%d') > '2009-12-24':
                    print "LP: #%s had patch %s added to it" % ( bug.id, attachment.title )
                        # subscribe the team and add the tag patch to the bug report
                        # the kernel's workflow is special just subscribe jfo
                    if task.bug_target_name in kernel_packages:
                        bug.subscribe(person=jfo)
                    # skip this package altogether
                    elif task.bug_target_name in ['ubuntu-docs (Ubuntu)', 'xserver-xorg-video-openchrome (Ubuntu)'] :
                        continue
                    bug.subscribe(person=reviewers)
                    if 'patch' not in bug.tags:
                        bug = launchpad.bugs[bug.id]# fresh bug object, LP: #336866 workaround
                        bug.tags = bug.tags + ['patch']# LP: #254901 workaround
                        save_entry(bug)
                    stop = False
                else:
                    # we want to check every attachment to see if one of them is newer than our target
                    # date - so don't break out here and above set stop = False if one is newer
                    stop = True
    except HTTPError, error:
        print "Launchpad hates me and LP: #%s" % ( bug.id )
