#!/bin/sh -e
#
# Copyright (C) 2007,2008 Canonical, Ltd.
# Author: Jamie Strandboge <jamie@canonical.com>
# License: GPLv3
#

ustconf="$HOME/.uqt-vm-tools.conf"
if [ -s "$ustconf" ]; then
    . "$ustconf"
else
    echo "Could not find '$ustconf'"
    exit 1
fi

abort_if_root() {
    if [ "$USER" = "root" ]; then
        echo "You must not be root. Aborting"
        exit 1
    fi
}

vm_exists() {
    if [ -z "$1" ]; then
        exit 1
    fi
    vm="$1"

    if virsh --connect ${vm_connect} list --all 2>/dev/null | grep -q " $vm " ; then
        return 0
    else
        return 1
    fi
}

vm_running() {
    if [ -z "$1" ]; then
        exit 1
    fi
    vm="$1"

    if virsh --connect ${vm_connect} domstate "$vm" 2>/dev/null | grep -q "running" ; then
        return 0
    else
        return 1
    fi
}

get_disks() {
    if [ -z "$1" ]; then
        exit 1
    fi
    vm="$1"

    found_disks=""
    in_disk="no"
    virsh --connect ${vm_connect} dumpxml "$vm" 2>/dev/null | while read line ; do
        if echo "$line" | egrep -q '^<disk '; then
            in_disk="yes"
            continue
        elif echo "$line" | egrep -q '^</devices>' ; then
            echo $found_disks
            break
        elif [ "$in_disk" = "yes" ]; then
            if echo "$line" | egrep -q "^<source file=.*\.iso'/>" ; then
                # found an iso so skip it, otherwise vm-remove might remove
                # your iso directory (owchy)
                in_disk="no"
                continue
            elif echo "$line" | egrep -q '^<source file=' ; then
                tmpdisk=`echo "$line" | cut -d "'" -f 2`
                found_disks="$found_disks $tmpdisk"
                in_disk="no"
            fi
        fi
    done
}

# waits ARG2 seconds for a host to come up and returns the hostname
vm_wait() {
    if [ -z "$1" ]; then
        exit 1
    fi
    local vm="$1"
    local timeout="$2"
    if [ -z "$timeout" ]; then
        timeout=100
    fi

    # wait until the machine is running
    local host=""
    echo -n "Waiting for '$vm' to come up" >&2
    for i in $(seq 1 $timeout) ; do
        host=$(vm_ping "$vm" 2>/dev/null)
        if [ -n "$host" ] ; then
            echo " host is up" >&2
            echo "$host"
            break
        fi
        echo -n "." >&2
        sleep 1
    done
    if [ -z "$host" ]; then
        echo " timed out waiting for '$vm'" >&2
    fi
}

# returns the hostname for a machine if it is up
vm_ping() {
    m="$1"
    if ! vm_exists $m ; then
        echo "'$m' does not exist" >&2
        return
    fi

    if ! vm_running $m ; then
        echo "'$m' is not running" >&2
        return
    fi

    for h in ${m}. ${m}.local $m ; do
        if ping -c1 -w1 $h >/dev/null 2>&1 ;then
            if nc -q1 $h 22 </dev/null 2>/dev/null | grep -q ^SSH-2; then
                echo "$h"
                break
            fi
        fi
    done
}

vm_get_machines_by_prefix() {
    p="$1"
    a="$2"
    virsh --connect ${vm_connect} list --all 2>/dev/null | awk '{print $2}' | egrep "^$p.*$a$" | sort
}

vm_get_release_by_machine_name() {
    m="$1"
    for r in $vm_release_list ; do
        echo "$m" | grep -q "$r" && {
            echo "$r"
            return
        }
    done
}

vm_can_snapshot() {
    m="$1"
    if ! vm_exists $m ; then
        echo "'$m' does not exist" >&2
        return 1
    fi

    disks=`get_disks $m`
    if [ -z "$disks" ]; then
        echo "Could not find any disks" >&2
        return 1
    fi

    # iterate through the disks. If all disks have the pristine file, that
    # means we can do snapshots.
    for d in $disks ; do
        real=`readlink -f "$d"`
        if ! echo "$real" | egrep -q '.qcow2$' ; then
            echo "File extension for '$real' is not 'qcow2'" >&2
            return 1
        fi
        dir=`dirname "$real"`
        bn=`basename "$real" .qcow2`
        pristine="$dir/$bn.pristine.qcow2"
        if [ ! -e "$pristine" ]; then
            echo "'$pristine' does not exist" >&2
            return 1
        fi
    done
    return
}
