/*
 * Copyright (c) 2004 Frank Denis. All rights reserved.
 * 
 * This crucial part of UCARP is derived from the OpenBSD project.
 * Original copyright follows. 
 * 
 * Copyright (c) 2002 Michael Shalayeff. All rights reserved.
 * Copyright (c) 2003 Ryan McBride. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR OR HIS RELATIVES BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF MIND, USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <config.h>
#include "ucarp.h"
#include "crypto.h"
#ifndef USE_SYSTEM_CRYPT_SHA1
# include "crypto-sha1.h"
#else
# include <sha1.h>
#endif
#include "ip_carp.h"
#include "fillmac.h"
#include "garp.h"
#include "spawn.h"
#include "carp_p.h"

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

static void carp_set_state(struct carp_softc *sc, int state)
{
    if ((int) sc->sc_state == state) {
        return;
    }
    switch (state) {
    case INIT:
        puts("Switching to state : INIT");
        break;
    case BACKUP:
        puts("Switching to state : BACKUP");
        (void) spawn_handler(dev_desc_fd, downscript);
        gratuitous_arp(dev_desc_fd, 0);
        break;
    case MASTER:
        puts("Switching to state : MASTER");
        (void) spawn_handler(dev_desc_fd, upscript);
        gratuitous_arp(dev_desc_fd, 1);
        break;
    default:
        fprintf(stderr, "Unknown state : [%d]\n", (int) state);
        abort();
    }
    sc->sc_state = state;
}

void carp_hmac_prepare(struct carp_softc *sc)
{
    unsigned char version = CARP_VERSION, type = CARP_ADVERTISEMENT;
    unsigned char vhid = sc->sc_vhid & 0xff;
    size_t i;
#ifdef INET6
    struct in6_addr in6;
#endif /* INET6 */
    
    /* compute ipad from key */
    memset(sc->sc_pad, 0, sizeof sc->sc_pad);
    memcpy(sc->sc_pad, sc->sc_key, sizeof sc->sc_key);
    for (i = 0; i < sizeof sc->sc_pad; i++) {
        sc->sc_pad[i] ^= 0x36;
    }    
    /* precompute first part of inner hash */
    SHA1Init(&sc->sc_sha1);
    SHA1Update(&sc->sc_sha1, sc->sc_pad, sizeof sc->sc_pad);
    SHA1Update(&sc->sc_sha1, (void *) &version, sizeof version);
    SHA1Update(&sc->sc_sha1, (void *) &type, sizeof type);
    SHA1Update(&sc->sc_sha1, (void *) &vhid, sizeof vhid);
    SHA1Update(&sc->sc_sha1, (void *) &vaddr.s_addr, sizeof vaddr.s_addr);
    
    /* convert ipad to opad */
    for (i = 0; i < sizeof sc->sc_pad; i++) {
        sc->sc_pad[i] ^= 0x36 ^ 0x5c;
    }
}

static unsigned short cksum(const void * const buf_, size_t len)
{
    const unsigned char *buf = (unsigned char *) buf_;
    unsigned long sum = 0U;
    size_t i = (size_t) 0U;
    
    if (len <= (size_t) 0U) {
        return 0U;
    }
    if ((len & 1) != 0) {
        return 42U; /* XXX - TODO */
    }
    do {
        sum += (buf[i] << 8) | buf[i + 1];
        if (sum > 0xffff) {
            sum &= 0xffff;
            sum++;
        }
        i += 2;
    } while (i < len);
    
    return (unsigned short) ~sum;
}

static void carp_hmac_generate(struct carp_softc *sc, u_int32_t counter[2],
                               unsigned char *md)
{
    SHA1_CTX ctx;
    
    /* fetch first half of inner hash */
    memcpy(&ctx, &sc->sc_sha1, sizeof ctx);

    SHA1Update(&ctx, (void *) counter, sizeof sc->sc_counter);

    SHA1Final(md, &ctx);

    /* outer hash */
    SHA1Init(&ctx);
    SHA1Update(&ctx, sc->sc_pad, sizeof sc->sc_pad);
    SHA1Update(&ctx, md, 20);
    SHA1Final(md, &ctx);    
}

int carp_prepare_ad(struct carp_header *ch, struct carp_softc *sc)
{
    if (sc->sc_init_counter != 0) {
        /* this could also be seconds since unix epoch */
        sc->sc_counter = random();
        sc->sc_counter <<= 32;
        sc->sc_counter += random();
    } else if (sc->sc_counter == 0xffffffffffffffffULL) {
        sc->sc_counter = 0ULL;
    } else {
        sc->sc_counter++;
    }    
    ch->carp_counter[0] = htonl((sc->sc_counter >> 32) & 0xffffffff);
    ch->carp_counter[1] = htonl(sc->sc_counter & 0xffffffff);

    carp_hmac_generate(sc, ch->carp_counter, ch->carp_md);
    
    return 0;
}

static void carp_send_ad(struct carp_softc *sc)
{
    struct carp_header ch;
    struct ether_header eh;
    struct timeval tv;
    struct carp_header *ch_ptr;
    struct ether_header *eh_ptr;
    struct ip ip;
    struct ip *ip_ptr;
    unsigned char *pkt;
    unsigned short sum;    
    size_t ip_len;
    size_t eth_len;
    int advbase;
    int advskew;

#ifdef DEBUG
    puts("-> carp_send_ad()");
#endif
    advbase = sc->sc_advbase;
    advskew = sc->sc_advskew;
    tv.tv_sec = advbase;
    tv.tv_usec = advskew * 1000000 / 256;

    ch.carp_version = CARP_VERSION;
    ch.carp_type = CARP_ADVERTISEMENT;
    ch.carp_vhid = sc->sc_vhid;
    ch.carp_advbase = advbase;
    ch.carp_advskew = advskew;
    ch.carp_authlen = CARP_AUTHLEN;
    ch.carp_pad1 = 0;   /* must be zero */
    ch.carp_cksum = 0;

    ip_len = sizeof ip + sizeof ch;
    eth_len = ip_len + sizeof eh;
    if ((pkt = ALLOCA(eth_len)) == NULL) {
        perror("Out of memory to create packet");
        sc->sc_ad_tmo.tv_sec = now.tv_sec + tv.tv_sec;
        sc->sc_ad_tmo.tv_usec = now.tv_usec + tv.tv_usec;            
        return;
    }
    ip.ip_v = IPVERSION;
    ip.ip_hl = (sizeof ip) >> 2;
    ip.ip_tos = IPTOS_LOWDELAY;
    ip.ip_len = htons(ip_len);
    ip.ip_id = htons(random() & 0xffff);
    ip.ip_off = htons(IP_DF);
    ip.ip_ttl = CARP_DFLTTL;
    ip.ip_p = IPPROTO_CARP;
    ip.ip_sum = 0;
    
    memcpy(&ip.ip_src, &srcip, sizeof ip.ip_src);
    
    ip.ip_dst.s_addr = INADDR_CARP_GROUP;
    
    memcpy(pkt, &eh, sizeof eh);
    memcpy(pkt + sizeof eh, &ip, sizeof ip);
    memcpy(pkt + sizeof ip + sizeof eh, &ch, sizeof ch);
    eh_ptr = (struct ether_header *) pkt;
    ip_ptr = (struct ip *) (pkt + sizeof eh);        
    ch_ptr = (struct carp_header *) (pkt + sizeof eh + sizeof ip);
    
    carp_prepare_ad(ch_ptr, sc);       
    
    ch_ptr->carp_cksum = 0;
    sum = cksum(ch_ptr, sizeof ch);
    ch_ptr->carp_cksum = htons(sum);
    sum = cksum(ip_ptr, ip_len);
    ip_ptr->ip_sum = htons(sum);
    
    eh_ptr->ether_shost[0] = 0x00;
    eh_ptr->ether_shost[1] = 0x00;
    eh_ptr->ether_shost[2] = 0x5e;
    eh_ptr->ether_shost[3] = 0x00;
    eh_ptr->ether_shost[4] = 0x00;
    eh_ptr->ether_shost[5] = vhid;
    
    eh_ptr->ether_dhost[0] = 0xff;
    eh_ptr->ether_dhost[1] = 0xff;
    eh_ptr->ether_dhost[2] = 0xff;
    eh_ptr->ether_dhost[3] = 0xff;
    eh_ptr->ether_dhost[4] = 0xff;
    eh_ptr->ether_dhost[5] = 0xff;        
    
    eh_ptr->ether_type = htons(ETHERTYPE_IP);
    
    if (write(dev_desc_fd, pkt, eth_len) < 0) {
#ifdef DEBUG
        perror("write");
#endif
    }
    
#ifdef DEBUG
    puts("* advertisement injected *");
#endif
    
    ALLOCA_FREE(pkt);
    if (advbase != 255 || advskew != 255) {
        sc->sc_ad_tmo.tv_sec = now.tv_sec + tv.tv_sec;
        sc->sc_ad_tmo.tv_usec = now.tv_usec + tv.tv_usec;            
        /* IPv6 ? */        
    }
}

static void carp_setrun(struct carp_softc *sc, sa_family_t af)
{
    struct timeval tv;    
    
#ifdef DEBUG
    puts("carp_setrun()");
#endif
    switch (sc->sc_state) {
    case INIT:
        if (preempt != 0) {
            carp_send_ad(sc);
            gratuitous_arp(dev_desc_fd, 1);
#ifdef INET6
            carp_send_na(sc);
#endif /* INET6 */
            carp_set_state(sc, MASTER);
        } else {
            carp_set_state(sc, BACKUP);
        }
        break;
    case BACKUP:
        sc->sc_ad_tmo.tv_sec = 0;
        tv.tv_sec = (unsigned int) sc->sc_advbase * dead_ratio;
        tv.tv_usec = sc->sc_advskew * 1000000U / 256U;
        switch (af) {        
        case AF_INET:
            sc->sc_md_tmo.tv_sec = now.tv_sec + tv.tv_sec;
            sc->sc_md_tmo.tv_usec = now.tv_usec + tv.tv_usec;            
            break;
#ifdef INET6
        case AF_INET6:
            sc->sc_md6_tmo.tv_sec = now.tv_sec + tv.tv_sec;
            sc->sc_md6_tmo.tv_usec = now.tv_usec + tv.tv_usec;            
            break;
#endif /* INET6 */
        default:
            if (sc->sc_md_tmo.tv_sec != 0) {
                sc->sc_md_tmo.tv_sec = now.tv_sec + tv.tv_sec;
                sc->sc_md_tmo.tv_usec = now.tv_usec + tv.tv_usec;
            }
            if (sc->sc_md6_tmo.tv_sec != 0) {
                sc->sc_md6_tmo.tv_sec = now.tv_sec + tv.tv_sec;
                sc->sc_md6_tmo.tv_usec = now.tv_usec + tv.tv_usec;
            }
            break;
        }        
        break;
    case MASTER:
        tv.tv_sec = (unsigned int) sc->sc_advbase;
        tv.tv_usec = sc->sc_advskew * 1000000U / 256U;
        sc->sc_md_tmo.tv_sec = now.tv_sec + tv.tv_sec;
        sc->sc_md_tmo.tv_usec = now.tv_usec + tv.tv_usec;
        /* No IPv6 scheduling ? */
        break;
    }
}

static void carp_master_down(struct carp_softc *sc)
{
#ifdef DEBUG
    puts("carp_master_down()");
#endif
    switch (sc->sc_state) {
    case INIT:
#ifdef DEBUG
        fprintf(stderr, "master_down event in INIT state\n");
#endif
        break;
    case MASTER:
        break;
    case BACKUP:
        carp_set_state(sc, MASTER);
        carp_send_ad(sc);
        gratuitous_arp(dev_desc_fd, 0);
#ifdef INET6
        carp_send_na(sc);
#endif /* INET6 */
        carp_setrun(sc, 0);
        break;
    }    
}

static void packethandler(unsigned char *dummy,
                          const struct pcap_pkthdr *header,
                          const unsigned char *sp)
{
    struct ether_header etherhead;
    struct ip iphead;
    unsigned int source;
    unsigned int dest;
    unsigned char proto;
    unsigned int caplen;
    unsigned int ip_len;
            
    (void) dummy;
    if (header->caplen < (sizeof etherhead + sizeof iphead)) {
        return;
    }    
    memcpy(&etherhead, sp, sizeof etherhead);
#ifdef DEBUG
    printf("Ethernet "
           "[%02x:%02x:%02x:%02x:%02x:%02x]->[%02x:%02x:%02x:%02x:%02x:%02x] "
           "type [%04x]\n",
           (unsigned int) etherhead.ether_shost[0],
           (unsigned int) etherhead.ether_shost[1],
           (unsigned int) etherhead.ether_shost[2],
           (unsigned int) etherhead.ether_shost[3],
           (unsigned int) etherhead.ether_shost[4],
           (unsigned int) etherhead.ether_shost[5],
           (unsigned int) etherhead.ether_dhost[0],
           (unsigned int) etherhead.ether_dhost[1],
           (unsigned int) etherhead.ether_dhost[2],
           (unsigned int) etherhead.ether_dhost[3],
           (unsigned int) etherhead.ether_dhost[4],
           (unsigned int) etherhead.ether_dhost[5],
           (unsigned int) ntohs(etherhead.ether_type));
#endif
    sp += sizeof etherhead;
    caplen = header->caplen - sizeof etherhead;
    memcpy(&iphead, sp, sizeof iphead);    
    ip_len = iphead.ip_hl << 2;
    source = ntohl(iphead.ip_src.s_addr);
    dest = ntohl(iphead.ip_dst.s_addr);
    proto = iphead.ip_p;    
    switch (proto) {
    case IPPROTO_CARP: {
        struct carp_header ch;
        unsigned long long tmp_counter;
        struct timeval sc_tv;
        struct timeval ch_tv;        

#ifdef DEBUG
        printf("\n---------------------------\n\n"
               "carp [%d.%d.%d.%d] -> [%d.%d.%d.%d]\n",
               source >> 24 & 0xff, source >> 16 & 0xff,
               source >> 8 & 0xff, source & 0xff,
               dest >> 24 & 0xff, dest >> 16 & 0xff,
               dest >> 8 & 0xff, dest & 0xff);
#endif
        if (caplen != ip_len + sizeof ch) {
#ifdef DEBUG
            fprintf(stderr,
                    "Bogus size : caplen=[%u], ip_len=[%u] ch_len=[%u]\n",
                    (unsigned int) caplen, (unsigned int) ip_len,
                    (unsigned int) sizeof ch);
#endif
            return;
        }
        memcpy(&ch, sp + ip_len, sizeof ch);
#ifdef EXTRA_DEBUG
        printf("Capture len : [%u] carp_header : [%u] ip_header : [%u]\n",
               (unsigned int) caplen, (unsigned int) sizeof ch,
               (unsigned int) ip_len);
        printf("CARP type : [%u] version : [%u]\n",
               (unsigned) ch.carp_type, (unsigned) ch.carp_version);
        printf("CARP vhid : [%u] advskew : [%u]\n",
               (unsigned) ch.carp_vhid, (unsigned) ch.carp_advskew);
        printf("CARP advbase : [%u] cksum : [%u]\n",
               (unsigned) ch.carp_advbase, (unsigned) ch.carp_cksum);
        printf("CARP counter : [%lu][%lu]\n",
               (unsigned long) ch.carp_counter[0],
               (unsigned long) ch.carp_counter[1]);
#endif
        if (iphead.ip_ttl != CARP_DFLTTL) {
            fprintf(stderr, "Bad ttl : [%u]\n", (unsigned int) iphead.ip_ttl);
            return;
        }        
        if (ch.carp_version != CARP_VERSION) {
            fprintf(stderr, "Bad version : [%u]\n",
                    (unsigned int) ch.carp_version);
            return;
        }
        if (ch.carp_vhid != vhid) {
#ifdef DEBUG
            fprintf(stderr, "Ignoring vhid : [%u]\n",
                    (unsigned int) ch.carp_vhid);
#endif
            return;            
        }
        if (cksum(sp, caplen) != 0) {
            fprintf(stderr, "Bad IP checksum\n");
            return;
        }        
        {
            SHA1_CTX ctx;
            unsigned char md2[20];            
     
            memcpy(&ctx, &sc.sc_sha1, sizeof ctx);
            SHA1Update(&ctx, 
                       (void *) &ch.carp_counter, sizeof ch.carp_counter);
            SHA1Final(md2, &ctx);
            
            SHA1Init(&ctx);
            SHA1Update(&ctx, sc.sc_pad, sizeof(sc.sc_pad));
            SHA1Update(&ctx, md2, sizeof md2);
            SHA1Final(md2, &ctx);
            
            if (sizeof md2 != sizeof ch.carp_md) {
                fprintf(stderr, "sizeof md2 != sizeof carp_md !!!\n");
                abort();
            }
            if (memcmp(md2, ch.carp_md, sizeof md2) != 0) {
                fprintf(stderr, "Bad digest - "
                        "md2=[%02x%02x%02x%02x...] md=[%02x%02x%02x%02x...]\n"
                        "Check vhid, password and virtual IP address\n",
                        (unsigned int) md2[0], (unsigned int) md2[1],
                        (unsigned int) md2[2], (unsigned int) md2[3],
                        (unsigned int) (ch.carp_md)[0],
                        (unsigned int) (ch.carp_md)[1],
                        (unsigned int) (ch.carp_md)[2],
                        (unsigned int) (ch.carp_md)[3]);
                return;
            }
        }
        tmp_counter = ntohl(ch.carp_counter[0]);
        tmp_counter = tmp_counter << 32;
        tmp_counter += ntohl(ch.carp_counter[1]);
        if (sc.sc_init_counter != 0) {
            sc.sc_init_counter = 0;
        }
        sc.sc_counter = tmp_counter;
        
        sc_tv.tv_sec = (unsigned int) sc.sc_advbase;
        sc_tv.tv_usec = sc.sc_advskew * 1000000U / 256U;
        ch_tv.tv_sec = (unsigned int) ch.carp_advbase;
        ch_tv.tv_usec = ch.carp_advskew * 1000000U / 256U;
        
#ifdef DEBUG
        printf("sc_tv(local) = [%lu] ch_tv(remote) = [%lu]\n",
               (unsigned long) sc_tv.tv_sec,
               (unsigned long) ch_tv.tv_sec);
#endif
        
        switch (sc.sc_state) {
        case INIT:
                break;
        case MASTER:
            /*
             * If we receive an advertisement from a master who's going to
             * be more frequent than us, go into BACKUP state.
             */
            if (timercmp(&sc_tv, &ch_tv, >) ||
                timercmp(&sc_tv, &ch_tv, ==)) {
                sc.sc_ad_tmo.tv_sec = 0;
                carp_set_state(&sc, BACKUP);
                carp_setrun(&sc, 0);
                puts("Going back to BACKUP state");
            }
            break;
        case BACKUP:
            /*
             * If we're pre-empting masters who advertise slower than us,
             * and this one claims to be slower, treat him as down.
             */
            if (preempt != 0 && timercmp(&sc_tv, &ch_tv, <)) {
                carp_master_down(&sc);
                puts("Putting MASTER down - preemption");                
                break;
            }
            
            /*
             *  If the master is going to advertise at such a low frequency
             *  that he's guaranteed to time out, we'd might as well just
             *  treat him as timed out now.
             */
            sc_tv.tv_sec = (unsigned int) sc.sc_advbase * dead_ratio;
            if (timercmp(&sc_tv, &ch_tv, <)) {
                carp_master_down(&sc);
                puts("Putting MASTER DOWN (going to time out)");
                break;
            }
            
            /*
             * Otherwise, we reset the counter and wait for the next
             * advertisement.
             */
            carp_setrun(&sc, AF_INET); /* XXX - TODO was af */
            break;
        }
        break;
    }
    default:
        return;
    }
}     

int docarp(void)
{
    struct bpf_program bpfp;
    struct pollfd pfds[1];
    int nfds;
    char errbuf[PCAP_ERRBUF_SIZE];

    sc.sc_vhid = vhid;
    sc.sc_advbase = advbase;
    sc.sc_advskew = advskew;
    sc.sc_init_counter = 1;
    sc.sc_naddrs = sc.sc_naddrs6 = 0;
#ifdef INET6
    sc.sc_im6o.im6o_multicast_hlim = CARP_DFLTTL;
#endif /* INET6 */
    carp_set_state(&sc, INIT);
    {
        const size_t passlen = strlen(pass) + (size_t) 1U;
        
        if (passlen > sizeof sc.sc_key) {
            fprintf(stderr, "Password too long\n");
            return -1;
        }
        memcpy(sc.sc_key, pass, passlen);
    }
    sc.sc_ad_tmo.tv_sec = 0;
    sc.sc_ad_tmo.tv_usec = 0;    
    sc.sc_md_tmo.tv_sec = 0;
    sc.sc_md6_tmo.tv_usec = 0;    
    
    carp_hmac_prepare(&sc);

    if (fill_mac_address() != 0) {
        fprintf(stderr, "Unable to manage interface [%s]\n", interface);
        return -1;
    }
    printf("Local advertised ethernet address is "
           "[%02x:%02x:%02x:%02x:%02x:%02x]\n",
           (unsigned int) hwaddr[0], (unsigned int) hwaddr[1],
           (unsigned int) hwaddr[2], (unsigned int) hwaddr[3],
           (unsigned int) hwaddr[4], (unsigned int) hwaddr[5]);
    if ((dev_desc = pcap_open_live(interface, ETHERNET_MTU, 1,
                                   CAPTURE_TIMEOUT, errbuf)) == NULL) {
        fprintf(stderr, "pcap_open_live() : %s [%s]\n", errbuf, interface);
        return -1;
    }    
    if (pcap_compile(dev_desc, &bpfp, (char *) BPF_CARP_RULE,
                     1, (bpf_u_int32) 0) != 0) {
        fprintf(stderr, "pcap_compile() : %s [%s]\n", errbuf, interface);
        return -1;
    }
    pcap_setfilter(dev_desc, &bpfp);
    dev_desc_fd = pcap_fileno(dev_desc);
    pfds[0].fd = dev_desc_fd;
    pfds[0].events = POLLIN | POLLERR | POLLHUP | POLLNVAL;
    
    carp_setrun(&sc, 0);
    for (;;) {
        nfds = poll(pfds, (nfds_t) 1, (int) (sc.sc_advbase * 1000));
        if (nfds == -1 || 
            (pfds[0].revents & (POLLERR | POLLHUP | POLLNVAL)) != 0) {
#ifdef DEBUG
            perror("poll error");
#endif
            break;
        }
        if (gettimeofday(&now, NULL) != 0) {
            perror("gettimeofday()");
            continue;
        }        
        if (nfds == 1) {
            pcap_dispatch(dev_desc, 1, packethandler, NULL);
        }
        if (sc.sc_md_tmo.tv_sec != 0 && timercmp(&now, &sc.sc_md_tmo, >)) {
            carp_master_down(&sc);
        }
        if (sc.sc_md6_tmo.tv_sec != 0 && timercmp(&now, &sc.sc_md6_tmo, >)) {
            carp_master_down(&sc);
        }
        if (sc.sc_ad_tmo.tv_sec != 0 && timercmp(&now, &sc.sc_ad_tmo, >)) {
            carp_send_ad(&sc);
        }
    }
    pcap_close(dev_desc);
    pcap_freecode(&bpfp);
    
    return 0;
}
