/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_api/DocumentModel.h>
#include <core_api/GObject.h>
#include <core_api/Task.h>
#include <core_api/IOAdapter.h>
#include <core_api/GHints.h>

#include <QtCore/QFileInfo>

/* TRANSLATOR GB2::DocumentFormat */

namespace GB2 {

const QString DocumentFormat::CREATED_NOT_BY_UGENE= DocumentFormat::tr( "document_is_created_not_by_ugene" );
const QString DocumentMimeData::MIME_TYPE("application/x-ugene-document-mime");

//objects hints are the part of the document hints.
//reason: keep hints save for unloaded document 
class DocManagedGObjectHints : public GHints {
public:
    DocManagedGObjectHints(Document* _doc, GObject* obj);
    ~DocManagedGObjectHints(); 
    virtual QVariantMap getMap() const;
    virtual void setMap(const QVariantMap& map);
    virtual QVariant get(const QString& key) const;
    virtual void set(const QString& key, const QVariant& val);
    virtual int remove(const QString& key);

    void cleanup();

private:
    Document*   doc;
    QString     prefix;
};


Document* DocumentFormat::createNewDocument(IOAdapterFactory* io, const QString& url, const QVariantMap& hints) {
    Document* d = new Document(this, io, url, hints);
    d->setLoaded(true);
    return d;
}


//////////////////////////////////////////////////////////////////////////
///Document

Document::Document(DocumentFormat* _df, IOAdapterFactory* _io, const QString& _url, 
                   const QVariantMap& hints, const QString& instanceModLockDesc)
: StateLockableTreeItem(), df(_df), io(_io), url(_url)
{
    ctxState = new GHintsDefaultImpl(hints);

    QFileInfo fi(url);
    name = fi.fileName();
    std::fill(modLocks, modLocks + DocumentModLock_NUM_LOCKS, (StateLock*)NULL);    
    initModLocks(instanceModLockDesc, false);
}

Document::Document(DocumentFormat* _df, IOAdapterFactory* _io, const QString& _url, 
                   const QList<GObject*>& _objects, const QVariantMap& hints, const QString& instanceModLockDesc)
: StateLockableTreeItem(), df(_df), io(_io), url(_url)
{
    ctxState = new GHintsDefaultImpl(hints);
    QFileInfo fi(url);
    name = fi.fileName();
    
    std::fill(modLocks, modLocks + DocumentModLock_NUM_LOCKS, (StateLock*)NULL);

    foreach(GObject* o, _objects) {
        _addObject(o);
    }

    initModLocks(instanceModLockDesc, true);
}

Document::~Document() {
    for (int i=0;i<DocumentModLock_NUM_LOCKS; i++) {
        StateLock* sl = modLocks[i];
        if (sl!=NULL) {
            unlockState(sl);
            delete sl;
        }
    }
    foreach(GObject* obj, objects) {
        obj->setGHints(new GHintsDefaultImpl());
    }
    delete ctxState;
}

void Document::addObject(GObject* obj){
    assert(obj != NULL && obj->getDocument()==NULL);
    assert(df->isObjectOpSupported(this, DocumentFormat::DocObjectOp_Add, obj->getGObjectType()));
    assert(findGObjectByName(obj->getGObjectName())==NULL);

    _addObject(obj);
}

void Document::_addObject(GObject* obj) {
    assert(!isStateLocked());
    
    obj->setParentStateLockItem(this);
    obj->setGHints(new DocManagedGObjectHints(this, obj));
    objects.append(obj);

    assert(objects.size() == getChildItems().size());
    emit si_objectAdded(obj);
}

void Document::removeObject(GObject* obj) {
    assert(df->isObjectOpSupported(this, DocumentFormat::DocObjectOp_Remove, obj->getGObjectType()));
    assert(obj->getParentStateLockItem() == this);
    
    obj->setParentStateLockItem(NULL);
    objects.removeOne(obj);
    DocManagedGObjectHints* h = (DocManagedGObjectHints*)obj->getGHints();
    h->cleanup();
    obj->setGHints(new GHintsDefaultImpl());

    assert(objects.size() == getChildItems().size());
    emit si_objectRemoved(obj);
}


void Document::makeClean()  {
    if (!isTreeItemModified()) {
        return;
    }
    setModified(false);
    foreach(GObject* obj, objects) {
        obj->setModified(false);
    }
}



GObject* Document::findGObjectByName(const QString& name) const {
    foreach(GObject* obj, objects) {
        if (obj->getGObjectName() == name) {
            return obj;
        }
    }
    return NULL;
}

QList<GObject*> Document::findGObjectByType(GObjectType t) const {
    QList<GObject*> res;
    foreach(GObject* obj, objects) {
        if (obj->getGObjectType() == t) {
            res.append(obj);
        }
    }
    return res;
}

Document* Document::clone() const {
    Document* doc = new Document(df, io, url, ctxState->getMap());
    doc->loadFrom(this);
    return doc;
}

void Document::loadFrom(const Document* d) {
    assert(d->getDocumentFormat() == df);
    assert(!isLoaded() && d->isLoaded());
    assert(objects.isEmpty());
    assert(!isTreeItemModified());
    
    ctxState->setMap(d->getGHints()->getMap());

    //clean all mod locks
    for(int i = 0; i < DocumentModLock_NUM_LOCKS; i++) {
        if (modLocks[i]!=0) {
            unlockState(modLocks[i]);
        }
    }

    QList<GObject*> objects = d->getObjects();
    foreach(GObject* o, objects) {
        o = o->clone();
        _addObject(o);
    }
    
    //copy instance modlocks if any
    if (d->modLocks[DocumentModLock_FORMAT_AS_INSTANCE]!=NULL) {
        if (modLocks[DocumentModLock_FORMAT_AS_INSTANCE]!=NULL) {
            delete modLocks[DocumentModLock_FORMAT_AS_INSTANCE];
        }
        modLocks[DocumentModLock_FORMAT_AS_INSTANCE] = new StateLock(d->modLocks[DocumentModLock_FORMAT_AS_INSTANCE]->getUserDesc());
    }
    
    //restore locks
    for(int i = 0; i < DocumentModLock_NUM_LOCKS; i++) {
        if (modLocks[i] != 0) { //was locked
            lockState(modLocks[i]);
        }
    }


    //TODO: rebind local objects relations if url!=d.url

    setModified(false);
    setLoaded(true); 
}

void Document::setLoaded(bool v) {
    if (v == isLoaded()) {
        return;
    }
    StateLock* l = modLocks[DocumentModLock_UNLOADED_STATE];
    if (v) {
        unlockState(l);
        modLocks[DocumentModLock_UNLOADED_STATE] = NULL;
        delete l;
    } else {
        assert(l == NULL);
        l = new StateLock(tr("unloaded_lock"));
        modLocks[DocumentModLock_UNLOADED_STATE] = l;
        lockState(l);
        assert(objects.isEmpty());
    }
    emit si_loadedStateChanged();
}

void Document::initModLocks(const QString& instanceModLockDesc, bool loaded) {
    setLoaded(loaded);
    
    // must be locked for modifications if io-adapter does not support writes
    if (!io->isIOModeSupported(IOAdapterMode_Write)) {
        modLocks[DocumentModLock_IO] = new StateLock(tr("io_adapter_does_not_support_writes"));
        lockState(modLocks[DocumentModLock_IO]);
    }

    // must be locked for modifications if not document format does not support writes
    DocumentFormatConstraints c;
    c.mustSupportWrite = true;
    if (!df->checkConstraints(c)) {
        modLocks[DocumentModLock_FORMAT_AS_CLASS] = new StateLock(tr("document_format_does_not_support_writes"));
        lockState(modLocks[DocumentModLock_FORMAT_AS_CLASS]);
    }

    if (!instanceModLockDesc.isEmpty()) {
        modLocks[DocumentModLock_FORMAT_AS_INSTANCE] = new StateLock(instanceModLockDesc);
        lockState(modLocks[DocumentModLock_FORMAT_AS_INSTANCE]);
    }
}

void Document::setURL(const QString& newUrl) {
    assert(!isStateLocked());
    if (url == newUrl) {
        return;
    }
    url = newUrl;
    QFileInfo fi(url);
    name = fi.fileName();
    emit si_urlChanged();
}

bool Document::checkConstraints(const Document::Constraints& c) const {
    if (c.stateLocked != TriState_Unknown) {
        if (c.stateLocked == TriState_No && isStateLocked()) {
            return false;
        }
        if (c.stateLocked == TriState_Yes && !isStateLocked()) {
            return false;
        }
    }

    if (!c.formats.isEmpty()) {
        bool found = false;
        foreach(DocumentFormatId f, c.formats) {
            if (df->getFormatId() == f) {
                found = true;
                break;
            }
        }
        if (!found) {
            return false;
        }
    }

    foreach(DocumentModLock l, c.notAllowedStateLocks) {
        if (modLocks[l]!=NULL) {
            return false;
        }
    }
    
    if (!c.objectTypeToAdd.isNull() && !df->isObjectOpSupported(this, DocumentFormat::DocObjectOp_Add, c.objectTypeToAdd)) {
        return false;
    }

    
    return true;
}


void Document::setUserModLock(bool v) {
    if (hasUserModLock() == v) {
        return;
    }
    if (v) {
        StateLock* sl = new StateLock(tr("user_force_read_only_lock"));
        modLocks[DocumentModLock_USER] = sl;
        lockState(sl);
    } else {
        StateLock* sl = modLocks[DocumentModLock_USER];
        modLocks[DocumentModLock_USER] = NULL;
        unlockState(sl);
        delete sl;
    }

    //hack: readonly settings are stored in project, so if document is in project -> mark project as modified
    if (getParentStateLockItem()!=NULL) { 
        getParentStateLockItem()->setModified(true);
    }
}

bool Document::unload() {
    assert(isLoaded());
    
    bool liveLocked = hasLocks(StateLockableTreeFlags_ItemAndChildren, StateLockFlag_LiveLock);
    if (liveLocked) {
        return false;
    }


    //clean all mod locks
    for(int i = 0; i < DocumentModLock_NUM_LOCKS; i++) {
        StateLock* sl = modLocks[i];
        if (sl != 0) {
            unlockState(sl);
            if (i == DocumentModLock_FORMAT_AS_INSTANCE) {
                delete sl;
                modLocks[i] = NULL;
            }
        }
    }

    QList<GObject*> objs = objects;
    while (!objects.isEmpty()) {
        GObject* obj = objects.first();
        obj->setParentStateLockItem(NULL);
        objects.removeOne(obj);
        assert(objects.size() == getChildItems().size());
    }
    
    //restore locks
    for(int i = 0; i < DocumentModLock_NUM_LOCKS; i++) {
        if (modLocks[i] != 0) { //was locked
            lockState(modLocks[i]);
        }
    }

    foreach(GObject* obj, objs) {
        emit si_objectRemoved(obj);    
    }
    foreach(GObject* obj, objs) {
        delete obj;
    }

    setLoaded(false);
    setModified(false);

    return true;
}

void Document::setGHints(GHints* newHints) {
    assert(newHints!=NULL);

    //gobjects in document keep states in parent document map -> preserve gobject hints
    QList<QVariantMap> objStates;
    for (int i=0;i<objects.size();i++) {
        objStates.append(objects[i]->getGHintsMap());
    }
    
    delete ctxState;
    ctxState = newHints;

    for (int i=0;i<objects.size();i++) {
        objects[i]->getGHints()->setMap(objStates[i]);
    }

}


QVariantMap Document::getGHintsMap() const {
    return ctxState->getMap();
}

//////////////////////////////////////////////////////////////////////////
// hints

DocManagedGObjectHints::DocManagedGObjectHints(Document* _doc, GObject* obj) : doc(_doc) {
    prefix = obj->getGObjectName() + "#";
    QVariantMap docMap = getMap(); //from unloaded state
    QVariantMap objMap = obj->getGHintsMap(); // from loaded state
    QVariantMap joinedMap = docMap.unite(objMap);
    setMap(joinedMap);
}

QVariantMap DocManagedGObjectHints::getMap() const {
    QVariantMap docMap = doc->getGHints()->getMap();
    QVariantMap objMap;
    foreach(const QString& key, docMap.keys()) {
        if (key.startsWith(prefix)) {
            QString objLocalKey = key.mid(prefix.length());
            objMap[objLocalKey] = docMap[key];
        }
    }
    return objMap;
}

DocManagedGObjectHints::~DocManagedGObjectHints() {
    //skip cleanup -> object can be deleted during unload
}

void DocManagedGObjectHints::cleanup() {
    foreach(const QString& k, getMap().keys()) {
        remove(k);
    }
}

void DocManagedGObjectHints::setMap(const QVariantMap& newMap) {
    QVariantMap prevMap = getMap();
    foreach(const QString& key, prevMap.keys()) {
        remove(key);
    }
    foreach(const QString& key, newMap.keys()) {
        set(key, newMap.value(key));
    }
}

QVariant DocManagedGObjectHints::get(const QString& key) const {
    return doc->getGHints()->get(prefix + key);
}

void DocManagedGObjectHints::set(const QString& key, const QVariant& val) {
    doc->getGHints()->set(prefix + key, val);
}

int DocManagedGObjectHints::remove(const QString& key) {
    return doc->getGHints()->remove(prefix + key);
}

}//namespace

