/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_DOCUMENT_MODEL_H_
#define _GB2_DOCUMENT_MODEL_H_

#include "core_api.h"
#include "StateLockableDataModel.h"

#include <QtCore/QMimeData>
#include <QtCore/QPointer>

namespace GB2 {

class TaskStateInfo;

class Document;
class GObject;
class DocumentFormat;
class IOAdapterFactory;
class DocumentFormatConstraints;
class GHints;

class GB2_COREAPI_EXPORT DocumentFormat: public QObject {
	Q_OBJECT
public:
	static const QString CREATED_NOT_BY_UGENE;
	
	enum DocObjectOp {
		DocObjectOp_Add,
		DocObjectOp_Remove
	};


	DocumentFormat(QObject* p) : QObject(p) {}

	virtual DocumentFormatId getFormatId() const = 0;

	virtual const QString& getFormatName() const = 0;

	virtual QStringList getSupportedDocumentFileExtensions() = 0;

	virtual Document* createNewDocument(IOAdapterFactory* io, const QString& url, const QVariantMap& hints = QVariantMap());

	virtual Document* loadExistingDocument(IOAdapterFactory* io, const QString& url, TaskStateInfo& ti, const QVariantMap& hints) = 0;

	virtual void storeDocument(Document* d, TaskStateInfo& ts, IOAdapterFactory* io = NULL, const QString& newDocURL = QString::null) = 0;

    virtual bool isObjectOpSupported(const Document* d, DocObjectOp op, GObjectType t) const = 0;

	virtual bool checkConstraints(const DocumentFormatConstraints& c) const = 0;
    
    virtual void updateFormatSettings(Document* d) const {Q_UNUSED(d);}
};

class GB2_COREAPI_EXPORT DocumentFormatRegistry  : public QObject {
	Q_OBJECT
public:
	DocumentFormatRegistry(QObject* p = NULL) : QObject(p) {}

	virtual bool registerFormat(DocumentFormat* dfs) = 0;

	virtual bool unregisterFormat(DocumentFormat* dfs) = 0;

	virtual QList<DocumentFormatId> getRegisteredFormats() const = 0;

	virtual DocumentFormat* getFormatById(DocumentFormatId id) const = 0;

	virtual QList<DocumentFormatId> selectFormats(const DocumentFormatConstraints& c) const = 0;
signals:
    void si_documentFormatRegistered(DocumentFormat*);
    void si_documentFormatUnregistered(DocumentFormat*);
};

enum DocumentModLock {
    DocumentModLock_IO,
    DocumentModLock_USER,
    DocumentModLock_FORMAT_AS_CLASS,
    DocumentModLock_FORMAT_AS_INSTANCE,
    DocumentModLock_UNLOADED_STATE,
    DocumentModLock_NUM_LOCKS
};

class GB2_COREAPI_EXPORT Document : public  StateLockableTreeItem {
	Q_OBJECT

public:
	class Constraints {
	public:
        Constraints() : stateLocked(TriState_Unknown) {}
        TriState                stateLocked;
        QList<DocumentModLock>  notAllowedStateLocks; // if document contains one of these locks -> it's not matched
		QList<DocumentFormatId> formats;              // document format must be in list to match
		GObjectType             objectTypeToAdd;      // document must be ready to add objects of the specified type
	};


	Document(DocumentFormat* _df, IOAdapterFactory* _io, const QString& _url, 
        const QVariantMap& hints = QVariantMap(), const QString& instanceModLockDesc = QString());

    Document(DocumentFormat* _df, IOAdapterFactory* _io, const QString& _url, 
                    const QList<GObject*>& objects, const QVariantMap& hints = QVariantMap(), 
                    const QString& instanceModLockDesc = QString());

	virtual ~Document();

	DocumentFormat* getDocumentFormat() const {return df;}
	
	DocumentFormatId getDocumentFormatId() const {return df->getFormatId();}
	
	IOAdapterFactory* getIOAdapterFactory() const {return io;}
	
	const QList<GObject*>& getObjects() const {return objects;}
	
	void addObject(GObject* ref);
	
	void removeObject(GObject* o);

	const QString& getName() const {return name;}

	const QString& getURL() const {return url;}

	void setURL(const QString& newUrl);

	void makeClean();

	GObject* findGObjectByName(const QString& name) const;

	QList<GObject*> findGObjectByType(GObjectType t) const;

	bool isLoaded() const {return modLocks[DocumentModLock_UNLOADED_STATE] == 0;}

    void setLoaded(bool v);

	void loadFrom(const Document* d);

    bool unload();

	Document* clone() const;

	bool checkConstraints(const Constraints& c) const;
    
    GHints* getGHints() const {return ctxState;}

    void setGHints(GHints* state);

    QVariantMap getGHintsMap() const;

    StateLock* getDocumentModLock(DocumentModLock type) const {return modLocks[type];}

    bool hasUserModLock() const {return modLocks[DocumentModLock_USER]!=NULL;}

    void setUserModLock(bool v);

    bool isModified() const { return isTreeItemModified(); }

protected:

    void _addObject(GObject* obj);

    void initModLocks(const QString& instanceModLockDesc, bool loaded);

    DocumentFormat* const		df;
	IOAdapterFactory* const		io;
	
	QString				url;
	QString				name; /* display name == short pathname, excluding the path */
	QList<GObject*>		objects;
    GHints*             ctxState;
    
    StateLock*          modLocks[DocumentModLock_NUM_LOCKS];

signals:
	void si_urlChanged();
	void si_nameChanged();
	void si_objectAdded(GObject* o);
	void si_objectRemoved(GObject* o);
	void si_loadedStateChanged();
};

class DocumentFormatConstraints {
public:
    DocumentFormatConstraints() : mustSupportWrite(false), checkRawData(false){}

    QList<GObjectType>  supportedObjectTypes;
    bool				mustSupportWrite;
    bool				checkRawData;
    QByteArray			rawData;
};

//TODO: decide if to use filters or constraints. May be it worth to remove Document::Constraints at all..

class GB2_COREAPI_EXPORT DocumentFilter {
public:
    virtual ~DocumentFilter(){};
    virtual bool matches(Document* doc) const = 0;
};

class GB2_COREAPI_EXPORT DocumentConstraintsFilter : public DocumentFilter {
public:
    DocumentConstraintsFilter(const Document::Constraints& _c) : constraints(_c){}
    
    virtual bool matches(Document* doc) const {
        return doc->checkConstraints(constraints);
    }

protected:
    Document::Constraints constraints;
};

class GB2_COREAPI_EXPORT DocumentMimeData : public QMimeData {
    Q_OBJECT
public:
    static const QString MIME_TYPE;
    DocumentMimeData(Document* obj) : objPtr(obj){};
    QPointer<Document> objPtr;
};

} //namespace

#endif


