/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_api/Log.h>
#include "BioStruct3D.h"

namespace GB2 { 

static LogCategory log(ULOG_CAT_IO);
    
QString BioStruct3D::MoleculeAnnotationTag("chain_");
QString BioStruct3D::AlphaHelixAnnotationTag("alpha_helix");
QString BioStruct3D::BetaStrandAnnotationTag("beta_strand");
QString BioStruct3D::TurnAnnotationTag("turn");
QString BioStruct3D::ChainIdQualifierName("chain_id");


void BioStruct3D::calcCenterAndMaxDistance() {

	Vector3D siteSum;
	Vector3D center;
	double dist;
	maxDistFromCenter = 0.0;
	int numberOfAtoms = getNumberOfAtoms();
	// loop trough all atoms twice - once to get average center, then once to
	// find max distance from this center
	for (int i = 0; i<2; ++i) {
        foreach (SharedMolecule molecule, moleculeMap) {
            foreach (SharedAtom atom, molecule->atomMap) {
			    Vector3D site = atom->coord3d;
			    if (i==0) 
				    siteSum += atom->coord3d;
			    else {
				    dist = (site - center).length();
				    if (dist > maxDistFromCenter)
					    maxDistFromCenter = dist;
			    }
            }
		}
        
		if (i == 0)
			center = siteSum / numberOfAtoms;

	}
	log.trace(QString("center: (%1,%2,%3)\n maxDistFromCenter: %4").arg(center.x).arg(center.y).arg(center.z).arg(maxDistFromCenter) );

	rotationCenter = center;

} 

int BioStruct3D::getNumberOfAtoms() const
{
    int numOfAtoms = 0;
    foreach (SharedMolecule mol, moleculeMap) {
        numOfAtoms += mol->atomMap.size();
    }
    numOfAtoms += hetAtomMap.size();

    return numOfAtoms;
}


QByteArray BioStruct3D::getRawSequence() const
{
	QByteArray sequence("");

    foreach(SharedMolecule molecule, moleculeMap) {
        foreach ( SharedResidue residue, molecule->residueMap) {
	    	QChar c = residue->acronym;
		    sequence.append(c);
	    }
    }

	return sequence;
}

void BioStruct3D::generateAnnotations()
{
    generateChainAnnotations();
    generateSecStructureAnnotations();
}

void BioStruct3D::generateChainAnnotations()
{
    const char* molNameQualifier = "molecule_name";
    //const char* pdbChainIdQualifier = "pdb_id";
    
    QMap<int, SharedMolecule>::iterator iter = moleculeMap.begin();
    while (iter != moleculeMap.end()) {
        SharedAnnotationData sd( new AnnotationData);
        sd->location.append(LRegion(0,0));
        sd->name = BioStruct3D::MoleculeAnnotationTag + QString("%1").arg(iter.key()) + "_info";
        sd->qualifiers.append(Qualifier(ChainIdQualifierName, QString("%1").arg(iter.key()) ));
        sd->qualifiers.append(Qualifier(molNameQualifier, (*iter)->name));
        
        (*iter)->annotations.append(sd);
        ++iter;
    }
    
}

int BioStruct3D::getNumberOfResidues() const
{
    int numResidues = 0;

    foreach (SharedMolecule mol, moleculeMap) {
        numResidues += mol->residueMap.size();
    }

    return numResidues;
}

const SharedAtom BioStruct3D::getAtomById( int index ) const
{
    foreach (SharedMolecule mol, moleculeMap) {
        if (mol->atomMap.contains(index)) {
            return mol->atomMap.value(index);
        }
    }

   if (hetAtomMap.contains(index))
            return hetAtomMap.value(index);
   
   return SharedAtom(NULL);
}

const SharedResidue BioStruct3D::getResidueById( int chainIndex, int residueIndex ) const
{
    Q_UNUSED(residueIndex);
    const SharedMolecule mol =  moleculeMap.value(chainIndex);
    if (mol->residueMap.contains(chainIndex)) {
        return mol->residueMap.value(chainIndex);
    }

    return SharedResidue(NULL);   
}


const QString BioStruct3D::getSecStrucAnnotationName( SecondaryStructureData::Type type )
{
    if (type== SecondaryStructureData::TYPE_HELIX)
        return BioStruct3D::AlphaHelixAnnotationTag;
    else if( type == SecondaryStructureData::TYPE_STRAND)
        return BioStruct3D::BetaStrandAnnotationTag;
    else if( type == SecondaryStructureData::TYPE_TURN)
        return BioStruct3D::TurnAnnotationTag;
    else
        return QString("unknown");


}


void BioStruct3D::generateSecStructureAnnotations()
{
    
    foreach (const SharedSecondaryStructure struc, secondaryStructures) {
        SharedAnnotationData sd(NULL); 
        int chainId = struc->chainIndex;
        int initResidueId = moleculeMap.value(chainId)->residueMap.constBegin().key();    
        sd = new AnnotationData;
        sd->name = getSecStrucAnnotationName(struc->type);
        int numResidues = struc->endSequenceNumber - struc->startSequenceNumber + 1;
        int startIndex = struc->startSequenceNumber - initResidueId;
        LRegion chainRegion(startIndex, numResidues);
        sd->location.append(chainRegion);
        Q_ASSERT(moleculeMap.contains(chainId));
        moleculeMap[chainId]->annotations.append(sd);
    }

}

QByteArray BioStruct3D::getRawSequenceByChainId( int id ) const
{
    QByteArray sequence("");

    Q_ASSERT(moleculeMap.contains(id));
    const SharedMolecule molecule = moleculeMap.value(id);
    foreach (const SharedResidue residue, molecule->residueMap) {
       QChar c = residue->acronym;
       sequence.append(c);
    }
    
    return sequence;
}

} //namespace
