/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_BIOSTRUCT3D_H_
#define _GB2_BIOSTRUCT3D_H_

#include <QtCore/QString>
#include <QtCore/QList>
#include <QtCore/QMap>
#include <QtCore/QSharedData>

#include <core_api/Vector3D.h>
#include <core_api/core_api.h>
#include <datatype/AnnotationData.h>

namespace GB2 {

/* Represents biopolimer residue */
class ResidueData : public QSharedData    
{ 
public:
    enum Type {
        TYPE_UNKNOWN, TYPE_PROTEIN, TYPE_DNA, TYPE_RNA
    };
    Type type;
    QByteArray name;
	QChar acronym;
	QChar chainIdentifier;
	int chainIndex;
        ResidueData() : acronym(' '), chainIndex(0) {}
};

typedef QSharedDataPointer<ResidueData> SharedResidue;


/* Represents atom */
class AtomData : public QSharedData
{
public:
    enum Type {
        SideChainAtom,
        AlphaBackboneAtom,     // C-alpha or P
        UnknownAtom            // anything that's not known to be of 
    };
    int atomicNumber;
	int chainIndex;
	int residueIndex;
	Type type;
    QByteArray name;
	Vector3D coord3d;
	double occupancy, temperature, charge;

public:
	AtomData() {
        type = UnknownAtom;
		chainIndex = 0;
		residueIndex = 0;
		atomicNumber = 0;
		temperature = -1.0;
		charge = 0;
	} ;

};

typedef QSharedDataPointer<AtomData> SharedAtom;

/* Represents chemical bond between atoms */
class Bond
{
    SharedAtom atom1;
	SharedAtom atom2;
public:
    Bond(const SharedAtom& a1, const SharedAtom& a2) : atom1(a1), atom2(a2) { }
    const SharedAtom getAtom1() { return atom1; }
    const SharedAtom getAtom2() { return atom2; }
};

/* Represents protein secondary structure: alpha-helix, beta-strand, turn */
class SecondaryStructureData : public QSharedData {
public:
	enum Type {
		TYPE_NONE = -1, TYPE_HELIX, TYPE_STRAND, TYPE_TURN
	};
	
	Type type;
    int chainIndex;
	int startSequenceNumber;
	int endSequenceNumber;
	
public:
    SecondaryStructureData() : type(TYPE_NONE)
	{	
		startSequenceNumber = 0;
		endSequenceNumber = 0;
        chainIndex = 0;
	};
};

typedef QSharedDataPointer<SecondaryStructureData> SharedSecondaryStructure;

/* Represents macromolecular contents */
class MoleculeData : public QSharedData {
public:
	MoleculeData()  {}
	QMap<int, SharedAtom> atomMap;
    QMap<int, SharedResidue> residueMap;
    QList<Bond> bonds;
   	QString name;
	bool engineered;
    QList<SharedAnnotationData> annotations;
};

typedef QSharedDataPointer<MoleculeData> SharedMolecule;

/* Biological 3D structure */
class GB2_COREAPI_EXPORT BioStruct3D {

private:
    void generateChainAnnotations();
    void generateSecStructureAnnotations();

public:
	static QString MoleculeAnnotationTag;
    static QString AlphaHelixAnnotationTag;
    static QString BetaStrandAnnotationTag;
    static QString TurnAnnotationTag;
    static QString ChainIdQualifierName;

    QMap <int, SharedMolecule> moleculeMap;
    QMap <int, SharedAtom> hetAtomMap;
    QList<SharedAnnotationData> annotations;
    QList<SharedSecondaryStructure> secondaryStructures;
    QList<Bond> interMolecularBonds;
    QString descr;
	QByteArray pdbId;
	double maxDistFromCenter;
	Vector3D rotationCenter;
	
	// Const member functions
	BioStruct3D() { }
   	Vector3D getCenter() const { return rotationCenter; }
	double getMaxDistFromCenter()  const { return maxDistFromCenter; }
	QByteArray getRawSequence() const;
    QByteArray getRawSequenceByChainId(int id) const;
    int getNumberOfAtoms() const;
    int getNumberOfResidues() const;
    const SharedAtom getAtomById(int index) const;
    const SharedResidue getResidueById(int chainIndex, int residueIndex) const;
    static const QString getSecStrucAnnotationName(SecondaryStructureData::Type type);   	
	// Modifiers
	void calcCenterAndMaxDistance();
    void generateAnnotations();

	
	
};

} //namespace



#endif // _GB2_BIOSTRUCT3D_H_
