/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "PanView.h"

#include "ADVSequenceObjectContext.h"
#include "PanViewRows.h"

#include <core_api/DNAAlphabet.h>
#include <core_api/SelectionModel.h>
#include <core_api/Log.h>
#include <core_api/AppContext.h>

#include <gobjects/DNASequenceObject.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/AnnotationSettings.h>

#include <selection/DNASequenceSelection.h>
#include <util_gui/GraphUtils.h>
#include <util_gui/GScrollBar.h>


#include <QtGui/QTextEdit>
#include <QtGui/QGridLayout>
#include <QtGui/QMenu>
#include <QtGui/QPainter>
#include <QtGui/QFontMetrics>
#include <QtGui/QDialog>


namespace GB2 {

static LogCategory log(ULOG_CAT_ADV);

PanView::PanView(QWidget* p, ADVSequenceObjectContext* ctx) : GSequenceLineViewAnnotated(p, ctx)
{
    rowBar = new QScrollBar(this);
    rowsManager = new PVRowsManager();
    renderArea = new PanViewRenderArea(this);

    visibleRange.len = seqLen;
    minNuclsPerScreen = qMin(seqLen, 10); 

    zoomInAction = new QAction(QIcon(":/core/images/zoom_in.png"), tr("zin_action"), this);
    connect(zoomInAction, SIGNAL(triggered()), SLOT(sl_zoomInAction()));

    zoomOutAction = new QAction(QIcon(":/core/images/zoom_out.png"), tr("zout_action"), this);
    connect(zoomOutAction, SIGNAL(triggered()), SLOT(sl_zoomOutAction()));

    zoomToSelectionAction= new QAction(QIcon(":/core/images/zoom_sel.png"), tr("zsel_action"), this);
    connect(zoomToSelectionAction, SIGNAL(triggered()), SLOT(sl_zoomToSelection()));

    zoomToSequenceAction = new QAction(QIcon(":/core/images/zoom_whole.png"), tr("zseq_action"), this);
    connect(zoomToSequenceAction, SIGNAL(triggered()), SLOT(sl_zoomToSequence()));

    drawSettings.drawAnnotationArrows = true;
    drawSettings.drawAnnotationNames = true;

    //can't move to the GSequenceLineViewAnnotated -> virtual calls does not work in  constructor
    foreach(AnnotationTableObject* obj, ctx->getAnnotationObjects()) {
        registerAnnotations(obj->getAnnotations());
    }

    updateActions();
    updateRowBar();

    pack();
}

void PanView::pack() {
    assert(layout() == NULL);
    QGridLayout* layout = new QGridLayout();
    layout->setMargin(0);
    layout->setSpacing(0);
    layout->addWidget(renderArea, 0, 0, 1, 1);
    layout->addWidget(rowBar, 0, 1, 2, 1);
    layout->addWidget(scrollBar, 1, 0, 1, 1);
    setLayout(layout);

    setFixedHeight(renderArea->height() + scrollBar->height());
}

PanView::~PanView() {
    delete rowsManager;
}

void PanView::registerAnnotations(const QList<Annotation*>& l) {
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    foreach(Annotation* a, l) {
        const AnnotationSettings* as = asr->getSettings(a->getAnnotationName());
        if (as->visible) {
            rowsManager->addAnnotation(a, a->getAnnotationName());
        }
    }
    updateRows();
}

void PanView::unregisterAnnotations(const QList<Annotation*>& l) {
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    foreach(Annotation* a, l) {
        const AnnotationSettings* as = asr->getSettings(a->getAnnotationName());
        if (as->visible) {
            rowsManager->removeAnnotation(a);
        }
    }
    updateRows();
}

void PanView::updateRows() {
    rowsManager->resort();

    PanViewRenderArea* ra = (PanViewRenderArea*)renderArea;
    bool heightChanged = ra->updateNumVisibleRows();
    if (heightChanged) {
        setFixedHeight(renderArea->height() + scrollBar->height());
    }

    int maxSteps = calculateNumRowBarSteps();
    if (qAbs(rowBar->maximum() - rowBar->minimum())!=maxSteps) {
        updateRowBar();
    }
}
int  PanView::calculateNumRowBarSteps() const {
    PanViewRenderArea* ra = (PanViewRenderArea*)renderArea;
    int visibleRows = ra->getNumVisibleRows();
    int numRows = rowsManager->getNumRows();
    int res = qMax(0, numRows - visibleRows);
    return res;
}

void PanView::updateRowBar() {
    rowBar->disconnect(this);

    PanViewRenderArea* ra = (PanViewRenderArea*)renderArea;
    int visibleRows = ra->getNumVisibleRows();
    int maxSteps = calculateNumRowBarSteps();
    rowBar->setMinimum(-maxSteps); //inverted appearance
    rowBar->setMaximum(0);
    rowBar->setSingleStep(1);
    rowBar->setPageStep(visibleRows - 1);
    rowBar->setSliderPosition(-ra->getRowLinesOffset());
    rowBar->setEnabled(maxSteps > 0);
    connect(rowBar, SIGNAL(valueChanged(int)), SLOT(sl_onRowBarMoved(int)));
}

void PanView::sl_onRowBarMoved(int v) {
    PanViewRenderArea* ra = (PanViewRenderArea*)renderArea;
    ra->setRowLinesOffset(-v); // '-' because of inverted appearance
    addUpdateFlags(GSLV_UF_NeedCompleteRedraw);    
    update();
}

void PanView::sl_onAnnotationSettingsChanged(const QStringList& changedSettings) {
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    foreach (const QString& name, changedSettings) {
        const AnnotationSettings* as =asr->getSettings(name);
        bool hasRow = rowsManager->contains(name);
        if (as->visible == hasRow) {
            continue;
        }
        QList<Annotation*> changed;
        foreach(AnnotationTableObject* ao, ctx->getAnnotationObjects()) {
            ao->selectAnnotationsByName(name, changed);
        }
        if (changed.isEmpty()) {
            continue;
        }
        foreach(Annotation* a, changed) {
            if (as->visible) {
                rowsManager->addAnnotation(a, a->getAnnotationName());
            } else  {
                rowsManager->removeAnnotation(a);
            }
        }
    }
    updateRows();
    GSequenceLineViewAnnotated::sl_onAnnotationSettingsChanged(changedSettings);
}



void PanView::setSelection(const LRegion& r) {
    ctx->getSequenceSelection()->clear();
    if (r.len!=0) {
        ctx->getSequenceSelection()->addRegion(r);
    }
}

void PanView::onVisibleRangeChanged(bool signal) {
    updateActions();
    GSequenceLineView::onVisibleRangeChanged(signal);
}

void PanView::updateActions() {
    zoomInAction->setEnabled(visibleRange.len > minNuclsPerScreen);
    zoomOutAction->setEnabled(visibleRange.len < seqLen);

    const QList<LRegion>& sel = ctx->getSequenceSelection()->getSelectedRegions();
    if (!sel.isEmpty() && sel.first().len >= minNuclsPerScreen) {
        zoomToSelectionAction->setEnabled(true);
    } else {
        zoomToSelectionAction->setEnabled(false);
    }
    zoomToSequenceAction->setEnabled(visibleRange.startPos != 0 || visibleRange.endPos() != seqLen);
}

void PanView::sl_zoomInAction() {
    assert(visibleRange.len >= minNuclsPerScreen);
    LRegion newVisibleRange = visibleRange;
    newVisibleRange.len = qMax((visibleRange.len + 1) / 2, minNuclsPerScreen);
    if (newVisibleRange.len!=visibleRange.len) {
        newVisibleRange.startPos = visibleRange.startPos + (visibleRange.len - newVisibleRange.len)/2;
        assert(newVisibleRange.startPos >=0 && newVisibleRange.endPos() <= seqLen); //todo: move to setVisibleRange
        setVisibleRange(newVisibleRange);
    }
}	

void PanView::sl_zoomOutAction() {
    assert(visibleRange.len <= seqLen);
    LRegion newVisibleRange = visibleRange;
    newVisibleRange.len = qMin(visibleRange.len * 2, seqLen);
    if (newVisibleRange.len != visibleRange.len) {
        newVisibleRange.startPos = qBound(0, visibleRange.startPos - (newVisibleRange.len - visibleRange.len)/2, seqLen-newVisibleRange.len);
        assert(newVisibleRange.startPos >=0 && newVisibleRange.endPos() <= seqLen); //todo: move to setVisibleRange
        setVisibleRange(newVisibleRange);
    }
}

void PanView::sl_onDNASelectionChanged(LRegionsSelection* s, const QList<LRegion>& added, const QList<LRegion>& removed) {
    GSequenceLineView::sl_onDNASelectionChanged(s, added, removed);
    updateActions();
}

void PanView::sl_zoomToSelection() {
    const QList<LRegion>& sel = ctx->getSequenceSelection()->getSelectedRegions();
    if (sel.isEmpty()) {
        return;
    }
    LRegion selRegion = sel.first();
    if (selRegion.len < minNuclsPerScreen) {
        return;
    }
    if (visibleRange==selRegion) {
        return;
    }
    assert(ctx->getSequenceObject()->getSequenceRange().contains(selRegion));
    visibleRange = selRegion;
    onVisibleRangeChanged();
}	

void PanView::sl_zoomToSequence() {
    LRegion wholeRange(0, seqLen);
    assert(visibleRange != wholeRange);
    visibleRange = wholeRange;
    onVisibleRangeChanged();
}

void PanView::setVisibleRange(const LRegion& newRange, bool signal) {
    if (newRange.len < minNuclsPerScreen) {
        //todo: log this
        return;//ignored
    }
    GSequenceLineView::setVisibleRange(newRange, signal);
}


void PanView::ensureVisible(Annotation* a, int locationIdx) {
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    const AnnotationSettings* as = asr->getSettings(a->getAnnotationName());
    if (as->visible) {
        int row = rowsManager->getAnnotationRow(a);
        PanViewRenderArea* pr = (PanViewRenderArea*)renderArea;
        if (!pr->isRowVisible(row)) {
            centerRow(row);
        }
    }
    GSequenceLineViewAnnotated::ensureVisible(a, locationIdx);
}

void PanView::centerRow(int row) {
    PanViewRenderArea* pr = (PanViewRenderArea*)renderArea;
    int targetFirstRowLine = qMax(0, row - pr->getNumVisibleRows() / 2);
    int rowOnTheFirstLine = pr->getRowLinesOffset();
    if (targetFirstRowLine == rowOnTheFirstLine) {
        return;
    }
    int dPos = targetFirstRowLine - rowOnTheFirstLine;
    int sliderPos = qBound(rowBar->minimum(), rowBar->value() -  dPos, rowBar->maximum());
    rowBar->setSliderPosition(sliderPos);
}


void PanView::sl_onRangeChangeRequest(int start, int end) {
    log.trace(tr("range_change_request_%1_%2").arg(start).arg(end));
    setVisibleRange(LRegion(start-1, end));
}

//////////////////////////////////////////////////////////////////////////
/// render
PanViewRenderArea::PanViewRenderArea(PanView* d) : GSequenceLineViewAnnotatedRenderArea(d, false), panView(d) {
    rowLinesOffset = 0;

    rulerFont.setFamily("Arial");
    rulerFont.setPointSize(8);

    lineHeight = afmNormal->height() + 2;

    updateNumVisibleRows();
}


void PanViewRenderArea::drawAll(QPaintDevice* pd) {
    GSLV_UpdateFlags uf = view->getUpdateFlags();
    bool completeRedraw = uf.testFlag(GSLV_UF_NeedCompleteRedraw) || uf.testFlag(GSLV_UF_ViewResized) || 
                          uf.testFlag(GSLV_UF_VisibleRangeChanged) || uf.testFlag(GSLV_UF_AnnotationsChanged);

    if (completeRedraw) {
        QPainter pCached(cachedView);
        pCached.fillRect(0, 0, pd->width(), pd->height(), Qt::white);
        pCached.setPen(Qt::black);

        drawRuler(pCached);

        drawAnnotations(pCached);

        pCached.end();
    }
    QPainter p(pd);
    p.drawPixmap(0, 0, *cachedView);

    drawFrame(p); //frame is always redrawn

    drawAnnotationsSelection(p);
    drawSequenceSelection(p);

    if (view->hasFocus()) {
        drawFocus(p);
    }
}
#define RULER_NOTCH_SIZE 2
void PanViewRenderArea::drawRuler(QPainter& p) {
    const LRegion& visibleRange = view->getVisibleRange();
    float halfChar = getCurrentScale() / 2;
    int firstCharCenter = qRound(posToCoordF(visibleRange.startPos) + halfChar);
    int lastCharCenter = qRound(posToCoordF(visibleRange.endPos()-1) + halfChar);
    int firstLastWidth = lastCharCenter - firstCharCenter;
    if (qRound(halfChar) == 0) {
        int w = width();
        assert(firstLastWidth == w); Q_UNUSED(w);
        firstLastWidth--; // make the end of the ruler visible
    }
    GraphUtils::RulerConfig c;
    c.notchSize = RULER_NOTCH_SIZE;
    int y = getLineY(getRulerLine()) + c.notchSize;
    GraphUtils::drawRuler(p, QPoint(firstCharCenter, y), firstLastWidth, visibleRange.startPos+1, visibleRange.endPos(), rulerFont, c);	
}


LRegion PanViewRenderArea::getAnnotationYRange(Annotation* a, const LRegion& r, const AnnotationSettings* as) const {
    Q_UNUSED(r);
    if (!as->visible) {
        return LRegion(-1, 0);
    }
    int row = getPanView()->getRowsManager()->getAnnotationRow(a);
    int line = getRowLine(row);
    return LRegion(getLineY(line) + 2, lineHeight - 4);
}

void PanViewRenderArea::drawAnnotations(QPainter& p) {
    QPen dotty(Qt::lightGray, 1, Qt::DotLine);
    p.setPen(dotty);
    p.setFont(*afSmall);

    //draw row names
    PVRowsManager* rm = getPanView()->getRowsManager();
    int maxVisibleRows = getNumVisibleRows();
    int dy = (lineHeight - afSmallCharHeight) / 2;
    for (int i = 0; i < maxVisibleRows; i++) {
        int row = i + rowLinesOffset;
        int rowLine = getRowLine(row);
        int lineY = getLineY(rowLine);
        p.drawLine(0, lineY, cachedView->width(), lineY);

        bool emptyRow = row >= rm->getNumRows();
        int numAnnotations = emptyRow ? 0 : rm->getNumAnnotationsInRow(row);
        QString rowName = emptyRow ? PanView::tr("empty") : rm->getRowKey(row);
        QString text = rowName;
        if (!emptyRow) {
            text+=" (" + QString::number(numAnnotations) + ")";
        }
        int y = lineY + lineHeight - dy;
        p.drawText(10, y, text);
    }

    int firstRowLine = getRowLine(0);
    int lineY = getLineY(firstRowLine) + lineHeight;
    p.drawLine(0, lineY, cachedView->width(), lineY);


    GSequenceLineViewAnnotatedRenderArea::drawAnnotations(p);
}

#define ARROW_DY 5
#define ARROW_DX 5
void PanViewRenderArea::drawSequenceSelection(QPainter& p) {
    const QList<LRegion>& selection = panView->getSequenceContext()->getSequenceSelection()->getSelectedRegions();
    bool showSequenceMode = false;
    if (selection.isEmpty()) {
        return;
    }
    const LRegion& visibleRange = view->getVisibleRange();
    QPen pen1(Qt::darkGray, 1, Qt::SolidLine);
    QPen pen2(QColor("#007DE3"), 2, Qt::SolidLine);
    p.setFont(rulerFont);
    QFontMetrics rfm(rulerFont);

    int lineY = getLineY(getSelectionLine());
    int ly = lineY + lineHeight/2; //ruler line
    
    bool drawRect = showSequenceMode;
    bool drawGraphics = true;
    if (showSequenceMode) {
        ly = lineY - lineHeight + RULER_NOTCH_SIZE; 
        drawGraphics = ly > 0;
    }
    int halfNum = rfm.boundingRect('1').height() / 2;
    int rty = ly + halfNum + 1;
    QString rangePattern = " "+tr("[%1 bp]")+" ";
    foreach(const LRegion& r, selection) {
        if (!visibleRange.intersects(r)) {
            continue;
        }
        int x1 = qMax(0, posToCoord(r.startPos, true));
        int x2 = qMin(cachedView->width(), posToCoord(r.endPos(), true));

        p.setPen(pen1);
        if (visibleRange.contains(r.startPos)) {
            p.drawLine(x1, 0, x1, ly);                    
        }
        if (visibleRange.contains(r.endPos()-1)) {
            p.drawLine(x2, 0, x2, ly);                    
        }

        if (drawRect) {
            p.setPen(Qt::black);
            p.drawRect(x1, lineY+1, x2 - x1, lineHeight-2);
        }
        
        if (drawGraphics) {
            //draw line
            p.setPen(pen2);
            p.drawLine(x1, ly, x2, ly);
            int dArrow = 2 * ARROW_DX;

            QString t1 = QString::number(r.startPos+1);
            QString t2 = QString::number(r.endPos());
            int tOffs = ARROW_DX-1;
            QRect t1Rect = rfm.boundingRect(t1); t1Rect.translate(x1 - t1Rect.width() - tOffs, rty);
            QRect t2Rect = rfm.boundingRect(t2).translated(x2 + tOffs, rty);

            // define range text coords
            QString rangeText = rangePattern.arg(r.len);
            QRect rtRect = rfm.boundingRect(rangeText);
            int rulerWidth = x2 - x1;
            bool rangeTextInTheMiddle = rulerWidth - dArrow > rtRect.width();
            if (rangeTextInTheMiddle) {
                int rtx = x1 + (rulerWidth - rtRect.width())/2 + 1;
                assert(rtx - x1 >= ARROW_DX);
                rtRect.translate(rtx, rty);
                p.fillRect(rtRect, Qt::white);
                p.drawText(rtRect, Qt::AlignCenter, rangeText);
            }  else if (!rangeTextInTheMiddle) { //if range text is not in the middle glue it to one of the boundary texts
                QString newT2 = t2 + rangeText;
                QRect newT2Rect = rfm.boundingRect(newT2).translated(x2 + tOffs, rty);
                if (newT2Rect.right() < width()) {
                    t2Rect = newT2Rect;
                    t2 = newT2;
                } else {
                    QString newT1 = rangeText + t1;
                    QRect newT1Rect = rfm.boundingRect(newT1); newT1Rect.translate(x1 - newT1Rect.width() - tOffs, rty);
                    if (newT1Rect.left() >=0) {
                        t1 = newT1;
                        t1Rect = newT1Rect;
                    }
                }
            }
            
            //check if regions overlap
            int interWidth = t2Rect.left() - t1Rect.right();
            if (interWidth < dArrow) {
                int deltaW = interWidth > 0 ? dArrow : qAbs(interWidth) + dArrow;
                if (t1Rect.x() - deltaW > 0) {
                    t1Rect.translate(-deltaW, 0);
                } else if (t2Rect.right() + deltaW < width()) {
                    t2Rect.translate(deltaW, 0);
                }
            }

            //draw regions
            p.fillRect(t1Rect, Qt::white);
            p.fillRect(t2Rect, Qt::white);
            p.drawText(t1Rect, Qt::AlignCenter, t1);
            p.drawText(t2Rect, Qt::AlignCenter, t2);

            //draw arrows (after the text -> can overlap with text rect boundaries)
            if (visibleRange.contains(r.startPos)) {
                p.drawLine(x1, ly, x1 + ARROW_DX, ly + ARROW_DY);
                p.drawLine(x1, ly, x1 + ARROW_DX, ly - ARROW_DY);
            }
            if (visibleRange.contains(r.endPos()-1)) {
                p.drawLine(x2, ly, x2 - ARROW_DX, ly + ARROW_DY);
                p.drawLine(x2, ly, x2 - ARROW_DX, ly - ARROW_DY);
            }
        }
    }
}



int PanViewRenderArea::getRowLine(int i) const {
    int line = getFirstRowLine() - i + rowLinesOffset;
    if (line < 0 || line > getFirstRowLine()) {
        return -1;
    }
    return line;
}

void PanViewRenderArea::setRowLinesOffset(int r) {
    int maxRows = getPanView()->getRowsManager()->getNumRows();
    int visibleRows = getNumVisibleRows();
    assert(r<=maxRows-visibleRows); Q_UNUSED(maxRows); Q_UNUSED(visibleRows);
    if (r!=rowLinesOffset) {
        rowLinesOffset = r;
        update();
    }
}

#define MIN_VISIBLE_ROWS  1
#define EXTRA_EMPTY_ROWS  0
#define MAX_VISIBLE_ROWS 10

bool PanViewRenderArea::updateNumVisibleRows() {
    int actualRows = getPanView()->getRowsManager()->getNumRows();
    int expectedRows = actualRows + EXTRA_EMPTY_ROWS;
    int actualRowsToShow = qBound(MIN_VISIBLE_ROWS, expectedRows, MAX_VISIBLE_ROWS);
    int currentRowsShown = getNumVisibleRows();
    if (actualRowsToShow == currentRowsShown) {
        return false;
    }
    numLines = 2 + actualRowsToShow; //2 == ruler + selection
    setFixedHeight(numLines * lineHeight);
    view->addUpdateFlags(GSLV_UF_ViewResized);
    view->update();
    return true;
}

}//namespace

