/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WorkflowUtils.h"

#include <workflow/Descriptor.h>
#include <core_api/GObject.h>
#include <core_api/DocumentModel.h>
#include <document_format/DocumentFormatUtils.h>
#include <util_gui/DialogUtils.h>

#include <QtCore/QDir>
#include <QtCore/QFileInfo>
#include <QtCore/QUrl>

namespace GB2 {

QString DesignerUtils::getRichDoc(const Descriptor& d) {
    return QString("<b>%1</b> : %2").arg(d.getDisplayName()).arg(d.getDocumentation());
}

const QString DesignerUtils::SCHEMA_FILE_EXT("uws");
QString DesignerUtils::getSchemaFileFilter(bool any) {
    return DialogUtils::prepareFileFilter(tr("UGENE workflow schemes"), QStringList(SCHEMA_FILE_EXT), any);
}

QString DesignerUtils::getDropUrl(QList<DocumentFormat*>& fs, const QMimeData* md) {
    QString url;
    const GObjectMimeData* gomd = qobject_cast<const GObjectMimeData*>(md);
    const DocumentMimeData* domd = qobject_cast<const DocumentMimeData*>(md);
    if (gomd) {
        GObject* obj = gomd->objPtr.data();
        if (obj) {
            fs << obj->getDocument()->getDocumentFormat();
            url = obj->getDocument()->getURL();
        }
    } else if (domd) {
        Document* doc = domd->objPtr.data();
        if (doc) {
            fs << doc->getDocumentFormat();
            url = doc->getURL();
        }
    } else if (md->hasUrls()) {
        QList<QUrl> urls = md->urls();
        if (urls.size() == 1)
        {
            url = urls.at(0).toLocalFile();
            fs += DocumentFormatUtils::detectFormat(url);
        }
    }
    return url;
}

void DesignerUtils::setQObjectProperties(QObject &o , const QVariantMap & params) {
    QMapIterator<QString, QVariant> i(params);
    while (i.hasNext()) {
        i.next();
        //log.debug("set param " + i.key() + "="+i.value().toString());
        o.setProperty(i.key().toAscii(), i.value());
    }
}

QStringList DesignerUtils::expandToUrls(const QString& s) {
    QStringList urls = s.split(";");
    QStringList result;
    QRegExp wcard("[*?\\[\\]]");
    foreach(QString url, urls) 
    {
        int idx = url.indexOf(wcard);
        if (idx >= 0) {
            int dirIdx = url.lastIndexOf('/', idx);
            QDir dir;
            if (dirIdx >= 0) {
                dir = QDir(url.left(dirIdx));
                url = url.right(url.length() - dirIdx - 1);
            }

            foreach(QFileInfo fi, dir.entryInfoList((QStringList() << url), QDir::Files|QDir::NoSymLinks)) {
                result << fi.absoluteFilePath();
            }
        } else {
            //if (QFile::exists(url)) 
            {
                result << url;
            }
        }
    }
    return result;
}

QVariant PrompterBaseImpl::getParameter(const QString& id) {
    if (map.contains(id)) {
        return map.value(id);
    } else {
        return target->getParameter(id)->value;
    }
}

QString PrompterBaseImpl::getURL(const QString& id) {
    QString url = getParameter(id).toString();
    if (url.isEmpty()) {
        url = "<font color='red'>"+tr("unset")+"</font>";
    } else if (url.indexOf(";") != -1) {
        url = tr("the list of files");
    } else {
        url = QFileInfo(url).fileName();
    }
    return url;
}

QString PrompterBaseImpl::getRequiredParam(const QString& id) {
    QString url = getParameter(id).toString();
    if (url.isEmpty()) {
        url = "<font color='red'>"+tr("unset")+"</font>";
    }
    return url;
}

QString PrompterBaseImpl::getScreenedURL(BusPort* input, const QString& id, const QString& slot) {
    Actor* origin = input->getProducer(slot);
    QString url; 
    if (origin) {
        url = tr("file(s) alongside of input sources of <u>%1</u>").arg(origin->getLabel());
    } else {
        url = QString("<u>%1</u>").arg(getURL(id));
    }
    return url;
}

QString PrompterBaseImpl::getProducers( const QString& port, const QString& slot )
{
    BusPort* input = qobject_cast<BusPort*>(target->getPort(port));
    QList<Actor*> producers = input->getProducers(slot);

    QStringList labels;
    foreach(Actor* a, producers) {
        labels << a->getLabel();
    }
    return labels.join(", ");
}
}//namespace
