/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <datatype/BioStruct3D.h>
#include <QtOpenGL>

#include "GraphicUtils.h"
#include "BioStruct3DColorScheme.h"
#include "WormsGLRenderer.h"


namespace GB2 { 

const QString WormsGLRenderer::ID(QObject::tr("Worms"));


WormsGLRenderer::WormsGLRenderer( const BioStruct3D& struc, const BioStruct3DColorScheme* s ) : BioStruct3DGLRenderer(struc,s)
{
    const char* alphaCarbonTag = "CA";
    const char* phosophorTag = "P";

    //const char* carbonylOxygenTag = "O";

    foreach (const SharedMolecule mol, struc.moleculeMap) {
        foreach (const SharedAtom atom, mol->atomMap) {
            if ((atom->name.trimmed() == alphaCarbonTag) || (atom->name.trimmed() == phosophorTag)) {
                bioPolymerMap[atom->chainIndex].monomerMap[atom->residueIndex].alphaCarbon = atom;
            }
            //TODO: Record carbonyl oxygen atom, watch out for rewriting data
			//if (atom->name.trimmed() == carbonylOxygenTag) {
            //    bioPolymerMap[atom->chainIndex].monomerMap[atom->residueIndex].carbonylOxygen = atom;
            //}
        }
    }
  
   createObjects3D();
   createWormCoordsData();    

}


void WormsGLRenderer::drawWorms(  )
{
    GLUquadricObj *pObj;    // Quadric Object

    // Draw atoms as spheres
    pObj = gluNewQuadric();
    gluQuadricNormals(pObj, GLU_SMOOTH);
    static float ribbonThickness = 0.3f;
    static float tension = 0.01f;

    foreach (int chainId, coordsDataMap.keys()) {
        const QVector<CoordData> wormCoords = coordsDataMap.value(chainId);
        int size = wormCoords.size();
        for (int i = 0; i + 3 < size; ++i) {
            glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, wormCoords[i+1].color.getConstData());
            Vector3D siteV = wormCoords[i+3].site;
            glDrawHalfWorm(&wormCoords[i].site, wormCoords[i+1].site, wormCoords[i+2].site, &siteV, ribbonThickness, false, false, tension);
            glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, wormCoords[i+2].color.getConstData());
            glDrawHalfWorm(&wormCoords[i+3].site, wormCoords[i+2].site, wormCoords[i+1].site, &wormCoords[i].site, ribbonThickness, false, false, tension);
        }

        if (wormCoords.size() >= 3) {
            glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, wormCoords[0].color.getConstData());
            Vector3D atomCoordFirst = fakeAtomCoords.value(chainId).first;
            Vector3D siteV2 = wormCoords[2].site;
            glDrawHalfWorm(&atomCoordFirst, wormCoords[0].site, wormCoords[1].site, &siteV2, ribbonThickness, true, false, tension);
            glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, wormCoords[1].color.getConstData());
            glDrawHalfWorm(&siteV2, wormCoords[1].site, wormCoords[0].site, &atomCoordFirst, ribbonThickness, false, false, tension);
     
            glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, wormCoords[size-1].color.getConstData());
            Vector3D atomCoordSecond = fakeAtomCoords.value(chainId).second;
            Vector3D siteV3 = wormCoords[size-3].site;
            glDrawHalfWorm(&atomCoordSecond, wormCoords[size-1].site, wormCoords[size-2].site, &siteV3, ribbonThickness, true, false, tension);
            glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, wormCoords[size - 1].color.getConstData());
            glDrawHalfWorm(&wormCoords[size - 3].site, wormCoords[size - 2].site, wormCoords[size - 1].site, &atomCoordSecond, ribbonThickness, false, false, tension);
        }
   }


}


void WormsGLRenderer::drawBioStruct3D()
{
	drawWorms();
    drawSecondaryStructure();

}

void WormsGLRenderer::drawSecondaryStructure(  )
{
    foreach( Object3D* obj, objects) {
        obj->draw();
    }  


}

void WormsGLRenderer::updateColorScheme( const BioStruct3DColorScheme* s )
{
    colorScheme = s;
    
    qDeleteAll(objects);
    objects.clear();
    
    //TODO: optimization -> don't have to create everything again
    createObjects3D();
    
    //Recalculate colors for worms
    QMap<int, BioPolymer>::const_iterator i;
    for (i = bioPolymerMap.constBegin(); i != bioPolymerMap.constEnd(); ++i) {
        int chainID = i.key();
        int j = 0;
        foreach (Monomer monomer, (*i).monomerMap) {
            const SharedAtom& atom = monomer.alphaCarbon;
            coordsDataMap[chainID][j++].color = colorScheme->getAtomColor(atom);
        }
    }
}

void WormsGLRenderer::createObjects3D()
{
    foreach (const SharedSecondaryStructure ss, bioStruct.secondaryStructures) {
        int startId = ss->startSequenceNumber;
        int endId = ss->endSequenceNumber;
        int chainId = ss->chainIndex;
        if (bioPolymerMap.contains(chainId)) {
            const BioPolymer& polymer = bioPolymerMap.value(chainId);
            if (polymer.monomerMap.contains(startId) && polymer.monomerMap.contains(endId)) {
                if (ss->type == SecondaryStructureData::TYPE_HELIX ) {
                    float radius = 1.5f;
                    QVector<Vector3D> helixPoints;
                    Color4f color(0,0,0);
                    for (int i = startId; i <= endId; ++i) {
                        if (!polymer.monomerMap.contains(i))
                            continue;
                        helixPoints.append(polymer.monomerMap.value(i).alphaCarbon->coord3d);
                        color += colorScheme->getAtomColor(polymer.monomerMap.value(i).alphaCarbon);
                    }
                    QPair<Vector3D, Vector3D> axis = calcBestAxisThroughPoints(helixPoints);
                    color /= (endId - startId + 1); 
                    objects.append(new Helix3D(color, axis.first, axis.second, radius) );
                } else if (ss->type == SecondaryStructureData::TYPE_STRAND) {
                    QVector<Vector3D> arrowPoints;
                    Color4f color(0,0,0);
                    for (int i = startId; i <= endId; ++i) {
                       if (!polymer.monomerMap.contains(i))
                            continue;
                       arrowPoints.append(polymer.monomerMap.value(i).alphaCarbon->coord3d);
                       color += colorScheme->getAtomColor(polymer.monomerMap.value(i).alphaCarbon);
                    }
                    color /= (endId - startId + 1); 
                    QPair<Vector3D, Vector3D> axis = calcBestAxisThroughPoints(arrowPoints);
                    objects.append(new Strand3D(color, axis.first, axis.second) );
                }
            }
        }
    }
}

void WormsGLRenderer::createWormCoordsData()
{

    QMap<int, BioPolymer>::const_iterator i;
    for (i = bioPolymerMap.constBegin(); i != bioPolymerMap.constEnd(); ++i) {
        int chainID = i.key();
        Vector3D r1, r2,a,b;
        Vector3D fakeFirst, fakeLast;
		const QMap<int,Monomer> monomers = (*i).monomerMap;
		QMap<int,Monomer>::const_iterator iter(monomers.constBegin());
        //TODO: some check for monomerMap size required
        //Q_ASSERT((*i).monomerMap.size() >= 2);
		Monomer m = iter.value();
		r1 = iter.value().alphaCarbon.constData()->coord3d;
        r2 = (++iter).value().alphaCarbon.constData()->coord3d;
        a = r1;
        b = (r2 - r1) / 100.f;
        fakeFirst = a + b*(-10.f);
        
        iter = (*i).monomerMap.constEnd();
        r1 = (--iter).value().alphaCarbon.constData()->coord3d;
        r2 = (--iter).value().alphaCarbon.constData()->coord3d;
        a = r1;
        b = (r2 - r1) / 100.f;
        fakeLast = a + b*(-10.f);
        
        fakeAtomCoords.insert(chainID, QPair<Vector3D,Vector3D> (fakeFirst, fakeLast));

        foreach (const Monomer monomer, (*i).monomerMap) {
            const SharedAtom& atom = monomer.alphaCarbon;
            coordsDataMap[chainID].append(CoordData(atom->coord3d, colorScheme->getAtomColor(atom), atom->residueIndex));
        }
        

    }

}




WormsGLRenderer::~WormsGLRenderer()
{
   qDeleteAll(objects);
}

} //namespace
