/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DeviationGraph.h"

#include <core_api/DNAAlphabet.h>
#include <gobjects/DNASequenceObject.h>
#include "DNAGraphPackPlugin.h"
#include <util_algorithm/RollingArray.h>


/* TRANSLATOR GB2::DeviationGraphFactory */

namespace GB2 {
static QString nameByType(DeviationGraphFactory::GDeviationType t) {
	if (t == DeviationGraphFactory::AT) {
		return DeviationGraphFactory::tr("at_deviation_graph");
	}
	return DeviationGraphFactory::tr("gc_deviation_graph");

}

DeviationGraphFactory::DeviationGraphFactory(GDeviationType t, QObject* p)
: GSequenceGraphFactory(nameByType(t), p)
{
	if ( t == DeviationGraphFactory::AT) {
		devPair.first = 'A'; devPair.second = 'T';
	} else {
		devPair.first = 'G'; devPair.second = 'C';
	}
}

bool DeviationGraphFactory::isEnabled(DNASequenceObject* o) const {
	DNAAlphabet* al = o->getAlphabet();
	return al->isNucleic();
}

QList<GSequenceGraphData*> DeviationGraphFactory::createGraphs(GSequenceGraphView* v) {
	QList<GSequenceGraphData*> res;
	assert(isEnabled(v->getSequenceObject()));
	GSequenceGraphData* d = new GSequenceGraphData(getGraphName());
	d->ga = new DeviationGraphAlgorithm(devPair);
	res.append(d);
	return res;
}

GSequenceGraphDrawer* DeviationGraphFactory::getDrawer(GSequenceGraphView* v) {
    GSequenceGraphWindowData wd(10, 30);
    return new GSequenceGraphDrawer(v, wd);
}


//////////////////////////////////////////////////////////////////////////
// DeviationGraphAlgorithm

DeviationGraphAlgorithm::DeviationGraphAlgorithm(const QPair<char, char>& _p)  :  p(_p)
{
}
void DeviationGraphAlgorithm::windowStrategyWithoutMemorize(QVector<float>& res, const QByteArray& seqArr, int startPos, const GSequenceGraphWindowData* d, int nSteps)
{
    assert(startPos>=0);
    const char* seq = seqArr.constData();
    
    for (int i = 0; i < nSteps; i++)	{
		int start = startPos + i * d->step;
		int end = start + d->window;
        assert(end <= seqArr.size());
		int first = 0;
		int second = 0;
		for (int x = start; x < end; x++) {
			char c = seq[x];
			if (c == p.first) {
				first++; continue;
			}
			if (c == p.second) {
				second++;
			}
		}
		res.append((first - second) / qMax(0.001f, (float)(first + second)));
	}
}
QPair<int, int> DeviationGraphAlgorithm::matchOnStep(const QByteArray& seqArr, int begin, int end)
{
    const char* seq = seqArr.constData();
    assert(begin >=0 && end <= seqArr.size());

	QPair<int, int> res(0, 0);
	for (int j = begin; j < end; ++j)	{
		char c = seq[j];
		if (c == p.first) {
			res.first++; 
            continue;
		}
		if (c == p.second) {
			res.second++;
		}
	}
	return res;
}
void DeviationGraphAlgorithm::sequenceStrategyWithMemorize(QVector<float>& res, const QByteArray& seq, const LRegion& vr, const GSequenceGraphWindowData* d)
{
	int rsize = d->window / d->step;
	RollingArray<int> raF(rsize);
	RollingArray<int> raS(rsize);
	int endPos = vr.endPos();
	int globalCountF = 0;
	int globalCountS = 0;
	int nextI = 0;
	int firstValue = vr.startPos + d->window - d->step;
	for (int i = vr.startPos; i < endPos; i = nextI)	{
		nextI = i + d->step;
		QPair<int, int> result = matchOnStep(seq, i, nextI);
		globalCountF += result.first;
		globalCountS += result.second;
		raF.push_back_pop_front(result.first);
		raS.push_back_pop_front(result.second);
		if (i >= firstValue)	{
			int vF = raF.get(0);
			int vS = raS.get(0);
			res.append((globalCountF - globalCountS) / qMax(0.001f, (float)(globalCountF + globalCountS)));
			globalCountF -= vF;
			globalCountS -= vS;
		}
	}
}

void DeviationGraphAlgorithm::calculate(QVector<float>& res, DNASequenceObject* o, const LRegion& vr, const GSequenceGraphWindowData* d) {
	assert(d!=NULL);
	int nSteps = GSequenceGraphUtils::getNumSteps(vr, d->window, d->step);
	res.reserve(nSteps);

	const QByteArray& seq = o->getSequence();
	int startPos = vr.startPos;
    if (d->window % d->step != 0) {
		windowStrategyWithoutMemorize(res, seq, startPos, d, nSteps);
    } else {
		sequenceStrategyWithMemorize(res, seq, vr, d);	
    }
}

} // namespace

