/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "RepeatFinderSArrayWK.h"

namespace GB2 {

RepeatFinderSArrayWK::RepeatFinderSArrayWK(const char* seqX, quint32 sizeX, const char* seqY, quint32 sizeY, DNAAlphabet* al,  quint32 w, quint32 k, quint32 threads) 
:RepeatFinder(seqX, sizeX, seqY, sizeY, al, w, k) , 
blockingMode(false),
index(NULL), 
ACTIVE_DIAGS(0),
q(w/(k+1)),
nRunning(0),
active(false),
nThreads(threads),
diagOffsets(NULL)
{
	if (sizeX > sizeY) {
		arrayIsX = false;
		SEARCH_SIZE = sizeX;
		ARRAY_SIZE = sizeY;
		searchSeq = seqX;
		arraySeq = seqY;
	} else {
		arrayIsX = true;
		SEARCH_SIZE = sizeY;
		ARRAY_SIZE = sizeX;
		searchSeq = seqY;
		arraySeq = seqX;
	
	}
}

RepeatFinderSArrayWK::~RepeatFinderSArrayWK() {
	active = FALSE;
	waitThreadsStop();
	if (index) {
		delete index;
	}
	if (diagOffsets) {
		delete diagOffsets;
	}
	qDeleteAll(workers);
}


RepeatResult* RepeatFinderSArrayWK::getResult(quint32 num) {
//	resultsMutex.lock();
	tmpEdge = results.at(num);
//	resultsMutex.unlock();
	return &tmpEdge;
}


quint32 RepeatFinderSArrayWK::getNumResults() {
	return results.size();
}

void RepeatFinderSArrayWK::startCalculation() {
	active = TRUE;
	quint32 sSize = SEARCH_SIZE;
	if (reflective) {
		tmpEdge.x = 1;
		tmpEdge.y = 1;
		tmpEdge.l = sSize;
		results.append(tmpEdge);
	}
	
	diagOffsets = new RollingArray(ARRAY_SIZE);
	reportCounter = 0;
	percentDone = 0;

	if(blockingMode) {
		WorkerThread* t = new WorkerThread(this, 0, sSize, 0);
		workers.append(t);
		t->run();
	} else {
		indexMutex.lock();
		quint32 n = nRunning = 1;//qMin(nThreads, qMax(1, sSize / (20*1000)));
		quint32 start = 0;
		quint32 len = sSize / n;
		for (quint32 i = 0; i < n; i++) {
			WorkerThread* t = new WorkerThread(this, start?start-WINDOW_SIZE+1:0, i<n-1?start+len: sSize, i);
			t->start(QThread::LowPriority);
			workers.append(t);
			start+=len;
		}
		nRunning = workers.count();
		indexMutex.unlock();
	}
}


void RepeatFinderSArrayWK::terminateCalculation() {
	active = FALSE;
//	printf("TERMINATE:calc finished 1\n");
	waitThreadsStop();
//	printf("TERMINATE:calc finished 2\n");

}

void RepeatFinderSArrayWK::onFinished(WorkerThread* t) {	
	indexMutex.lock();
	nRunning--;
	if (!blockingMode && nRunning == 0 ) {
		//join boundary
		for (int j=0; j < bresultsA.size(); j++) {
			quint32 d = bresultsD[j];
			quint32 a = bresultsA[j];
			quint32 l = bresultsL[j];
			bool merged = FALSE;
			for (int i=j+1; i < bresultsA.size(); i++) {
				if (d != bresultsD[i]) {
					continue;
				}
				quint32 a1 = bresultsA[i];
				quint32 l1 = bresultsL[i];
				if ( a > a1 && a <= a1+l1) {
					bresultsL[i]=(a+l)-a1;	
					merged = TRUE;
					break;
				} else if (a1 > a && a1 <= a+l) {
					quint32 dl=(a1-a);
					bresultsA[i]=a;
					bresultsS[i]-=dl;
					bresultsL[i]+=dl;
					merged = TRUE;
       					break;
				}
			}
			if (!merged) {
				addToResults2(a, bresultsS[j], l);
			}
		}
		if (active && resultsListener!=NULL) {
			resultsListener->percentDone(100);
			resultsListener->calculationFinished();
		}
		if (index!=NULL) {
			delete index;
			index = NULL;
		}
		active = FALSE;
	}
	indexMutex.unlock();
}

void RepeatFinderSArrayWK::waitThreadsStop(){
	if (blockingMode) {
		return;
	}
	//printf("wait thread stop 1\n");
	foreach (WorkerThread* t, workers) {
		t->wait();
	}
//	printf("wait thread stop 2\n");
}


void RepeatFinderSArrayWK::calculate(WorkerThread* t) {
	indexMutex.lock();
	if(index == NULL) {
		//printf("creating index..\n");
		index = new SArrayIndex(arraySeq, ARRAY_SIZE, q, active, unknownChar);	
	}
	indexMutex.unlock();
//	printf("index created\n");

	SArrayIndex* _index =  index;

	qint32 W = WINDOW_SIZE;
	qint32 W_Q = W-q;
	qint32 KMAX = K;

	const char* dataA = arraySeq;
	const char* dataAEnd = dataA + ARRAY_SIZE;
	const char* dataAWEnd = dataAEnd-W;
	const char* dataS = searchSeq + t->sStart;
	const char* dataSEnd = searchSeq + t->sEnd;
	const char* dataSWEnd = dataSEnd-W;
	bool reflect = reflective;//local
	char unknown = unknownChar;
	int nMatches = 0;
	int percentLen = (t->sEnd - t->sStart) / 100;
	const char* nextReportPos = dataS + percentLen;
	RollingArray* diags = diagOffsets;
	quint32 ARRAY_SIZE_1 = ARRAY_SIZE-1;
	for (const char* runner = dataS; runner < dataSEnd && active; runner++) {
		diagOffsets->roll();
		if (runner==nextReportPos) {
			report();
			nextReportPos+=percentLen;
		}
		if (!_index->find(t, runner)) {
			continue;
		}
		quint32 s = runner - dataS;
		quint32 a;
		while ((a = _index->nextArrSeqPos(t))!=0xFFFFFFFF) {
			nMatches++;
			if (reflect && s >= a) {
				continue;
			}

			const qint32 rollDiag = ARRAY_SIZE_1-a;
			quint32 lastCheckedS = diags->get(rollDiag);
			if (lastCheckedS > s ) {
				continue;
			}

			//1. building initial window (back)
			int tk=0;
			quint32 sBack = runner - dataS - lastCheckedS;
			int back = qMin(sBack, a);
			const char* runnerA = dataA+a-1;
			const char* runnerS = runner-1;
			const char* lastCheckedDataS= runner-back;
			for (; runnerS>=lastCheckedDataS; runnerS--, runnerA--) {
				if (*runnerA != *runnerS || *runnerA == unknown) {
					if (tk < KMAX) {
						tk++;
						continue;
					}
					break;
				}
			}
			const char* startS = runnerS+1;
			const char* startA = runnerA+1;
			int tw = runner-startS;
			if (tw > W_Q) {
				//DBG(printf("Errrr0\n"));
			}
			if (tw == W_Q) { 
				runnerA+=W-1;
				runnerS+=W-1;
			} else {//finishing window build
				if (startS > dataSWEnd || startA > dataAWEnd) {
					continue;
				}
				runnerA = startA+tw+q;
				runnerS = runner+q;
				int tk2=0;
				for (const char* endS = startS+W; runnerS < endS; runnerS++, runnerA++) {
					if (*runnerA != *runnerS || *runnerA == unknown) {
						if (tk2 < KMAX) {
							tk2++;	
							continue;
						}
						break;
					}
				}
				if (runnerS-startS < W) {// to many mismatches on forward strand
					runnerS=runner+q;
					runnerA=startA+tw+q;
                    for ( ;*runnerS==*runnerA && *runnerS!=unknown; runnerS++, runnerA++){}
					diags->set(rollDiag, runnerS-dataS-1);
					continue; 
				}
				tk+=tk2;				
				runnerA--; //making end inclusive
				runnerS--;
			}
			
		//	if (runnerS-startS != W-1) {
		//		printf("Errrr1\n");
		//	}

			//2. going forward, checking window start
			const char* globalWStartS=0;
			const char* globalWEndS=runner+q;
			while(startS<=globalWEndS && runnerS<dataSEnd && runnerA<dataAEnd) {
				if (tk <= KMAX) {
					if (globalWStartS==0) {
						globalWStartS = startS;
					}
					globalWEndS = runnerS+1;
				} 
				bool looseMatch = *startA == *startS && *startA != unknown;
				runnerA++; runnerS++; startA++; startS++;
				bool gainMatch = *runnerA == *runnerS && *runnerA != unknown;
				tk+=(looseMatch-gainMatch);
				if (tk < 0) {
					//DBG(printf("Errrr2\n"));
				}
			}
			if (globalWStartS) {
				int resultS = globalWStartS-dataS; 
				int resultA = a+(resultS-s);
				int len = globalWEndS-globalWStartS;
				addToResults(resultA, resultS, len, t);
			}
			diags->set(rollDiag, globalWEndS-dataS);
		}

	}
//	printf("Calc Done:matches %d\n", nMatches);
}
void RepeatFinderSArrayWK::addToResults(quint32 a, quint32 s, quint32 l, RepeatFinderSArrayWK::WorkerThread* t) {
	bool boundary = nThreads > 1 && (s == 0 || s + l == t->sEnd - t->sStart); 
	a++;
	s+=t->sStart+1;
	if (boundary) {
		boundaryMutex.lock();
		bresultsA.append(a);
		bresultsS.append(s);
		bresultsL.append(l);
		bresultsD.append(a-s);
		boundaryMutex.unlock();
	} else {
		addToResults2(a, s, l);
	}
}

void RepeatFinderSArrayWK::addToResults2(quint32 a, quint32 s, quint32 l) {
	resultsMutex.lock();
	RepeatResult e;
	e.l = l;
	if (!reflective) {
		if (arrayIsX) {
			e.x = a;
			e.y = s;
		} else {
			e.y = a;
			e.x = s;
		}
		results.append(e);
	} else {
		e.x = a;
		e.y = s;
		results.append(e);
		e.x = s;
		e.y = a;
		results.append(e);
	}
	resultsMutex.unlock();
}


RepeatFinderSArrayWK::WorkerThread::WorkerThread(RepeatFinderSArrayWK* _owner, quint32 _sStart, quint32 _sEnd, quint32 _tid)
 : owner(_owner), sStart(_sStart), sEnd(_sEnd), tid(_tid) 
{
}

void RepeatFinderSArrayWK::WorkerThread::run() {
	owner->calculate(this);
	owner->onFinished(this);
}

	
void RepeatFinderSArrayWK::report() {
	if (resultsListener!=NULL) {
		indexMutex.lock();
		reportCounter++;
		if (reportCounter == nThreads) {
			reportCounter=0;
			percentDone++;
			//qDebug("percent done %d\n", percentDone);
			resultsListener->percentDone(percentDone);
		}
		indexMutex.unlock();
	}
}

RepeatFinderSArrayWK::RollingArray::RollingArray(int _size) : size(_size){
	buf.resize(size);
	buf.fill(0);
	data = buf.data();
	rollPos = 0;
}

void RepeatFinderSArrayWK::RollingArray::set(int pos, int val) {
	pos+=rollPos;
	if (pos >= size) {
		pos-=size;
	}
	data[pos] = val;
}

int RepeatFinderSArrayWK::RollingArray::get(int pos) {
	pos+=rollPos;
	if (pos >= size) {
		pos-=size;
	}
	return data[pos];
}

void RepeatFinderSArrayWK::RollingArray::roll() {
	data[rollPos] = 0;
	rollPos++;
	if (rollPos == size) {
		rollPos=0;
	}
}

}//namespace
