/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WorkflowRunTask.h"
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowManager.h>
#include <workflow/Schema.h>
#include <workflow_support/SchemaSerializer.h>

#include <core_api/AppContext.h>
#include <core_api/Log.h>
#include <core_api/Settings.h>


namespace GB2 {

WorkflowRunTask::WorkflowRunTask(const Schema& sh, QList<Iteration> lst) :
Task(tr("Execute workflow schema"), TaskFlags_NR_DWF) {
    if (lst.isEmpty()) {
        lst.append(Iteration(tr("Main")));
    }
    foreach(Iteration it, lst) {
        addSubTask(new WorkflowIterationRunTask(sh, it));
    }
    reportSupported = true;
}

QString WorkflowRunTask::generateReport() const {
    QString res;
    res+="<table width='75%'>";
    res+=QString("<tr><th>%1</th><th>%2</th><th>%3</th></tr>").arg(tr("Iterations")).arg(tr("Status")).arg(tr("Details"));
    foreach(Task* sub, getSubtasks()) {
        QString name = Qt::escape(sub->getTaskName());
        QString status = sub->hasErrors() ? tr("Failed") : sub->isCanceled() ? tr("Canceled") : tr("Finished");
        QString error = Qt::escape(sub->getError()).replace("\n", "<br>");
            //AppContext::getTaskScheduler()->getStateName(sub);
        if (sub->hasErrors()) {
            name = "<font color='red'>"+name+"</font>";
            status = "<font color='red'>"+status+"</font>";
        } else if (sub->isCanceled()) {
            status = "<font color='blue'>"+status+"</font>";
        } else {
            status = "<font color='green'>"+status+"</font>";
        }
        res+=QString("<tr><td>%1</td><td>%2</td><td>%3</td></tr>").arg(name).arg(status).arg(error);
    }
    res+="</table>";
    return res;
}

WorkflowIterationRunTask::WorkflowIterationRunTask(const Schema& sh, const Iteration& it) : 
Task(QString("%1").arg(it.name), TaskFlags_RASF_DWF_SSSOR), schema(new Schema()), scheduler(NULL) {
    QMap<ActorId, ActorId> map = SchemaSerializer::deepCopy(sh, schema);
    schema->applyConfiguration(it, map);
}

WorkflowIterationRunTask::~WorkflowIterationRunTask() {
    DomainFactory* df = WorkflowEnv::getDomainRegistry()->getById(schema->domain);
    if (df) {
        df->destroy(scheduler, schema);
    }
    delete schema;
}

void WorkflowIterationRunTask::prepare() {
    DomainFactory* df = WorkflowEnv::getDomainRegistry()->getById(schema->domain);
    if (!df) {
        stateInfo.error = tr("Unknown domain %1").arg(schema->domain);
        return;
    }
    foreach(Actor* a, schema->procs) {
        Worker* w = df->createWorker(a);
        if (!w) {
            stateInfo.error = tr("Failed to create worker %1 %2 in domain %3")
                .arg(a->getProto()->getId()).arg(a->getId()).arg(schema->domain);
            return;
        }
    }
    foreach(Link* l, schema->flows) {
        CommunicationChannel* cc = df->createConnection(l);
        if (!cc) {
            stateInfo.error = tr("Failed to create connection %1 %2 in domain %3"); //fixme
            return;
        }
    }
    scheduler = df->createScheduler(schema, df->getDefaultSettings());
    if (!scheduler) {
        stateInfo.error = tr("Failed to create scheduler in domain %1").arg(df->getDisplayName());
        return;
    }
    scheduler->init();
    while(scheduler->isReady() && !isCanceled()) {
        Task* t = scheduler->tick();
        if (t) {
            addSubTask(t);
            break;
        }
    }
}

QList<Task*> WorkflowIterationRunTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> tasks;
    if (subTask->hasErrors()) {
        propagateSubtaskError();
        return tasks;
    }
    while(scheduler->isReady() && !isCanceled()) {
        Task* t = scheduler->tick();
        if (t) {
            tasks << t;
            break;
        }
    }
    return tasks;
}

Task::ReportResult WorkflowIterationRunTask::report() {
    if (scheduler) {
        scheduler->cleanup();
        if (!scheduler->isDone()) {
            stateInfo.error += tr("\nSchema did not finish.");
        }
    }
    return ReportResult_Finished;
}

}//namespace
