/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "GenericReadWorker.h"
#include "GenericReadActor.h"
#include "CoreLib.h"
#include <workflow_library/BioActorLibrary.h>
#include <workflow/WorkflowEnv.h>
#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/Log.h>
#include <core_api/DocumentModel.h>
#include <core_api/IOAdapter.h>
#include <gobjects/GObjectTypes.h>
#include <gobjects/DNASequenceObject.h>
#include <gobjects/MAlignmentObject.h>
#include <document_format/DocumentFormatUtils.h>

namespace GB2 {
using namespace Workflow;
namespace LocalWorkflow {

static LogCategory log(ULOG_CAT_WD);

void GenericMSAReader::init() {
    mtype = WorkflowEnv::getDataTypeRegistry()->getById(GenericMAActorProto::TYPE);
    urls = DesignerUtils::expandToUrls(actor->getParameter("URL")->value.toString());
    assert(ports.size() == 1);
    ch = ports.values().first();
}

bool GenericMSAReader::isReady() {
    return !isDone();
}

Task* GenericMSAReader::tick() {
    if (cache.isEmpty() && !urls.isEmpty()) {
        Task* t = createReadTask(urls.takeFirst());
        connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
        return t;
    }
    while (!cache.isEmpty()) {
        ch->put(cache.takeFirst());
    }
    if (urls.isEmpty()) {
        done = true;
        ch->setEnded();
    }
    return NULL;
}

bool GenericMSAReader::isDone() {
    return done && cache.isEmpty();
}

void GenericMSAReader::sl_taskFinished() {
    LoadMSATask* t = qobject_cast<LoadMSATask*>(sender());
    if (!t->isFinished() || t->hasErrors()) {
        return;
    }
    foreach(MAlignment ma, t->results) {
        QVariantMap m;
        m.insert(CoreLib::URL_SLOT_ID, t->url);
        m.insert(BioActorLibrary::MA_SLOT_ID, qVariantFromValue<MAlignment>(ma)); 
        cache.append(Message(mtype, m));
    }
}

void LoadMSATask::run() {
    DocumentFormatConstraints mc;
    mc.supportedObjectTypes.append(GObjectTypes::MULTIPLE_ALIGNMENT);
    DocumentFormat* format = NULL;
    QList<DocumentFormat*> fs = DocumentFormatUtils::detectFormat(url);
    foreach(DocumentFormat* f, fs) {
        if (f->checkConstraints(mc)) {
            format = f;
            break;
        }
    }
    if (!format) {
        DocumentFormatConstraints sc;
        sc.supportedObjectTypes.append(GObjectTypes::DNA_SEQUENCE);
        foreach(DocumentFormat* f, fs) {
            if (f->checkConstraints(sc)) {
                format = f;
                break;
            }
        }
    }
    if (!format) {
        stateInfo.error = tr("Unsupported document format");
        return;
    }
    log.info(tr("Reading MSA from %1 [%2]").arg(url).arg(format->getFormatName()));
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
    Document *doc = format->loadExistingDocument(iof, url, stateInfo, QVariantMap());
    assert(isCanceled() || doc!=NULL || hasErrors());
    assert(doc == NULL || doc->isLoaded());
    if (!isCanceled() && doc && doc->isLoaded()) {
        if (format->checkConstraints(mc)) {
            foreach(GObject* go, doc->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT)) {
                results.append(((MAlignmentObject*)go)->getMAlignment());
            }
        } else {
            MAlignment ma;
            foreach(GObject* obj, doc->findGObjectByType(GObjectTypes::DNA_SEQUENCE)) {
                DNASequenceObject* dnaObj = qobject_cast<DNASequenceObject*>(obj);
                const DNASequence seq = dnaObj->getSequence();
                if (ma.alphabet == NULL) {
                    ma.alphabet = dnaObj->getAlphabet();
                } else {
                    ma.alphabet = DNAAlphabet::deriveCommonAlphabet(ma.alphabet, dnaObj->getAlphabet());
                    if (ma.alphabet == NULL) {
                        stateInfo.error = tr("Sequences in file have different alphabets.");
                        break;
                    }
                }
                if (seq.length() > MAX_ALI_LEN) {
                    stateInfo.error = tr("Sequence is too large for alignment.");
                    break;
                }
                ma.alignedSeqs.append(MAlignmentItem(dnaObj->getGObjectName(), seq.seq));
            }
            if (!hasErrors()) {
                ma.normalizeModel();
                results.append(ma);
            }
        }
    }
    if (doc && doc->isLoaded()) {
        doc->unload();
    }
}

void GenericSeqReader::init() {
    GenericMSAReader::init();
    mtype = WorkflowEnv::getDataTypeRegistry()->getById(GenericSeqActorProto::TYPE);
    GenericSeqActorProto::Mode mode = GenericSeqActorProto::Mode(actor->getParameter(GenericSeqActorProto::MODE_ATTR)->value.toInt());
    if (GenericSeqActorProto::MERGE == mode) {
        QString mergeToken = MERGE_MULTI_DOC_GAP_SIZE_SETTINGS;
        cfg[mergeToken] = actor->getParameter(GenericSeqActorProto::GAP_ATTR)->value.toInt();
    }
}

void GenericSeqReader::sl_taskFinished() {
    LoadSeqTask* t = qobject_cast<LoadSeqTask*>(sender());
    if (!t->isFinished() || t->hasErrors()) {
        return;
    }
    foreach(DNASequence seq, t->results) {
        QVariantMap m;
        m.insert(CoreLib::URL_SLOT_ID, t->url);
        m.insert(BioActorLibrary::SEQ_SLOT_ID, qVariantFromValue<DNASequence>(seq));
        cache.append(Message(mtype, m));
    }
}

void LoadSeqTask::run() {
    DocumentFormatConstraints constraints;
    constraints.supportedObjectTypes.append(GObjectTypes::DNA_SEQUENCE);
    DocumentFormat* format = NULL;
    foreach(DocumentFormat* f, DocumentFormatUtils::detectFormat(url)) {
        if (f->checkConstraints(constraints)) {
            format = f;
            break;
        }
    }
    if (!format) {
        stateInfo.error = tr("Unsupported document format");
        return;
    }
    log.info(tr("Reading sequences from %1 [%2]").arg(url).arg(format->getFormatName()));
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
    Document *doc = format->loadExistingDocument(iof, url, stateInfo, cfg);
    assert(isCanceled() || doc!=NULL || hasErrors());
    assert(doc == NULL || doc->isLoaded());
    if (!isCanceled() && doc && doc->isLoaded()) {
        foreach(GObject* go, doc->findGObjectByType(GObjectTypes::DNA_SEQUENCE)) {
            results.append(((DNASequenceObject*)go)->getDNASequence());
        }
    }
    if (doc && doc->isLoaded()) {
        doc->unload();
    }
}

} // Workflow namespace
} // GB2 namespace
