/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SiteconIOWorkers.h"
#include "SiteconWorkers.h"
#include "SiteconSearchDialogController.h"
#include <workflow/TypeSet.h>
#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_support/DelegateEditors.h>

#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>
#include <core_api/Log.h>

#include <util_gui/DialogUtils.h>

#include "SiteconIO.h"

/* TRANSLATOR GB2::SiteconIO */

namespace GB2 {
namespace LocalWorkflow {

static const QString IN_PORT("in");
static const QString OUT_PORT("out");
static const QString URL("URL");
static const QString URL_SLOT("URL");

const QString SiteconReader::ACTOR_ID("sitecon.read");
const QString SiteconWriter::ACTOR_ID("sitecon.write");

const QString SiteconWorkerFactory::SITECON_MODEL_TYPE_ID("sitecon.model");

DataTypePtr const SiteconWorkerFactory::SITECON_MODEL_TYPE() {
    DataTypeRegistry* dtr = WorkflowEnv::getDataTypeRegistry();
    assert(dtr);
    static bool startup = true;
    if (startup)
    {
        dtr->registerEntry(DataTypePtr(new DataType(SITECON_MODEL_TYPE_ID, SiteconIO::tr("Sitecon model"), "")));
        startup = false;
    }
    return dtr->getById(SITECON_MODEL_TYPE_ID);
}

const Descriptor SiteconWorkerFactory::SITECON_CATEGORY() {return Descriptor("hsitecon", SiteconIO::tr("SITECON"), "");}

static LogCategory log(ULOG_CAT_WD);

SiteconIOProto::SiteconIOProto(const Descriptor& desc, const QList<PortDescriptor*>& _ports, 
                       const QList<Attribute*>& _attrs) : BusActorPrototype(desc, _ports, _attrs)
{
    bool isReader = ports.first()->isOutput();
    Descriptor ud(URL, SiteconIO::tr("Location"), SiteconIO::tr("Location of the data file(s)."));
    attrs << new Attribute(ud, CoreDataTypes::STRING_TYPE(), isReader);
    setEditor(new DelegateEditor(URL, new URLDelegate(SiteconIO::getFileFilter(), SiteconIO::SITECON_ID, isReader)));
    icon = QIcon(":sitecon/images/sitecon.png");
    if (!isReader) {
        setValidator(new ScreenedParamValidator(URL, ports.first()->getId(), URL_SLOT));
    }
}

bool SiteconIOProto::isAcceptableDrop(const QMimeData* md, QVariantMap* params) const {
    if (md->hasUrls()) {
        QList<QUrl> urls = md->urls();
        if (urls.size() == 1)
        {
            QString url = urls.at(0).toLocalFile();
            QString ext = DialogUtils::getDocumentExtension(url);
            if (SiteconIO::SITECON_EXT == ext) {
                if (params) {
                    params->insert(URL, url);
                }
                return true;
            }
        }
    }

    return false;
}


void SiteconWorkerFactory::init() 
{
    WProtoRegistry* r = WorkflowEnv::getProtoRegistry();
    assert(r);
    DataTypePtr dt = SITECON_MODEL_TYPE();
    {        
        QMap<Descriptor, DataTypePtr> m;
        Descriptor sd(URL_SLOT, SiteconIO::tr("Location"), SiteconIO::tr("Location hint for the target file."));
        m[sd] = CoreDataTypes::STRING_TYPE();
        m[*dt] = dt;
        DataTypePtr t(new DataTypeSet(Descriptor("write.sitecon.content"), m));

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor pd(IN_PORT, SiteconIO::tr("Sitecon model"), SiteconIO::tr("Input Sitecon model"));
        p << new PortDescriptor(pd, t, true /*input*/);
        Descriptor desc(SiteconWriter::ACTOR_ID, SiteconIO::tr("Write Sitecon model"), SiteconIO::tr("Saves all input SITECON profiles to specified location."));
        BusActorPrototype* proto = new SiteconIOProto(desc, p, a);
        proto->setPrompter(new SiteconWritePrompter());
        r->registerProto(SITECON_CATEGORY(), proto);
    }
    {
        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor pd(OUT_PORT, SiteconIO::tr("Sitecon model"), SiteconIO::tr("Loaded SITECON profile data."));
        p << new PortDescriptor(pd, dt, false /*input*/, true /*multi*/);
        Descriptor desc(SiteconReader::ACTOR_ID, SiteconIO::tr("Read Sitecon model"), SiteconIO::tr("Reads SITECON profiles from file(s). The files can be local or Internet URLs."));
        BusActorPrototype* proto = new SiteconIOProto(desc, p, a);
        proto->setPrompter(new SiteconReadPrompter());
        r->registerProto(SITECON_CATEGORY(), proto);
    }

    SiteconBuildWorker::registerProto();
    SiteconSearchWorker::registerProto();

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new SiteconWorkerFactory(SiteconReader::ACTOR_ID));
    localDomain->registerEntry(new SiteconWorkerFactory(SiteconWriter::ACTOR_ID));
    localDomain->registerEntry(new SiteconWorkerFactory(SiteconSearchWorker::ACTOR_ID));
    localDomain->registerEntry(new SiteconWorkerFactory(SiteconBuildWorker::ACTOR_ID));
}

Worker* SiteconWorkerFactory::createWorker(Actor* a) {
    BaseWorker* w = NULL;
    if (SiteconReader::ACTOR_ID == a->getProto()->getId()) {
        w = new SiteconReader(a);
    } 
    else if (SiteconWriter::ACTOR_ID == a->getProto()->getId()) {
        w = new SiteconWriter(a);
    }
    else if (SiteconBuildWorker::ACTOR_ID == a->getProto()->getId()) {
        w = new SiteconBuildWorker(a);
    }
    else if (SiteconSearchWorker::ACTOR_ID == a->getProto()->getId()) {
        w = new SiteconSearchWorker(a);
    }

    return w;    
}

QString SiteconReadPrompter::composeRichDoc() {
    return tr("Read model(s) from <u>%1</u>").arg(getURL(URL));
}

QString SiteconWritePrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(IN_PORT));
    Actor* producer = input->getProducer(SiteconWorkerFactory::SITECON_MODEL_TYPE_ID);
    if (!producer) {
        return getURL(URL);
    }
    QString url = getScreenedURL(input, URL, URL_SLOT); 
    QString doc = tr("Save the profile(s) from <u>%1</u> to %2.")
        .arg(producer->getLabel())
        .arg(url);
    return doc;
}

void SiteconReader::init() {
    output = ports.value(OUT_PORT);
    urls = DesignerUtils::expandToUrls(actor->getParameter(URL)->value.toString());
    mtype = SiteconWorkerFactory::SITECON_MODEL_TYPE();
}

Task* SiteconReader::tick() {
    Task* t = new SiteconReadTask(urls.takeFirst());
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    tasks.append(t);
    return t;
}

void SiteconReader::sl_taskFinished() {
    SiteconReadTask* t = qobject_cast<SiteconReadTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    tasks.removeAll(t);
    if (output) {
        if (!t->hasErrors()) {
            QVariant v = qVariantFromValue<SiteconModel>(t->getResult());
            output->put(Message(mtype, v));
        }
        if (urls.isEmpty() && tasks.isEmpty()) {
            output->setEnded();
        }
        log.info(tr("Loaded SITECON model from %1").arg(t->getUrl()));
    }
}

void SiteconWriter::init() {
    input = ports.value(IN_PORT);
    url = actor->getParameter(URL)->value.toString();
}

Task* SiteconWriter::tick() {
    QVariantMap data = input->get().getData().toMap();
    SiteconModel model = data.value(SiteconWorkerFactory::SITECON_MODEL_TYPE_ID).value<SiteconModel>();
    QString anUrl = url;
    if (anUrl.isEmpty()) {
        anUrl = data.value(URL_SLOT).toString();
    }
    if (anUrl.isEmpty()) {
        //TODO anUrl = url;
    }
    assert(!anUrl.isEmpty());
    int count = ++counter[anUrl];
    if (count != 1) {
        anUrl = DialogUtils::prepareFileName(anUrl, count, QStringList("sitecon"));
    } else {
        anUrl = DialogUtils::ensureFileExt(anUrl, QStringList("sitecon"));
    }
    log.info(tr("Writing SITECON model to %1").arg(anUrl));
    return new SiteconWriteTask(anUrl, model);
}

} //namespace LocalWorkflow
} //namespace GB2

