/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "HMMIOWorker.h"

#include <workflow/TypeSet.h>
#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_support/DelegateEditors.h>

#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>
#include <core_api/Log.h>

#include <util_gui/DialogUtils.h>

#include "HMMIO.h"
#include "u_search/HMMSearchDialogController.h"
#include "u_build/HMMBuildWorker.h"
#include "u_search/HMMSearchWorker.h"

/* TRANSLATOR GB2::LocalWorkflow::HMMLib */

namespace GB2 {
namespace LocalWorkflow {

static const QString IN_PORT("in");
static const QString OUT_PORT("out");
static const QString URL_SLOT("URL");

static const QString URL("URL");

const QString HMMReader::ACTOR("uhmmer.read");
const QString HMMWriter::ACTOR("uhmmer.write");

const QString HMMLib::HMM_PROFILE_TYPE_ID("hmm.profile");

DataTypePtr HMMLib::HMM_PROFILE_TYPE() {
    DataTypeRegistry* dtr = WorkflowEnv::getDataTypeRegistry();
    assert(dtr);
    static bool startup = true;
    if (startup)
    {
        dtr->registerEntry(DataTypePtr(new DataType(HMM_PROFILE_TYPE_ID, tr("HMM Profile"), "")));
        startup = false;
    }
    return dtr->getById(HMM_PROFILE_TYPE_ID);
}

const Descriptor HMMLib::HMM_CATEGORY() {return Descriptor("hmmer", tr("HMMER tools"), "");}

static LogCategory log(ULOG_CAT_WD);

HMMIOProto::HMMIOProto(const Descriptor& desc, const QList<PortDescriptor*>& _ports, 
                       const QList<Attribute*>& _attrs) : BusActorPrototype(desc, _ports, _attrs)
{
    bool isReader = ports.first()->isOutput();
    Descriptor ud(URL, HMMLib::tr("Location"), HMMLib::tr("Location of the data file."));
    attrs << new Attribute(ud, CoreDataTypes::STRING_TYPE(), isReader);
    setEditor(new DelegateEditor(URL, new URLDelegate(HMMIO::getHMMFileFilter(), HMMIO::HMM_ID, isReader)));
    icon = QIcon(":/uhmmer/images/hmmer_16.png");
    if (!isReader) {
        setValidator(new ScreenedParamValidator(URL, ports.first()->getId(), URL_SLOT));
    }
}

bool HMMIOProto::isAcceptableDrop(const QMimeData* md, QVariantMap* params) const {
    if (md->hasUrls()) {
        QList<QUrl> urls = md->urls();
        if (urls.size() == 1)
        {
            QString url = urls.at(0).toLocalFile();
            QString ext = DialogUtils::getDocumentExtension(url);
            if (ext == HMMIO::HMM_EXT) {
                if (params) {
                    params->insert(URL, url);
                }
                return true;
            }
        }
    }

    return false;
}


void HMMIOWorkerFactory::init() {
 
     WProtoRegistry* r = WorkflowEnv::getProtoRegistry();
     assert(r);
     DataTypePtr dt = HMMLib::HMM_PROFILE_TYPE();
     {
         Descriptor id(IN_PORT, HMMLib::tr("HMM profile"), HMMLib::tr("Input HMM profile"));
         Descriptor ud(URL_SLOT, HMMLib::tr("Location"), HMMLib::tr("Location hint for the target file."));

         QMap<Descriptor, DataTypePtr> m;
         m[ud] = CoreDataTypes::STRING_TYPE();
         m[*dt] = dt;
         DataTypePtr t(new DataTypeSet(Descriptor("write.hmm.content"), m));

         QList<PortDescriptor*> p; QList<Attribute*> a;
         p << new PortDescriptor(id, t, true /*input*/);
         
         Descriptor desc(HMMWriter::ACTOR, HMMLib::tr("Write HMM profile"), HMMLib::tr("Saves all input HMM profiles to specified location."));
         BusActorPrototype* proto = new HMMIOProto(desc, p, a);
         proto->setPrompter(new HMMWritePrompter());
         r->registerProto(HMMLib::HMM_CATEGORY(), proto);
     }
     {
         Descriptor od(OUT_PORT, HMMLib::tr("HMM profile"), HMMLib::tr("Loaded HMM profile"));

         QList<PortDescriptor*> p; QList<Attribute*> a;
         p << new PortDescriptor(od, dt, false /*input*/, true);
         Descriptor desc(HMMReader::ACTOR, HMMLib::tr("Read HMM profile"), HMMLib::tr("Reads HMM profiles from file(s). The files can be local or Internet URLs."));
         BusActorPrototype* proto = new HMMIOProto(desc, p, a);
         proto->setPrompter(new HMMReadPrompter());
         r->registerProto(HMMLib::HMM_CATEGORY(), proto);
     }
 
     DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
     localDomain->registerEntry(new HMMIOWorkerFactory(HMMReader::ACTOR));
     localDomain->registerEntry(new HMMIOWorkerFactory(HMMWriter::ACTOR));
}

QString HMMReadPrompter::composeRichDoc() {
    return tr("Read HMM profile(s) from %1").arg(getURL(URL));
}

QString HMMWritePrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(IN_PORT));
    Actor* producer = input->getProducer(IN_PORT);
    if (!producer) {
        return getURL(URL);
    }
    QString url = getScreenedURL(input, URL, URL_SLOT); 
    QString doc = tr("Save HMM profile(s) from <u>%1</u> to <u>%2</u>.")
        .arg(producer->getLabel())
        .arg(url);
    return doc;
}

void HMMIOWorkerFactory::cleanup() {
     DomainFactory* ld = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
     DomainFactory* f = ld->unregisterEntry(HMMReader::ACTOR); 
     delete f;
     f = ld->unregisterEntry(HMMWriter::ACTOR);
     delete f;
 
     WProtoRegistry* r = WorkflowEnv::getProtoRegistry();
     ActorPrototype* p = r->unregisterProto(HMMReader::ACTOR);
     assert(p); delete p;
     p = r->unregisterProto(HMMWriter::ACTOR);
     assert(p); delete p;
}

Worker* HMMIOWorkerFactory::createWorker(Actor* a) {
    BaseWorker* w = NULL;
     if (HMMReader::ACTOR == a->getProto()->getId()) {
         w = new HMMReader(a);
     } 
     else if (HMMWriter::ACTOR == a->getProto()->getId()) {
         w = new HMMWriter(a);
     }

    return w;    
}

void HMMReader::init() {
    output = ports.value(OUT_PORT);
    urls = DesignerUtils::expandToUrls(actor->getParameter(URL)->value.toString());
}

Task* HMMReader::tick() {
    Task* t = new HMMReadTask(urls.takeFirst());
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

void HMMReader::sl_taskFinished() {
    HMMReadTask* t = qobject_cast<HMMReadTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    if (output) {
        if (!t->hasErrors()) {
            QVariant v = qVariantFromValue<plan7_s*>(t->getHMM());
            output->put(Message(HMMLib::HMM_PROFILE_TYPE(), v));
        }
        if (urls.isEmpty()) {
            output->setEnded();
        }
        log.info(tr("Loaded HMM profile from %1").arg(t->getUrl()));
    }
}

void HMMWriter::init() {
    input = ports.value(IN_PORT);
    url = actor->getParameter(URL)->value.toString();
}

Task* HMMWriter::tick() {
    QVariantMap data = input->get().getData().toMap();
    plan7_s* hmm = data.value(HMMLib::HMM_PROFILE_TYPE_ID).value<plan7_s*>();
    QString anUrl = url;
    if (anUrl.isEmpty()) {
        anUrl = data.value(URL_SLOT).toString();
    }
    if (anUrl.isEmpty()) {
        //TODO anUrl = url;
    }
    assert(!anUrl.isEmpty());
    int count = ++counter[anUrl];
    if (count != 1) {
        anUrl = DialogUtils::prepareFileName(anUrl, count, QStringList(HMMIO::HMM_EXT));
    } else {
        anUrl = DialogUtils::ensureFileExt(anUrl, QStringList(HMMIO::HMM_EXT));
    }
    log.info(tr("Writing HMM profile to %1").arg(anUrl));
    return new HMMWriteTask(anUrl, hmm);
}


void HMMLib::init() {
    HMMIOWorkerFactory::init();
    HMMBuildWorkerFactory::init();
    HMMSearchWorkerFactory::init();
}

void HMMLib::cleanup() {
    //FIXME need locking
    //HMMIOWorkerFactory::cleanup();
    //HMMBuildWorkerFactory::cleanup();
    //HMMSearchWorkerFactory::cleanup();

    //DataTypeRegistry* dr = WorkflowEnv::getDataTypeRegistry();
    //dr->unregisterEntry(HMM_PROFILE_TYPE->getId());
}

} //namespace LocalWorkflow
} //namespace GB2
