/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "CheckUpdatesTask.h"
#include "AppGlobals.h"

#include <core_api/AppContext.h>
#include <core_api/AppSettings.h>
#include <core_api/NetworkConfiguration.h>
#include <util_gui/GUIUtils.h>

#include <QtGui/QMessageBox>
#include <QtGui/QPushButton>

namespace GB2 {

SyncHTTP::SyncHTTP(const QString& hostName, quint16 port, QObject* parent)
: QHttp(hostName,port,parent), requestID(-1)
{
    connect(this,SIGNAL(requestFinished(int,bool)),SLOT(finished(int,bool)));
}

QString SyncHTTP::syncGet(const QString& path) {
    assert(requestID == -1);
    QBuffer to;
    requestID = get(path, &to);
    loop.exec();
    return QString(to.data());
}

void SyncHTTP::finished(int idx, bool err) {
    Q_UNUSED(err);
    assert(idx = requestID);
    loop.exit();
}


CheckUpdatesTask::CheckUpdatesTask() 
:Task(tr("Check for updates"), TaskFlag_DeleteWhenFinished)
{
    setVerboseLogMode(true);
}

#define SITE_URL  QString("ugene.unipro.ru")
#define PAGE_NAME QString("/current_version.html")

void CheckUpdatesTask::run() {
    stateInfo.stateDesc = tr("Connecting to updates server");
    NetworkConfiguration* nc = AppContext::getAppSettings()->getNetworkConfiguration();
    bool isProxy = nc->isProxyUsed(QNetworkProxy::HttpProxy);
    bool isException = nc->getExceptionsList().contains(SITE_URL);
    SyncHTTP http(SITE_URL);
    if (isProxy && !isException) {
        http.setProxy(nc->getProxy(QNetworkProxy::HttpProxy));
    }
    siteVersion = http.syncGet(PAGE_NAME);
    stateInfo.stateDesc.clear();

    if (http.error() != QHttp::NoError) {
        stateInfo.error = tr("Connection error: %1").arg(http.errorString());
        return;
    }
}

Task::ReportResult CheckUpdatesTask::report() {
    if (hasErrors()) {
        return ReportResult_Finished;
    }
    QString thisVersion = QString("%1.%2.%3").arg(UGENE_VERSION_1).arg(UGENE_VERSION_2).arg(UGENE_VERSION_3);
    QString message = "<table><tr><td>"+tr("Your version:") + "</td><td><b>"+thisVersion+"</b></td></tr>";
    message+="<tr><td>" + tr("Latest version:") + "</td><td><b>"+siteVersion+"</b></td></tr></table>";
    bool needUpdate = thisVersion != siteVersion;
    if (!needUpdate) {
        message += "<p>" + tr("You have the latest  version");
    }
    
    QMessageBox box(QMessageBox::Information, tr("Version information"), message);
    box.addButton(QMessageBox::Ok);
    QAbstractButton* updateButton = NULL;
    
    if (needUpdate) {
        updateButton = box.addButton(tr("Visit web site"), QMessageBox::ActionRole);
    }
    box.exec();

    if (box.clickedButton() == updateButton) {
        GUIUtils::runWebBrowser("http://ugene.unipro.ru/download.html");
    }
    return ReportResult_Finished;    
}


} //namespace
