/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "PluginSupportImpl.h"
#include "ServiceRegistryImpl.h"

#include <core_api/AppContext.h>
#include <core_api/Settings.h>
#include <core_api/Log.h>

#include <QtCore/QCoreApplication>
#include <QtCore/QLibrary>
#include <QtCore/QDir>
#include <QtCore/QFileInfo>

#include <algorithm>
#include <memory>


namespace GB2 {

/* TRANSLATOR GB2::PluginSupportImpl */
#define PLUGINS_LIST_SETTINGS QString("plugin_support/list/")
#define SKIP_LIST_SETTINGS QString("plugin_support/skip_list/")

static LogCategory log(ULOG_CAT_USER_INTERFACE);

static QStringList findAllPluginsInDefaultPluginsDir();

static QString prepareURL(QString url) {
#ifdef Q_OS_WIN
    //uppercase first letter -> QLibrary.fileName returned in upper case but QDir in lower (Qt4.4) !
    url = url[0].toUpper() + url.mid(1); 
#endif
    return url;
}

PluginRef::PluginRef(Plugin* _plugin, QLibrary* _library, const QString& _absPath)
: plugin(_plugin), library(_library), url(prepareURL(_absPath)), removeFlag(false) 
{
    pluginId = getPluginId(url);
}

QString PluginRef::getPluginId(const QString& url) {
    return QFileInfo(url).fileName();//TODO: get from plugin
}


PluginSupportImpl::PluginSupportImpl(Plugin* corePlugin) {
	serviceRegistry = new ServiceRegistryImpl();
    
    //1. load core services;
    registerPlugin(new PluginRef(corePlugin, NULL, ""));

    //2. read the settings and load all plugins
    Settings* settings = AppContext::getSettings();
    QStringList allKeys= settings->getAllKeys(PLUGINS_LIST_SETTINGS);
    QStringList processedPlugins;
    foreach (QString pluginId, allKeys) {
        QString file = settings->getValue(PLUGINS_LIST_SETTINGS + pluginId).toString();
        processedPlugins.append(file);
		AppContext::getTaskScheduler()->registerTopLevelTask(new AddPluginTask(this, file));
    }

    //3. read all plugins from the current folder and from ./plugins folder
    // use SKIP list to learn which plugin should not be loaded
    QStringList skipFiles = settings->getValue(SKIP_LIST_SETTINGS, QStringList()).toStringList();
    QStringList pluginFiles = findAllPluginsInDefaultPluginsDir();
    foreach (QString file, pluginFiles) {
        if (!skipFiles.contains(file) && !processedPlugins.contains(file)) {
            AppContext::getTaskScheduler()->registerTopLevelTask(new AddPluginTask(this, file));
        }
    }

}


PluginSupportImpl::~PluginSupportImpl() {
    while (!plugins.empty()) {
        Plugin* p = plugins.back();
        unregisterPlugin(p);
    }
	delete serviceRegistry;
}


static QStringList findAllPluginsInDefaultPluginsDir() {
    QDir d = PluginSupportImpl::getDefaultPluginsDir();
    QStringList filter; filter<<QString("*.")+PLUGIN_FILE_EXT;
    QStringList fileNames = d.entryList(filter, QDir::Readable | QDir::Files, QDir::NoSort);
    QStringList res;
    foreach(const QString& name, fileNames) {
        QString filePath = prepareURL(d.absolutePath() + "/" + name);
        res.append(filePath);
    }
    return res;
}

ServiceRegistry* PluginSupportImpl::getServiceRegistry() const {
	return serviceRegistry;
}


PluginRef::~PluginRef() {
    assert(plugin!=NULL);
    delete plugin;
    if (library!=NULL) { //library == null for core plugin
        delete library;
    } 
}

void PluginSupportImpl::registerPlugin(PluginRef* ref) {
    plugRefs.push_back(ref);
    plugins.push_back(ref->plugin);
    updateSavedState(ref);
    emit si_pluginAdded(ref->plugin);

	foreach(Service* s, ref->plugin->getServices()) {
		AppContext::getTaskScheduler()->registerTopLevelTask(serviceRegistry->registerServiceTask(s));
	}
}

void PluginSupportImpl::unregisterPlugin(Plugin* p) {
    assert(plugins.size() == plugRefs.size());

	
	foreach(PluginRef* ref, plugRefs) {
        if (ref->plugin == p) {
			plugRefs.removeAll(ref);
			plugins.removeAll(p);
			serviceRegistry->unregisterPluginServices(p);
            delete ref;
            break;
        } 
    }
    assert(plugins.size() == plugRefs.size());
}

QString PluginSupportImpl::getPluginFileURL(Plugin* p) const {
    assert(plugins.size() == plugRefs.size());

    foreach(PluginRef* ref, plugRefs) {
        if (ref->plugin == p) {
            if (ref->library == NULL) {
                return "";
            }
            return ref->library->fileName();
            
        } 
    }
    return QString::null;
}


Task* PluginSupportImpl::addPluginTask(const QString& pathToPlugin) {
	return new AddPluginTask(this, pathToPlugin);
}

PluginRef* PluginSupportImpl::findRef(Plugin* p) const {
    foreach(PluginRef* r, plugRefs) {
        if (r->plugin == p) {
            return r;
        }
    }
    return NULL;
}

PluginRef* PluginSupportImpl::findRefById(const QString& pluginId) const {
foreach(PluginRef* r, plugRefs) {
        if (r->pluginId == pluginId) {
            return r;
        }
    }
    return NULL;
}

PluginRef* PluginSupportImpl::findRefByURL(const QString& url) const {
    foreach(PluginRef* r, plugRefs) {
        if (r->url == url) {
            return r;
        }
    }
    return NULL;
}

//plugin will not be removed from the plugin list during the next app run
void PluginSupportImpl::setRemoveFlag(Plugin* p, bool v) {
    PluginRef* r = findRef(p);
    assert(r!=NULL);
    if (r->removeFlag == v) {
        return;
    }
    r->removeFlag = v;
    updateSavedState(r);
    emit si_pluginRemoveFlagChanged(p);
}

bool PluginSupportImpl::getRemoveFlag(Plugin* p) const {
    PluginRef* r = findRef(p);
    return r->removeFlag;
}

void PluginSupportImpl::updateSavedState(PluginRef* ref) {
    if (ref->library == NULL) {
        // skip core plugin
        return;
    }
    Settings* settings = AppContext::getSettings();

    if (ref->removeFlag) {
        settings->remove(PLUGINS_LIST_SETTINGS + ref->pluginId);

        //add to skip-list if auto-loaded
        if (isDefaultPluginsDir(ref->url)) {
            QStringList skipFiles = settings->getValue(SKIP_LIST_SETTINGS, QStringList()).toStringList();
            if (!skipFiles.contains(ref->url)) {
                skipFiles.append(ref->url);
                settings->setValue(SKIP_LIST_SETTINGS, skipFiles);
            }
        }
    } else {
        settings->setValue(PLUGINS_LIST_SETTINGS + ref->pluginId, ref->url);

        //remove from skip-list if present
        if (isDefaultPluginsDir(ref->url)) {
            QStringList skipFiles = settings->getValue(SKIP_LIST_SETTINGS, QStringList()).toStringList();
            if (skipFiles.removeOne(ref->url)) {
                settings->setValue(SKIP_LIST_SETTINGS, skipFiles);
            }
        }
    }
}

QDir PluginSupportImpl::getDefaultPluginsDir() {
    return QDir(QCoreApplication::applicationDirPath() + "/plugins");
}

bool PluginSupportImpl::isDefaultPluginsDir(const QString& url) {    
    QDir urlAbsDir = QFileInfo(url).absoluteDir();
    QDir plugsDir = getDefaultPluginsDir();
    return  urlAbsDir == plugsDir;
}


//////////////////////////////////////////////////////////////////////////
/// Tasks

//todo: improve task naming
AddPluginTask::AddPluginTask(PluginSupportImpl* _ps, const QString& _url) 
: Task(tr("add_plugin_task_url_%1").arg(_url), TaskFlags_NR_DWF), ps(_ps), url(_url)
{
}

Task::ReportResult AddPluginTask::report() {
	QFileInfo libFile(url);
	if (!libFile.exists()) {
		stateInfo.error = tr("plugin_file_not_found_%1").arg(url);
		return ReportResult_Finished;
	}

	if (!libFile.isFile()) { 
		stateInfo.error = tr("plugin_invalid_file_format_%1").arg(url);
		return ReportResult_Finished;
	}

	//check that plugin is not loaded
	QString absPath = prepareURL(libFile.absoluteFilePath());
    PluginRef* ref = ps->findRefByURL(absPath); //TODO: remove because getPluginId is enough?
    if (ref == NULL) {
        ref = ps->findRefById(PluginRef::getPluginId(absPath));
    }
    if (ref != NULL) {
        stateInfo.error = tr("plugin_already_loaded_%1").arg(url);
		return ReportResult_Finished;
	}

    //load library
    std::auto_ptr<QLibrary> lib(new QLibrary(absPath));
    bool loadOk = lib->load();
    if (!loadOk) {
        stateInfo.error = tr("plugin_library_loading_error_%1").arg(absPath);
        return ReportResult_Finished;
    }

	//TODO: version check

	//instantiate plugin
	PLUG_INIT_FUNC init_fn = PLUG_INIT_FUNC((lib->resolve(GB2_PLUGIN_INIT_FUNC_NAME)));
	if (!init_fn) {
		stateInfo.error = tr("plugin_init_function_not_found_%1").arg(absPath);
		return ReportResult_Finished;
	}

	Plugin* p = init_fn();
	if (p == NULL) {
		stateInfo.error = tr("plugin_init_failed_%1").arg(absPath);
		return ReportResult_Finished;
	}

    ref = new PluginRef(p, lib.release(), absPath);
	ps->registerPlugin(ref);

    return ReportResult_Finished;
}


}//namespace
