/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "TaskSchedulerImpl.h"

#include <core_api/Log.h>

#include <QtGui/QApplication>

/* TRANSLATOR GB2::TaskSchedulerImpl */

namespace GB2 {

static LogCategory log(ULOG_CAT_TASKS);

TaskSchedulerImpl::TaskSchedulerImpl() {
    stateNames << tr("state_name_new") <<  tr("state_name_prepared") <<  tr("state_name_running") << tr("state_name_finished");
    connect(&timer, SIGNAL(timeout()), this, SLOT(update()));
    timer.start(100);
}

TaskSchedulerImpl::~TaskSchedulerImpl() {
	assert(topLevelTasks.empty());
	assert(priorityQueue.isEmpty());
}


void TaskSchedulerImpl::cancelTask(Task* task) {
	//TODO: do not run canceled tasks!
    if (task->getState() < Task::State_Finished) {
        log.info(tr("canceling task: %1").arg(task->getTaskName()));
		getTaskStateInfo(task).cancelFlag = true;
		foreach(Task* t, task->getSubtasks()) {
			cancelTask(t);
		}
	}
}

void TaskSchedulerImpl::cancelAllTasks() {
	foreach(Task* t, topLevelTasks) {
		cancelTask(t);
	}
}


bool TaskSchedulerImpl::processFinishedTasks() {
	bool hasFinished = false;
	for (int i = priorityQueue.size(); --i>=0;) {
		TaskInfo* ti = priorityQueue[i];
		TaskInfo* pti = ti->parentTaskInfo;
		assert(ti->task->getState()!=Task::State_Finished);
        updateTaskProgressAndDesc(ti);
		if (!readyToFinish(ti)) {
			continue;
		}
        if (ti->wasPrepared) {
            Task::ReportResult res = ti->task->report();
            if (res == Task::ReportResult_CallMeAgain) {
                continue;
            }
        }
		hasFinished = true;

		Task* task = ti->task;
		promoteTask(ti, Task::State_Finished);
		priorityQueue.removeAt(i);
		delete ti;

		// check if there are new subtasks from parent
		if (pti != NULL) {
			Task* parentTask = pti->task;
            if (!parentTask->hasErrors() && !parentTask->isCanceled()) {
                if (task->isCanceled() && parentTask->getFlags().testFlag(TaskFlag_FailOnSubtaskCancel)) {
                    getTaskStateInfo(parentTask).error = tr("Subtask {%1} is canceled").arg(task->getTaskName()); 
                    getTaskStateInfo(parentTask).cancelFlag = true;
                } else if (task->hasErrors() && parentTask->getFlags().testFlag(TaskFlag_FailOnSubtaskError)) {
                    getTaskStateInfo(parentTask).error = tr("Subtask {%1} is failed").arg(task->getTaskName()); 
                    //TODO: ? getTaskStateInfo(parentTask).cancelFlag = true;
                }
            }
			QList<Task*> newSubTasks = onSubTaskFinished(parentTask, task);
			if (!newSubTasks.isEmpty() || !pti->newSubtasks.isEmpty()) {
				if (!tasksWithNewSubtasks.contains(pti)) {
					tasksWithNewSubtasks.append(pti);
				}
			}
			foreach(Task* newSub, newSubTasks) {
                pti->newSubtasks.append(newSub);
                addSubTask(parentTask, newSub);
			}
		}
	}
	return hasFinished;
}

void TaskSchedulerImpl::unregisterFinishedTopLevelTasks() {
	QList<Task*> tasksToDelete;
	foreach(Task* task, topLevelTasks) {
		if (task->getState() == Task::State_Finished) {
			tasksToDelete.push_back(task);
		}
	}
	foreach(Task* task, tasksToDelete) {
		unregisterTopLevelTask(task);
	}
}

void TaskSchedulerImpl::processNewSubtasks() {
	for (int i=0, n = tasksWithNewSubtasks.size();i<n; i++) {
		TaskInfo* ti = tasksWithNewSubtasks[i];
		TaskFlags flags =  ti->task->getFlags();
		assert(ti->newSubtasks.size() > 0);
		if (!flags.testFlag(TaskFlag_SerialSubtasks)) {
			foreach(Task* newSub, ti->newSubtasks) {
				addToPriorityQueue(newSub, ti);
			}
			ti->newSubtasks.clear();
		} else if (ti->numRunningSubtasks == 0) { //serial
            Task* newSub = ti->newSubtasks.takeFirst();						
			addToPriorityQueue(newSub, ti);
		}
		if (ti->newSubtasks.isEmpty()) {
			tasksWithNewSubtasks[i] = NULL;
		}
	}
	tasksWithNewSubtasks.removeAll(NULL);
}


void TaskSchedulerImpl::runReady() {
	foreach(TaskInfo* ti, priorityQueue) {
		Task* task = ti->task;
		assert(task->getState() >= Task::State_Prepared);
		if (task->getState() == Task::State_Prepared) {
			promoteTask(ti, Task::State_Running);
		}
		if ( ti->runFinished || ti->thread!=NULL ) {
			continue;
		}
		bool ready = !task->hasFlags(TaskFlag_RunAfterAllSubtasksFinished) || ti->numFinishedSubtasks == task->getSubtasks().size();
        if (ready) {
			runTask(ti);
		}
	}
}


void TaskSchedulerImpl::update() {
	static bool recursion = false;
	if (recursion) {
		return;
	}
	recursion = true;
	
	bool finishedFound = processFinishedTasks();
	if (finishedFound) {
		unregisterFinishedTopLevelTasks();
	}
	processNewSubtasks();
	
	prepareNewTasks();

	runReady();

	recursion = false;
}

void TaskSchedulerImpl::prepareNewTasks() {
    if (newTasks.empty()) {
        return;
    }
    QList<Task*> newCopy = newTasks;
    newTasks.clear();
	foreach(Task* task, newCopy) {
		addToPriorityQueue(task, NULL);
	}
}

void TaskSchedulerImpl::registerTopLevelTask(Task* task) {

#ifdef _DEBUG
    QThread* appThread = QApplication::instance()->thread();
    QThread* thisThread = QThread::currentThread();
    QThread* taskThread = task->thread();
    assert(appThread == thisThread);
    assert(taskThread == thisThread);
#endif

	assert(task->getState() == Task::State_New);

	log.details(tr("registering new task %1").arg(task->getTaskName()));
	topLevelTasks.push_back(task);
	emit si_topLevelTaskRegistered(task);
	newTasks.append(task);
}

void TaskSchedulerImpl::addToPriorityQueue(Task* task, TaskInfo* pti) {
    TaskFlags flags = task->getFlags();

    //TaskFlag_StopOnSubtaskError supported only in TaskFlag_SerialSubtasks mode today
    assert(!flags.testFlag(TaskFlag_StopOnSubtaskError) || flags.testFlag(TaskFlag_SerialSubtasks));

    
	TaskInfo* ti = new TaskInfo(task, pti);
	priorityQueue.push_back(ti);

    bool cancelTask = false;
    if (pti!=NULL) {
        cancelTask = pti->task->isCanceled()
                    || (pti->task->getFlags().testFlag(TaskFlag_StopOnSubtaskError) && pti->task->hasSubtasksWithErrors());
    }
    if (cancelTask) { //prepare not called if task is canceled
        task->cancel();
    } else {
	    task->prepare();
        ti->wasPrepared = true;
    } 
	promoteTask(ti, Task::State_Prepared);

    bool serial = task->getFlags().testFlag(TaskFlag_SerialSubtasks);
	const QList<Task*>& subtasks = task->getSubtasks();
	for (int i = 0, n = subtasks.size(); i < n; i++) {
		Task* sub = subtasks[i];
		assert(sub->getState() == Task::State_New);
		if (!serial || i == 0) {
			addToPriorityQueue(sub, ti);
		} else {
			assert(serial && i > 0);
			ti->newSubtasks.append(sub);
		}
	}
}
	
void TaskSchedulerImpl::unregisterTopLevelTask(Task* task) {
	assert(topLevelTasks.contains(task));
	log.trace(tr("unregistering task %1").arg(task->getTaskName()));
	stopTask(task);
	topLevelTasks.removeAll(task);
	
    emit si_topLevelTaskUnregistered(task);

    if (task->hasFlags(TaskFlag_DeleteWhenFinished)) {
        deleteTask(task);
    }
}

void TaskSchedulerImpl::stopTask(Task* task) {
	foreach(Task* sub, task->getSubtasks()) {
		stopTask(sub);
	}

	foreach(TaskInfo* ti,  priorityQueue) { //stop task if its running
		if (ti->task == task) {
			cancelTask(task);
			if (ti->thread!=NULL && !ti->runFinished) {
				ti->thread->wait();//TODO: try avoid blocking here
			}
			assert(readyToFinish(ti));
			break;
		}
	}
}


bool TaskSchedulerImpl::readyToFinish(TaskInfo* ti) {
	if (ti->task->getState() == Task::State_Finished) {
		return true;
	}
	if (ti->task->getState() != Task::State_Running) {
		return false;
	}
	if (ti->numFinishedSubtasks < ti->task->getSubtasks().size()) {
		return false;
	}
	if (!ti->runFinished) {
		return false;
	}
#ifdef _DEBUG	
	foreach(Task* sub, ti->task->getSubtasks()) {
		assert(sub->getState() == Task::State_Finished);
	}
	assert(ti->newSubtasks.isEmpty());
#endif
	return true;
}

QString TaskSchedulerImpl::getStateName(Task* t) const {
    Task::State s = t->getState();
    return stateNames[s];
}


QDateTime TaskSchedulerImpl::estimatedFinishTime(Task* task) const {
	if (task->getState()!=Task::State_Running) {
		assert(0);
		return QDateTime();
	}
	const TaskTimeInfo& tti = task->getTimeInfo();
	int secsPassed = tti.startTime.secsTo(QDateTime::currentDateTime());
	float percentInSecs = task->getProgress() / (float)secsPassed;
	int secsTotal = int(percentInSecs * 100);
	int secsLeft = secsTotal - secsPassed;
	QDateTime res = QDateTime::currentDateTime();
	res.addSecs(secsLeft);
	return res;
}

static QString state2String(Task::State state) {
	switch(state) {
		case Task::State_New:
			return TaskSchedulerImpl::tr("state_name_new");
		case Task::State_Prepared:
			return TaskSchedulerImpl::tr("state_name_prepared");
		case Task::State_Running:
			return TaskSchedulerImpl::tr("state_name_running");
		case Task::State_Finished:
			return TaskSchedulerImpl::tr("state_name_finished");
		default: assert(0);
	}
	return TaskSchedulerImpl::tr("invalid state name");
}

static void checkSerialPromotion(TaskInfo* pti, Task* serialSubtask) {
#ifdef _DEBUG
	Task* task = pti == NULL ? NULL : pti->task;
	if (task == NULL || !task->getFlags().testFlag(TaskFlag_SerialSubtasks)) {
		return;
	}
	bool before = true;
	foreach(Task* sub, task->getSubtasks()) {
		if (sub == serialSubtask) {
			before = false;
		} else if (before) {
			assert(sub->getState() == Task::State_Finished); 
		} else {
			assert(sub->getState() == Task::State_New); 
		}
	}
#else
    Q_UNUSED(pti); Q_UNUSED(serialSubtask);
#endif
}

static void checkFinishedState(TaskInfo* ti) {
#ifdef _DEBUG
	foreach(Task* sub, ti->task->getSubtasks()) {
		assert(sub->getState() == Task::State_Finished);
	}
	assert(ti->newSubtasks.empty());
	assert(ti->task->getSubtasks().size() == ti->numFinishedSubtasks);
	assert(ti->numRunningSubtasks  == 0);
#else
    Q_UNUSED(ti);
#
#endif
}

void TaskSchedulerImpl::promoteTask(TaskInfo* ti, Task::State newState) {
	Task* task = ti->task;
	assert(newState > task->getState());
	setTaskState(task, newState);

	TaskStateInfo& tsi = getTaskStateInfo(task);
	TaskTimeInfo& tti = getTaskTimeInfo(task);
	TaskInfo* pti = ti->parentTaskInfo;

	if (!tsi.hasErrors()) {
        log.trace(tr("promoting task {%1} to '%2'").arg(task->getTaskName()).arg(state2String(newState)));
	} else {
        log.trace(tr("promoting task {%1} to '%2', error '%3'").arg(task->getTaskName()).arg(state2String(newState)).arg(tsi.error));
	}

	checkSerialPromotion(pti, ti->task);
	switch(newState) {
		case Task::State_Prepared:
            if (ti->task->isTopLevelTask() && ti->task->isVerboseLogMode()) {
                log.info(tr("Starting {%1} task").arg(ti->task->getTaskName()));
            }
			break;
		case Task::State_Running:
			tti.startTime = QDateTime::currentDateTime();
			if (pti!=NULL) {
				pti->numRunningSubtasks++;
				if (pti->task->getState() < Task::State_Running) {
					assert(pti->task->getState() == Task::State_Prepared);
					promoteTask(pti, Task::State_Running);
				}
			}
			break;
		case Task::State_Finished:
			checkFinishedState(ti);
			tti.finishTime = QDateTime::currentDateTime();
            tsi.stateDesc.clear();
            getTaskStateInfo(ti->task).progress = 100;
			if (pti!=NULL) {
				pti->numRunningSubtasks--;
				assert(pti->numRunningSubtasks>=0);
				pti->numFinishedSubtasks++;
				assert(pti->numFinishedSubtasks <= pti->task->getSubtasks().size());
			}
            if (ti->task->isTopLevelTask()) {
                if (tsi.hasErrors()) {
                    log.error(tr("Task {%1} finished with error: %2").arg(task->getTaskName()).arg(tsi.error));
                } else if (tsi.cancelFlag) {
                    log.info(tr("Task {%1} canceled").arg(ti->task->getTaskName()));
                } else if (ti->task->isVerboseLogMode()) {
                    log.info(tr("Task {%1} finished").arg(ti->task->getTaskName()));
                }
            }
			break;
		default: assert(0);
	}
	emit si_stateChanged(task);

	updateTaskProgressAndDesc(ti);
}


void TaskSchedulerImpl::updateTaskProgressAndDesc(TaskInfo* ti) {
    Task* task = ti->task;
	TaskStateInfo& tsi = getTaskStateInfo(task);
	

    //update desc
    if (ti->task->useDescriptionFromSubtask()) {
        const QList<Task*>& subs = task->getSubtasks();
        if (!subs.isEmpty()) {
            Task* sub = subs.last();
            tsi.stateDesc = sub->getStateInfo().stateDesc;
        }
    } 

    if (tsi.stateDesc != ti->prevDesc) {
        ti->prevDesc = tsi.stateDesc;
        emit_taskDescriptionChanged(task);
    }

    //update progress
    if (task->getProgressManagementType() == Task::Progress_Manual) {
		int prevProgress = ti->prevProgress;
        if (tsi.progress != prevProgress) {
            ti->prevProgress = tsi.progress;
            emit_taskProgressChanged(task);
        }
	}  else {
		assert(task->getProgressManagementType() == Task::Progress_SubTasksBased);
        const QList<Task*>& subs = task->getSubtasks();
        int nsubs = subs.size();
        int newProgress = -1;
        if (nsubs > 0) {
            float sum = 0;
            float maxSum = 0.001F;
            foreach(Task* sub, subs) {
                float w = sub->getSubtaskProgressWeight();
                sum += sub->getProgress() * w;
                maxSum += w;
            }
            newProgress = qRound(sum / maxSum);
        } else if (task->isFinished()) {
            newProgress = 100;
        }
		if (tsi.progress != newProgress) {
            tsi.progress  = newProgress;
            emit_taskProgressChanged(task);
        }
	} 
}

void TaskSchedulerImpl::deleteTask(Task* task) {
	foreach(Task* sub, task->getSubtasks()) {
		//todo: check subtask autodelete ??
		deleteTask(sub);
	}
	log.trace(tr("deleting task %1").arg(task->getTaskName()));
	delete task;
}


void TaskSchedulerImpl::runTask(TaskInfo* ti) {
#ifdef _DEBUG
	assert(ti->task->getState() == Task::State_Running);
	assert(!ti->task->hasFlags(TaskFlag_RunAfterAllSubtasksFinished) || ti->numFinishedSubtasks == ti->task->getSubtasks().size());

    if (!ti->task->isCanceled()) {
        //check there are no errors with TaskFlag_SerialSubtasks_StopOnError tasks
        if (ti->parentTaskInfo!=NULL && ti->parentTaskInfo->task->hasFlags(TaskFlag_StopOnSubtaskError)) {
            foreach (Task* sub, ti->parentTaskInfo->task->getSubtasks()) {
                assert(sub == ti->task || !sub->hasErrors());
            }
        }
    }
#endif
	bool noRun = ti->task->isCanceled() || ti->task->getFlags().testFlag(TaskFlag_NoRun) || ti->task->hasErrors();
	if (noRun) {
		ti->runFinished = true;
	} else {
		ti->thread = new TaskThread(ti);
		ti->thread->start();
	}
}

void TaskThread::run() {
    setPriority(QThread::LowPriority);

	assert(ti->task->getState()== Task::State_Running);
	ti->task->run();
	assert(ti->task->getState()== Task::State_Running);
	ti->runFinished = true;
}

}//namespace
