/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_DNA_ALPHABET_H_
#define _GB2_DNA_ALPHABET_H_

#include "core_api.h"
#include "LRegion.h"

#include <QtCore/QBitArray>

namespace GB2 {

class GB2_COREAPI_EXPORT BaseDNAAlphabetIds : public QObject	{
public:
	static const QString RAW;
	static const QString NUCL_DNA_DEFAULT;
	static const QString NUCL_DNA_EXTENDED;
    static const QString NUCL_RNA_DEFAULT;
    static const QString NUCL_RNA_EXTENDED;
	static const QString AMINO_DEFAULT;
};



class GB2_COREAPI_EXPORT DNAAlphabet {

public:

    DNAAlphabet(const QString& id, const QString& name, DNAAlphabetType t, const QBitArray& map, Qt::CaseSensitivity caseMode, char defSym);

	const QBitArray& getMap() const {return map;}

	const QString& getName() const {return name;}

	const QString& getId() const {return id;}

	bool contains(char c) const {return map.at(c);} 

	DNAAlphabetType getType() const {return type;}

    bool isCaseSensitive() const {return caseMode == Qt::CaseSensitive;}

    char getDefaultSymbol() const {return defSym;}

    bool isRaw() const {return type == DNAAlphabet_RAW;}
    
    bool isNucleic() const {return type == DNAAlphabet_NUCL;}

    bool isAmino() const {return type == DNAAlphabet_AMINO;}
    
    // returns sorted array of all chars used in alphabet
    // forceBothCases == true : even case-insensitive alphabet will return both cases
    // forceBothCases == false: case-sensitivity in result depends on alphabets case-sensitivity
    QByteArray getAlphabetChars(bool forceBothCases = false) const;

    int getNumAlphabetChars() const {return numChars;}

    static DNAAlphabet* deriveCommonAlphabet(DNAAlphabet* al1, DNAAlphabet* al2);

private:
    QString		    id;
    QString		    name;
    DNAAlphabetType type;
    QBitArray	    map;
    Qt::CaseSensitivity     caseMode;
    char            defSym;
    int             numChars;
};

class GB2_COREAPI_EXPORT DNAAlphabetRegistry : public QObject {
public:
    virtual bool registerAlphabet(DNAAlphabet* a) = 0;

    virtual void unregisterAlphabet(DNAAlphabet* a) = 0;

    virtual DNAAlphabet* findById(const QString id) const = 0;

    virtual QList<DNAAlphabet*> getRegisteredAlphabets() const =  0;

    virtual DNAAlphabet* findAlphabet(const QByteArray& seq) const = 0;

    virtual QList<DNAAlphabet*> findAlphabets(const QByteArray& seq) const = 0;

    virtual QList<DNAAlphabet*> findAlphabets(const QByteArray& seq, const QList<LRegion>& regionsToProcess, bool onlyOne) const = 0;
};

}//namespace

#endif

