/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_DOCUMENT_MODEL_H_
#define _GB2_DOCUMENT_MODEL_H_

#include "core_api.h"
#include "StateLockableDataModel.h"

#include <gobjects/UnloadedObject.h>

#include <QtCore/QMimeData>
#include <QtCore/QPointer>
#include <QtScript>

namespace GB2 {

class TaskStateInfo;

class Document;
class GObject;
class DocumentFormat;
class IOAdapterFactory;
class IOAdapter;
class DocumentFormatConstraints;
class GHints;

class GB2_COREAPI_EXPORT DocumentFormat: public QObject {
	Q_OBJECT
public:
	static const QString CREATED_NOT_BY_UGENE;
    static const QString MERGED_SEQ_LOCK;
	
	enum DocObjectOp {
		DocObjectOp_Add,
		DocObjectOp_Remove
	};


	DocumentFormat(QObject* p) : QObject(p) {}

	virtual DocumentFormatId getFormatId() const = 0;

	virtual const QString& getFormatName() const = 0;

	virtual QStringList getSupportedDocumentFileExtensions() = 0;

	virtual Document* createNewDocument(IOAdapterFactory* io, const QString& url, const QVariantMap& hints = QVariantMap());

	virtual Document* loadExistingDocument(IOAdapterFactory* io, const QString& url, TaskStateInfo& ti, const QVariantMap& hints) = 0;
    
    /* io - opened IOAdapter.
     * if document format supports streaming reading
     * it should return a piece of a document.
     * otherwise, it will load all file from starting position ( default )
     */
    virtual Document* loadExistingDocument( IOAdapter* io, TaskStateInfo& ti, const QVariantMap& formatSettings );
	
    virtual void storeDocument(Document* d, TaskStateInfo& ts, IOAdapterFactory* io = NULL, const QString& newDocURL = QString::null);
    
    /* io - opened IOAdapter
     * so you can store many documents to this file
     */
    virtual void storeDocument( Document* d, TaskStateInfo& ts, IOAdapter* io );
    
    virtual bool isObjectOpSupported(const Document* d, DocObjectOp op, GObjectType t) const = 0;
    
	virtual bool checkConstraints(const DocumentFormatConstraints& c) const = 0;
    
    virtual void updateFormatSettings(Document* d) const {Q_UNUSED(d);}

    // any number (positive or negative). A higher priority is better
    virtual int getFormatDetectionPriority() const {return 0;}
};

class GB2_COREAPI_EXPORT DocumentFormatRegistry  : public QObject {
	Q_OBJECT
public:
	DocumentFormatRegistry(QObject* p = NULL) : QObject(p) {}

	virtual bool registerFormat(DocumentFormat* dfs) = 0;

	virtual bool unregisterFormat(DocumentFormat* dfs) = 0;

	virtual QList<DocumentFormatId> getRegisteredFormats() const = 0;

	virtual DocumentFormat* getFormatById(DocumentFormatId id) const = 0;

	virtual QList<DocumentFormatId> selectFormats(const DocumentFormatConstraints& c) const = 0;

    virtual QList<DocumentFormatId> sortByDetectPriority(const QList<DocumentFormatId>& ids) const = 0;
    
    virtual QList<DocumentFormat*> sortByDetectPriority(const QList<DocumentFormat*>& formats) const = 0;

signals:
    void si_documentFormatRegistered(DocumentFormat*);
    void si_documentFormatUnregistered(DocumentFormat*);
};

enum DocumentModLock {
    DocumentModLock_IO,
    DocumentModLock_USER,
    DocumentModLock_FORMAT_AS_CLASS,
    DocumentModLock_FORMAT_AS_INSTANCE,
    DocumentModLock_UNLOADED_STATE,
    DocumentModLock_NUM_LOCKS
};

class GB2_COREAPI_EXPORT Document : public  StateLockableTreeItem {
	Q_OBJECT
    Q_PROPERTY( QString name WRITE setName READ getName )
    Q_PROPERTY( QString url WRITE setURL READ getURL )

public:
	class Constraints {
	public:
        Constraints() : stateLocked(TriState_Unknown) {}
        TriState                stateLocked;
        QList<DocumentModLock>  notAllowedStateLocks; // if document contains one of these locks -> it's not matched
		QList<DocumentFormatId> formats;              // document format must be in list to match
		GObjectType             objectTypeToAdd;      // document must be ready to add objects of the specified type
	};


    //Creates document in unloaded state. Populates it with unloaded objects
	Document(DocumentFormat* _df, IOAdapterFactory* _io, const QString& _url, 
        const QList<UnloadedObjectInfo>& unloadedObjects = QList<UnloadedObjectInfo>(),
        const QVariantMap& hints = QVariantMap(), const QString& instanceModLockDesc = QString());

    //Creates document in loaded state. 
    Document(DocumentFormat* _df, IOAdapterFactory* _io, const QString& _url, 
                    const QList<GObject*>& objects, const QVariantMap& hints = QVariantMap(), 
                    const QString& instanceModLockDesc = QString());

	virtual ~Document();

	DocumentFormat* getDocumentFormat() const {return df;}
	
	DocumentFormatId getDocumentFormatId() const {return df->getFormatId();}
	
	IOAdapterFactory* getIOAdapterFactory() const {return io;}
	
	const QList<GObject*>& getObjects() const {return objects;}
	
	void addObject(GObject* ref);
	
	void removeObject(GObject* o);

	const QString& getName() const {return name;}
    
    void setName(const QString& newName);

	const QString& getURL() const {return url;}

	void setURL(const QString& newUrl);

	void makeClean();

	GObject* findGObjectByName(const QString& name) const;

	QList<GObject*> findGObjectByType(GObjectType t, UnloadedObjectFilter f = UOF_LoadedOnly) const;

	bool isLoaded() const {return modLocks[DocumentModLock_UNLOADED_STATE] == 0;}

    void setLoaded(bool v);

	void loadFrom(const Document* d);

    bool unload();

	Document* clone() const;

	bool checkConstraints(const Constraints& c) const;
    
    GHints* getGHints() const {return ctxState;}

    void setGHints(GHints* state);

    QVariantMap getGHintsMap() const;

    StateLock* getDocumentModLock(DocumentModLock type) const {return modLocks[type];}

    bool hasUserModLock() const {return modLocks[DocumentModLock_USER]!=NULL;}

    void setUserModLock(bool v);

    bool isModified() const { return isTreeItemModified(); }

    static void setupToEngine(QScriptEngine *engine);
private:
    static QScriptValue toScriptValue(QScriptEngine *engine, Document* const &in);
    static void fromScriptValue(const QScriptValue &object, Document* &out);
protected:
    void _removeObject(GObject* o, bool ignoreLocks = false);
    void _addObject(GObject* obj, bool ignoreLocks = false);
    void _addObjectToHierarchy(GObject* obj, bool ignoreLocks = false);

    void initModLocks(const QString& instanceModLockDesc, bool loaded);
    
    void checkUnloadedState() const;
    void checkLoadedState() const;
    void checkUniqueObjectNames() const;
    void addUnloadedObjects(const QList<UnloadedObjectInfo>& info);

    DocumentFormat* const		df;
	IOAdapterFactory* const		io;
	
	QString				url;
	QString				name; /* display name == short pathname, excluding the path */
	QList<GObject*>		objects;
    GHints*             ctxState;
    
    StateLock*          modLocks[DocumentModLock_NUM_LOCKS];

signals:
	void si_urlChanged();
	void si_nameChanged();
	void si_objectAdded(GObject* o);
	void si_objectRemoved(GObject* o);
	void si_loadedStateChanged();
};

class DocumentFormatConstraints {
public:
    DocumentFormatConstraints() : mustSupportWrite(false), checkRawData(false), supportsStreamingRead(false){}

    QList<GObjectType>  supportedObjectTypes;
    bool				mustSupportWrite;
    bool				checkRawData;
    QByteArray			rawData;
    bool                supportsStreamingRead;
};

//TODO: decide if to use filters or constraints. May be it worth to remove Document::Constraints at all..

class GB2_COREAPI_EXPORT DocumentFilter {
public:
    virtual ~DocumentFilter(){};
    virtual bool matches(Document* doc) const = 0;
};

class GB2_COREAPI_EXPORT DocumentConstraintsFilter : public DocumentFilter {
public:
    DocumentConstraintsFilter(const Document::Constraints& _c) : constraints(_c){}
    
    virtual bool matches(Document* doc) const {
        return doc->checkConstraints(constraints);
    }

protected:
    Document::Constraints constraints;
};

class GB2_COREAPI_EXPORT DocumentMimeData : public QMimeData {
    Q_OBJECT
public:
    static const QString MIME_TYPE;
    DocumentMimeData(Document* obj) : objPtr(obj){};
    QPointer<Document> objPtr;
};

} //namespace
Q_DECLARE_METATYPE(GB2::Document*)
#endif


