/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_ASN_FORMAT_H_
#define _GB2_ASN_FORMAT_H_

#include <core_api/DocumentFormats.h>
#include <core_api/DocumentModel.h>
#include <datatype/BioStruct3D.h>
#include <QtCore/QSharedDataPointer>
#include <QtCore/QStack>

#include "StdResidueDictionary.h"

namespace GB2 {

class IOAdapter;
class BioStruct3D;
class MoleculeData;
class ResidueData;
class AnnotationTableObject;
class DNASequenceObject;
class AtomData;
typedef QSharedDataPointer<AtomData> SharedAtom;
typedef QHash<int, SharedAtom> AtomCoordSet;


enum AsnElementKind {
    ASN_NO_KIND, ASN_SEQ, ASN_VALUE, ASN_ROOT
};

class GB2_COREAPI_EXPORT AsnNode : public QObject
{
    Q_OBJECT

public:
    AsnNode();
    AsnNode(const QByteArray& _name, AsnElementKind _kind);
    ~AsnNode();
    QByteArray name, value;
    AsnElementKind kind;
    QList<AsnNode*> children;
    AsnNode* findChildByName(const QByteArray& name);
    AsnNode* getChildById(int id);

private:
    static void deleteChildren(AsnNode* node);
};

typedef QList<AsnNode*> AsnNodeList;

class GB2_COREAPI_EXPORT  ASNFormat : public DocumentFormat 
{
    Q_OBJECT
public:
    ASNFormat(QObject* p);
    ~ASNFormat();
    virtual DocumentFormatId getFormatId() const {return BaseDocumentFormats::PLAIN_ASN;}
    virtual const QString& getFormatName() const {return formatName;}
    virtual QStringList getSupportedDocumentFileExtensions();
    virtual Document* loadExistingDocument(IOAdapterFactory* iof, const QString& url, TaskStateInfo& ti, const QVariantMap& fs);
    virtual bool isObjectOpSupported(const Document* d , DocumentFormat::DocObjectOp op, GObjectType t) const;
    virtual bool isDataFormatSupported(const char* data, int size) const;
    virtual bool checkConstraints(const DocumentFormatConstraints& c) const;

    class AsnParser 
    {
        static const char* filetypeTag;

        // Parsing state
        struct ParseState {
            QByteArray parentName;
            bool atEnd;
            int numOpenedTags;
        };

        // Data
        IOAdapter *io;        
        TaskStateInfo& ts;
        QByteArray buffer;
        char prev;
        ParseState curState;
        QByteArray curElementName, curElementValue;
        AsnElementKind curElementKind;
        bool validFile;
        bool haveErrors;
        bool insideRoot;
        bool fileAtEnd;
        QStack<ParseState> states;

        // Core
        bool readRootElement();
        bool readNextElement();
        void processValue();
        void parseNextElement(AsnNode* node);
        void saveState();
        void restoreState();
        void dbgPrintCurrentState();
        void initState(const QByteArray& parentName);

        // Utility
        static bool isQuoted(const QByteArray& str);
        static void removeQuotes(QByteArray& str);
        static void dbgPrintAsnTree(const AsnNode* rootElem, int level = 0);


    public:
        AsnParser(IOAdapter* _io, TaskStateInfo& _ts) : 
          io(_io), ts(_ts) {}
        ~AsnParser() {}
        AsnNode* loadAsnTree();

    };

    static QString getAsnNodeTypeName(const AsnNode* node);
    static AsnNode* findFirstNodeByName(AsnNode* rootElem, const QString& nodeName);
    static AsnNodeList findNodesByName(AsnNode* root, const QString& nodeName, AsnNodeList& lst );

private:

    QString formatName;
    const StdResidueDictionary *standardDictionary, *localDictionary;
    QHash<quint64, StdResidue> stdResidueCache;
    QMap<quint64, AtomCoordSet> atomSetCache;

    struct AsnBaseException {
        QString msg;
        AsnBaseException( const QString& what ) : msg( what ){}
    };

    struct AsnReadError : AsnBaseException 
    {
        AsnReadError() : AsnBaseException(ASNFormat::tr("read error occurred")) { }
    };

    struct AsnParserError : public AsnBaseException {
        AsnParserError( const QString& what ) : AsnBaseException( what ){}
    };

    struct AsnBioStructError : public AsnBaseException {
        AsnBioStructError( const QString& what ) : AsnBaseException(QString(ASNFormat::tr("biostruct3d obj loading error: %1")).arg(what) ){}
    };

    void loadBioStructFromAsnTree(AsnNode* rootNode, BioStruct3D& struc, TaskStateInfo& info);
    void loadBioStructModels(QList<AsnNode*> models, BioStruct3D& struc);
    void loadBioStructGraph(AsnNode* graphNode, BioStruct3D& struc);
    void loadBioStructPdbId( AsnNode* rootNode, BioStruct3D& struc);
    void loadBioStructSecondaryStruct( AsnNode* rootNode, BioStruct3D& struc );
    void loadBioStructFeature( AsnNode* featureNode, BioStruct3D& struc );
    void loadModelCoordsFromNode( AsnNode* node, AtomCoordSet& coordSet, QMap<int, Molecule3DModel>& molModels, const BioStruct3D& struc);
    void loadMoleculeFromNode(AsnNode* moleculeNode, MoleculeData* molecule);
    void loadIntraResidueBonds(BioStruct3D& struc);
    const StdResidue loadResidueFromNode(AsnNode* resNode, ResidueData* residue);

};



}//namespace

#endif //_GB2_ASN_FORMAT_H_
