/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "StockholmFormat.h"

#include "DocumentFormatUtils.h"

#include <core_api/Task.h>
#include <core_api/IOAdapter.h>
#include <gobjects/GObjectTypes.h>
#include <gobjects/MAlignmentObject.h>
#include <util_text/TextUtils.h>
#include <core_api/DNAAlphabet.h>
#include <util_gui/GUIUtils.h>
#include <datatype/MAlignmentInfo.h>

#include <memory>
#include <QFileInfo>

/* TRANSLATOR GB2::StockholmFormat */
namespace GB2 {
const QByteArray StockholmFormat::FILE_ANNOTATION_ID = "#=GF ID";
const QByteArray StockholmFormat::FILE_ANNOTATION_AC = "#=GF AC";
const QByteArray StockholmFormat::FILE_ANNOTATION_DE = "#=GF DE";
const QByteArray StockholmFormat::FILE_ANNOTATION_GA = "#=GF GA";
const QByteArray StockholmFormat::FILE_ANNOTATION_NC = "#=GF NC";
const QByteArray StockholmFormat::FILE_ANNOTATION_TC = "#=GF TC";

const QByteArray StockholmFormat::UNI_ANNOTATION_MARK = "# UNIMARK";

const QByteArray StockholmFormat::COLUMN_ANNOTATION_SS_CONS = "#=GC SS_cons";
const QByteArray StockholmFormat::COLUMN_ANNOTATION_RF      = "#=GC RF";

StockholmFormat::ReadError::ReadError() : StockholmBaseException( Translations::errorReadingFile("") ){}

StockholmFormat::WriteError::WriteError() : StockholmBaseException( Translations::errorWritingFile("") ){}
} // GB2

const int  BUF_SZ = 1024;
const int  SMALL_BUF_SZ = 128;
const char TERM_SYM = '\0';
const int  NO_BYTES = 0;
const char NEW_LINE = '\n';

const char* HEADER = "# STOCKHOLM 1.0\n\n";
const char* HEADER_MIN = "# STOCKHOLM 1.";
const int HEADER_SZ_MIN = 15;
const char* EOF_STR = "//";

const char COMMENT_OR_MARKUP_LINE = '#';
const char* EMPTY_STR = "";

const int WRITE_BLOCK_LENGTH = 50;

using namespace GB2;

//other not supported
enum AnnotationTag {
	NO_TAG = -1,
	ID,
    AC,
    DE,
    SS_CONS,
    RF,
    GA,
    NC,
    TC
};
//other types not supported
enum AnnotationType {
	FILE_ANNOTATION,
	COLUMN_ANNOTATION,
    UNI_ANNOTATION
};

struct Annotation {
	AnnotationType type;
	AnnotationTag tag;
	QString val;
	Annotation( AnnotationType ty, AnnotationTag t, QString v ) { type = ty, tag = t; val = v; }
	virtual ~Annotation(){}
};
//#=GF annotations
struct FileAnnotation : public Annotation {
	FileAnnotation( AnnotationTag t, QString v ): Annotation( FILE_ANNOTATION, t, v ){}
};
//unipro ugene annotations
struct UniAnnotation : public Annotation {
	UniAnnotation( AnnotationTag t, QString v ): Annotation( UNI_ANNOTATION, t, v ){}
};
//#=GC annotations
struct ColumnAnnotation : public Annotation {
    ColumnAnnotation( AnnotationTag t, QString v ): Annotation( COLUMN_ANNOTATION, t, v ){}
};

static Annotation* findAnnotation( const QList< Annotation* >& annList, AnnotationType t, AnnotationTag tag );

//you should put annotations here after creation
struct AnnotationBank {
    QList<Annotation*> ann_list;
    
    void addAnnotation( Annotation* ann ) {
        if( NULL == ann ) {
            return;
        }
        if( COLUMN_ANNOTATION == ann->type ) { /* Column annotations usually written as blocks with seqs */
            assert( SS_CONS == ann->tag || RF == ann->tag );
            Annotation* nAnn = findAnnotation( ann_list, COLUMN_ANNOTATION, ann->tag );
            if( NULL != nAnn ) {
                nAnn->val.append( ann->val );
                delete ann;
                return;
            }
        }
        ann_list.append( ann );
    }
    ~AnnotationBank() {
        qDeleteAll( ann_list );
    }
}; // AnnotationBank

static Annotation* findAnnotation( const QList< Annotation* >& annList, AnnotationType t, AnnotationTag tag ) {
    Annotation* ret = NULL;
    foreach( Annotation* a, annList ) {
        assert( NULL != a );
        if( a->type == t && a->tag == tag ) {
            ret = a;
            break;
        }
    }
    return ret;
}

static Annotation* getAnnotation( const QByteArray& l ) {
	QByteArray line = l.trimmed();
	
    if ( line.startsWith( StockholmFormat::FILE_ANNOTATION_ID ) ) {
		QByteArray val = line.mid( StockholmFormat::FILE_ANNOTATION_ID.size() ).trimmed();
		return ( val.size() )? new FileAnnotation( ID , val ) : NULL;
    }
    else if( line.startsWith( StockholmFormat::FILE_ANNOTATION_AC ) ) {
        QByteArray val = line.mid( StockholmFormat::FILE_ANNOTATION_AC.size() ).trimmed();
        return ( val.size() )? new FileAnnotation( AC, val ) : NULL;
    }
    else if( line.startsWith( StockholmFormat::FILE_ANNOTATION_DE ) ) {
        QByteArray val = line.mid( StockholmFormat::FILE_ANNOTATION_DE.size() ).trimmed();
        return ( val.size() )? new FileAnnotation( DE, val ) : NULL;
    }
    else if( line.startsWith( StockholmFormat::FILE_ANNOTATION_GA ) ) {
        QByteArray val = line.mid( StockholmFormat::FILE_ANNOTATION_GA.size() ).trimmed();
        return ( val.size() )? new FileAnnotation( GA, val ) : NULL;
    }
    else if( line.startsWith( StockholmFormat::FILE_ANNOTATION_NC ) ) {
        QByteArray val = line.mid( StockholmFormat::FILE_ANNOTATION_NC.size() ).trimmed();
        return ( val.size() )? new FileAnnotation( NC, val ) : NULL;
    }
    else if( line.startsWith( StockholmFormat::FILE_ANNOTATION_TC ) ) {
        QByteArray val = line.mid( StockholmFormat::FILE_ANNOTATION_TC.size() ).trimmed();
        return ( val.size() )? new FileAnnotation( TC, val ) : NULL;
    }
    else if ( StockholmFormat::UNI_ANNOTATION_MARK == line ) {
		return new UniAnnotation( NO_TAG, line );
    }
    else if( line.startsWith( StockholmFormat::COLUMN_ANNOTATION_SS_CONS ) ) {
        QByteArray val = line.mid( StockholmFormat::COLUMN_ANNOTATION_SS_CONS.size() ).trimmed();
        return ( val.size() )? new ColumnAnnotation( SS_CONS, val ) : NULL;
    }
    else if( line.startsWith( StockholmFormat::COLUMN_ANNOTATION_RF ) ) {
        QByteArray val = line.mid( StockholmFormat::COLUMN_ANNOTATION_RF.size() ).trimmed();
        return ( val.size() )? new ColumnAnnotation( RF, val ) : NULL;
    }
	return NULL;
}

static QString getMsaName( const AnnotationBank& ann_bank ) {
	foreach( Annotation* ann, ann_bank.ann_list ) {
		assert( NULL != ann );
		if ( FILE_ANNOTATION == ann->type && ID == ann->tag ) {
			return ann->val;
		}
	}
	return QString::null;
}

static bool isUniFile( const AnnotationBank& ann_bank ) {
	foreach( Annotation* ann, ann_bank.ann_list ) {
		assert( NULL != ann );
		if ( UNI_ANNOTATION == ann->type && StockholmFormat::UNI_ANNOTATION_MARK == ann->val ) {
			return true;
		}
	}
	return false;
}

static QString getAnnotationName( Annotation* ann ) {
    assert( NULL != ann );
    
    AnnotationType t = ann->type;
    switch( t ) {
    case UNI_ANNOTATION:
        return QString( StockholmFormat::UNI_ANNOTATION_MARK );
    case FILE_ANNOTATION:
        {
            AnnotationTag tag = ann->tag;
            switch( tag ) {
            case ID:
                return QString( StockholmFormat::FILE_ANNOTATION_ID );
            case AC:
                return QString( StockholmFormat::FILE_ANNOTATION_AC );
            case DE:
                return QString( StockholmFormat::FILE_ANNOTATION_DE );
            case GA:
                return QString( StockholmFormat::FILE_ANNOTATION_GA );
            case NC:
                return QString( StockholmFormat::FILE_ANNOTATION_NC );
            case TC:
                return QString( StockholmFormat::FILE_ANNOTATION_TC );
            default:
                assert( false );
            }
        }
    case COLUMN_ANNOTATION:
        {
            AnnotationTag tag = ann->tag;
            switch( tag ) {
            case SS_CONS:
                return QString( StockholmFormat::COLUMN_ANNOTATION_SS_CONS );
            case RF:
                return QString( StockholmFormat::COLUMN_ANNOTATION_RF );
            default:
                assert( false );
            }
        }
    default:
        assert( false );
    }
    return QString();
}

static QHash< QString, QString > getAnnotationMap( const AnnotationBank& annBank ) {
    QHash< QString, QString > ret;
    foreach( Annotation* ann, annBank.ann_list ) {
        assert( NULL != ann );
        QString annName = getAnnotationName( ann );
        ret[annName] = QString( ann->val );
    }
    return ret;
}

template<class T>
static void checkValThrowException( bool expected, T val1, T val2,  const StockholmFormat::StockholmBaseException& e ) {
	if ( expected != ( val1 == val2 ) ) {
		throw e;
	}
}

static bool checkHeader( const char* data, int sz ) {
    assert( NULL != data && 0 <= sz );
	
    if ( HEADER_SZ_MIN > sz ) {
        return false;
    }
    return QByteArray( data, sz ).startsWith( HEADER_MIN );
}

//returns true if the line was skipped
static bool skipCommentOrMarkup( IOAdapter* io, AnnotationBank& ann_bank ) {
    assert( NULL != io );
	
    QByteArray buf( BUF_SZ, TERM_SYM );
    QByteArray line;
	bool term_there = false;
    int ret = io->readUntil( buf.data(), BUF_SZ, TextUtils::LINE_BREAKS, IOAdapter::Term_Exclude, &term_there );
	
	checkValThrowException<int>( false, -1, ret, StockholmFormat::ReadError() );
	if ( COMMENT_OR_MARKUP_LINE == buf[0] ) {
        line.append( QByteArray( buf.data(), ret ) );
		while ( !term_there ) {
            ret = io->readUntil( buf.data(), BUF_SZ, TextUtils::LINE_BREAKS, IOAdapter::Term_Exclude, &term_there );
			checkValThrowException<int>( false, -1, ret,StockholmFormat::ReadError() );
			if ( NO_BYTES == ret ) {
                break;
            }
			line.append( QByteArray( buf.data(), ret ) );
		}
        ann_bank.addAnnotation( getAnnotation( line ) );
		return true;
    }
    io->skip( -ret );
    return false;
}

static void skipBlankLines( IOAdapter* io, QByteArray* lines = NULL ) {
    assert( NULL != io );
	
    char c = 0;
    bool work = true;
    while ( work ) {
        int ret = io->readBlock( &c, 1 );
		checkValThrowException<int>( false, -1, ret, StockholmFormat::ReadError() );
		if ( NO_BYTES == ret ) {
			return;
		}
		work = TextUtils::LINE_BREAKS[(uchar)c] || TextUtils::WHITES[(uchar)c];
		if ( lines && TextUtils::LINE_BREAKS[(uchar)c] ) {
            lines->append( c );
        }
    }
    io->skip( -1 );
}

//skips all that it can
static void skipMany( IOAdapter* io, AnnotationBank& ann_bank ) {
    assert( NULL != io );
	
    char c = 0;
    while ( 1 ) {
        bool ret = io->getChar( &c );
		checkValThrowException<bool>( false, false, ret, StockholmFormat::ReadError() );
		if ( COMMENT_OR_MARKUP_LINE == c ) {
            io->skip( -1 );
            skipCommentOrMarkup( io, ann_bank );
            continue;
        }
		else if ( TextUtils::LINE_BREAKS[(uchar)c] || TextUtils::WHITES[(uchar)c] ) {
            skipBlankLines( io );
            continue;
        }
        io->skip( -1 );
        break;
    }
}

static bool eofMsa( IOAdapter* io ) {
    assert( NULL != io );
    
    QByteArray buf( SMALL_BUF_SZ, TERM_SYM );
    int ret = io->readUntil( buf.data(), SMALL_BUF_SZ, TextUtils::LINE_BREAKS, IOAdapter::Term_Include );
	checkValThrowException( false, -1, ret, StockholmFormat::ReadError() );
	io->skip( -ret );
    return EOF_STR == QByteArray( buf.data(), ret ).trimmed();
}

static void readEofMsa( IOAdapter* io ) {
    assert( eofMsa( io ) );
    QByteArray buf( SMALL_BUF_SZ, TERM_SYM );
    int ret = io->readUntil( buf.data(), SMALL_BUF_SZ, TextUtils::LINE_BREAKS, IOAdapter::Term_Include );		
	checkValThrowException( false, -1, ret, StockholmFormat::ReadError() );
}

//returns end of sequence name in line
static int getLine( IOAdapter* io, QByteArray& to ) {
    assert( NULL != io );

    QByteArray buf( BUF_SZ, TERM_SYM );
    bool there = false;

    while ( !there ) {
        int ret = io->readUntil( buf.data(), BUF_SZ, TextUtils::LINE_BREAKS, IOAdapter::Term_Exclude, &there );
		checkValThrowException<int>( false, -1, ret, StockholmFormat::ReadError() );
		if( NO_BYTES == ret ) {
            break;
        }
        to.append( QByteArray( buf.data(), ret ) );
    }
	
    int sz = to.size();
    int i = 0;
    for ( i = 0; i < sz; ++i ) {
        if ( TextUtils::WHITES[(uchar)to[i]] ) {
            break;
        }
    }
    return i;
}

static bool blockEnded( IOAdapter* io ) {
    assert( NULL != io );
	
    QByteArray lines;
    skipBlankLines( io, &lines );
    if ( eofMsa( io ) ) {
        return true;
    }
	int nl_count = 0;
	int lines_sz = lines.size();
	
	for( int i = 0; i < lines_sz; ++i ) {
		nl_count = ( NEW_LINE == lines[i] )? nl_count + 1: nl_count;
	}
	return 1 < nl_count;
}

//calls after reading a block
static bool checkSeqLength( const MAlignment& msa ) {
    int sz = msa.alignedSeqs.size();
    int seq_length = ( sz )? msa.alignedSeqs[0].sequence.size(): -1;
    bool ret = ( sz )? true: false;
	
    for ( int i = 0; i < sz; ++i ) {
        if ( !( ret = ret && ( seq_length == msa.alignedSeqs[i].sequence.size() ) ) ) {
            break;
        }
    }
    return ret;
}

static bool nameWasBefore( const MAlignment& msa, const QString& name ) {
    int seq_num = msa.getNumSequences();
    bool ret = false;

    for( int i = 0; i < seq_num; ++i ) {
        if ( name == msa.alignedSeqs[i].name ) {
            ret = true;
            break;
        }
    }
    return ret;
}

static void changeGaps( QByteArray& seq ) {
    int seq_sz = seq.size();
    for( int i = 0; i < seq_sz; ++i ) {
        if ( '.' == seq[i] ) {
            seq[i] = '-';
        }
    }
}

// returns true if operation was not canceled
static bool loadOneMsa( IOAdapter* io, TaskStateInfo& tsi, MAlignment& msa, AnnotationBank& ann_bank ) {
	assert( NULL != io );
	
	QByteArray buf( BUF_SZ, TERM_SYM );
	int ret = 0;
	
	//skip header
	skipBlankLines( io );
	ret = io->readUntil( buf.data(), BUF_SZ, TextUtils::LINE_BREAKS, IOAdapter::Term_Exclude );
	checkValThrowException<int>( false, -1, ret, StockholmFormat::ReadError() );
	if ( !checkHeader( buf.data(), ret ) ) {
		throw StockholmFormat::BadFileData( StockholmFormat::tr( "invalid file: bad header line" ) );
	}
	//read blocks
	bool firstBlock = true;
	while ( 1 ) {
        if( tsi.cancelFlag ) {
            return false;
        }
        skipMany( io, ann_bank );
		if ( eofMsa( io ) ) {
			break;
		}
		
		bool hasSeqs = true;
		int seq_ind = 0;
		while ( hasSeqs ) {
			QByteArray line;
			int name_end = getLine( io, line );
			QByteArray name = line.left( name_end );
			QByteArray seq  = line.mid( name_end + 1 ).trimmed();
			
			if ( name.startsWith( COMMENT_OR_MARKUP_LINE ) ) {
				ann_bank.addAnnotation( getAnnotation( line ) );
				hasSeqs = !blockEnded( io );
				tsi.progress = io->getProgress();
				continue;
			}
			changeGaps( seq );
			if ( firstBlock ) {
				if ( EMPTY_STR == name ) {
					throw StockholmFormat::BadFileData( StockholmFormat::tr( "invalid file: empty sequence name" ) );
				}
				if ( nameWasBefore( msa, QString( name.data() ) ) ) {
					throw StockholmFormat::BadFileData( StockholmFormat::tr( "invalid file: equal sequence names in one block" ) );
				}
				msa.alignedSeqs.append( MAlignmentItem( name.data(), seq ) );
			}
			else {
				MAlignmentItem& item = msa.alignedSeqs[seq_ind];
				if( name != item.name ) {
					throw StockholmFormat::BadFileData( StockholmFormat::tr( "invalid file: sequence names are not equal in blocks" ) );
				}
				item.sequence.append( seq );
			}
			seq_ind++;
			hasSeqs = !blockEnded( io );
			tsi.progress = io->getProgress();
		}
		firstBlock = false;
		//check sequence length after every block
		if ( !checkSeqLength( msa ) ) {
			throw StockholmFormat::BadFileData( StockholmFormat::tr( "invalid file: sequences in block are not of equal size" ) );
		}
	}// while( 1 )
    readEofMsa( io );
    skipBlankLines( io );
    
	if ( !msa.getNumSequences() ) {
		throw StockholmFormat::BadFileData( StockholmFormat::tr( "invalid file: empty sequence alignment" ) );
	}
    DocumentFormatUtils::assignAlphabet( msa );
	if ( NULL == msa.alphabet ) {
		throw StockholmFormat::BadFileData( StockholmFormat::tr( "invalid file: unknown alphabet" ) );
	}
    return true;
}

static void setMsaInfoCutoffs( QVariantMap& info, const QString& string, MAlignmentInfo::Cutoffs cof1, 
                               MAlignmentInfo::Cutoffs cof2 ) {
    QByteArray str = string.toAscii();
    QTextStream txtStream( str );
    float val1 = .0f;
    float val2 = .0f;
    txtStream >> val1 >> val2;
    MAlignmentInfo::setCutoff( info, cof1, val1 );
    MAlignmentInfo::setCutoff( info, cof2, val2 );
}

static void setMsaInfo( const QHash< QString, QString>& annMap, MAlignment& ma ) {
    QVariantMap& info = ma.info;
    
    if( annMap.contains( StockholmFormat::FILE_ANNOTATION_AC ) ) {
        MAlignmentInfo::setAccession( info, annMap[StockholmFormat::FILE_ANNOTATION_AC] );
    }
    if( annMap.contains( StockholmFormat::FILE_ANNOTATION_DE ) ) {
        MAlignmentInfo::setDescription( info, annMap[StockholmFormat::FILE_ANNOTATION_DE] );
    }
    if( annMap.contains( StockholmFormat::COLUMN_ANNOTATION_SS_CONS ) ) {
        MAlignmentInfo::setSSConsensus( info, annMap[StockholmFormat::COLUMN_ANNOTATION_SS_CONS] );
    }
    if( annMap.contains( StockholmFormat::COLUMN_ANNOTATION_RF ) ) {
        MAlignmentInfo::setReferenceLine( info, annMap[StockholmFormat::COLUMN_ANNOTATION_RF] );
    }
    if( annMap.contains( StockholmFormat::FILE_ANNOTATION_GA ) ) {
        setMsaInfoCutoffs( info, annMap[StockholmFormat::FILE_ANNOTATION_GA], MAlignmentInfo::CUTOFF_GA1,
                                                                              MAlignmentInfo::CUTOFF_GA2 );
    }
    if( annMap.contains( StockholmFormat::FILE_ANNOTATION_NC ) ) {
        setMsaInfoCutoffs( info, annMap[StockholmFormat::FILE_ANNOTATION_NC], MAlignmentInfo::CUTOFF_NC1,
                                                                              MAlignmentInfo::CUTOFF_NC2 );
    }
    if( annMap.contains( StockholmFormat::FILE_ANNOTATION_TC ) ) {
        setMsaInfoCutoffs( info, annMap[StockholmFormat::FILE_ANNOTATION_TC], MAlignmentInfo::CUTOFF_TC1,
                                                                              MAlignmentInfo::CUTOFF_TC2 );
    }
}

static QString getFilename( IOAdapter* io ) {
    assert( NULL != io );
    QString url = io->getUrl();
    assert( !url.isEmpty() );
    return QFileInfo( url ).baseName();
}

static void load( IOAdapter* io, QList<GObject*>& l, TaskStateInfo& tsi, bool& uni_file, bool onlyOne ) {
	assert( NULL != io );
	
	QStringList names_list;
	QString filename = getFilename( io );
    while( !io->isEof() ) {
		MAlignment msa;
		AnnotationBank ann_bank;
		QString name;
	    bool notCanceled = true;
        QHash< QString, QString > annMap;
        
		notCanceled = loadOneMsa( io, tsi, msa, ann_bank );
        if( !notCanceled ) {
            break;
        }
		uni_file = uni_file || isUniFile( ann_bank );
		
        name = getMsaName( ann_bank );
		name = ( QString::null == name || names_list.contains( name ) )?
			filename + "_" + QString::number( l.size() ): name;
		names_list.append( name );
        msa.setName( name );
        
        annMap = getAnnotationMap( ann_bank );
        setMsaInfo( annMap, msa );
        MAlignmentObject* obj = new MAlignmentObject(msa);
        obj->setIndexInfo(annMap);
		l.append( obj );
        if( onlyOne ) {
            break;
        }
    }
}

static int getMaxNameLen( const MAlignment& msa ) {
    assert( msa.getNumSequences() );
    int sz = msa.getNumSequences();
    int max_len = msa.alignedSeqs[0].name.size();
	
    for( int i = 0; i < sz; ++i ) {
        int name_len =  msa.alignedSeqs[i].name.size();
        max_len = ( max_len < name_len )? name_len: max_len;
    }
    return max_len;
}
//returns a gap between name and sequence in block	
static QByteArray getNameSeqGap( int diff ) {
    assert( 0 <= diff );
    QByteArray ret = "    ";
    for( int i = 0; i < diff; ++i ) {
        ret.append( " " );
    }
    return ret;
}

static void save( IOAdapter* io, const MAlignment& msa, const QString& name ) {
    assert( NULL != io );
    assert( checkSeqLength( msa ) && msa.getNumSequences() );
    int ret = 0;
	
    QByteArray header( HEADER );
    ret = io->writeBlock( header );
	checkValThrowException<int>( true, header.size(), ret, StockholmFormat::WriteError() );
	QByteArray unimark = StockholmFormat::UNI_ANNOTATION_MARK + "\n\n";
	ret = io->writeBlock( unimark );
	checkValThrowException<int>( true, unimark.size(), ret, StockholmFormat::WriteError() );
	QByteArray idAnn = StockholmFormat::FILE_ANNOTATION_ID + " " + name.toAscii() + "\n\n";
    ret = io->writeBlock( idAnn );
    checkValThrowException<int>( true, idAnn.size(), ret, StockholmFormat::WriteError() );
    
	//write sequences
    int name_max_len = getMaxNameLen( msa );
    int seq_len = msa.getLength();
    int cur_seq_pos = 0;
    while ( 0 < seq_len ) {
        int block_len = ( WRITE_BLOCK_LENGTH >= seq_len )? seq_len: WRITE_BLOCK_LENGTH;
		
        //write block
        int seq_num = msa.getNumSequences();
        for( int i = 0; i < seq_num; ++i ) {
            QByteArray name = msa.alignedSeqs[i].name.toAscii();
            TextUtils::replace(name.data(), name.length(), TextUtils::WHITES, '_');
            name += getNameSeqGap( name_max_len - msa.alignedSeqs[i].name.size() );
            ret = io->writeBlock( name );
			checkValThrowException<int>( true, name.size(), ret, StockholmFormat::WriteError() );
			QByteArray seq = msa.alignedSeqs[i].sequence.mid( cur_seq_pos, block_len ) + NEW_LINE;
            ret = io->writeBlock( seq );
			checkValThrowException<int>( true, seq.size(), ret, StockholmFormat::WriteError() );
		}
        ret = io->writeBlock( QByteArray( "\n\n" ) );
		checkValThrowException<int>( true, 2, ret, StockholmFormat::WriteError() );
		seq_len -= block_len;
        cur_seq_pos += block_len;
    }
    //write eof
    ret = io->writeBlock( QByteArray( "//\n" ) );
	checkValThrowException<int>( true, 3, ret, StockholmFormat::WriteError() );
}

namespace GB2 {
StockholmFormat::StockholmFormat( QObject *obj ) : DocumentFormat( obj ) {
    format_name = tr( "Stockholm" );
}

QStringList StockholmFormat::getSupportedDocumentFileExtensions() {
    QStringList list;
	
    list << "sto";
    return list;
}

Document* StockholmFormat::loadExistingDocument( IOAdapterFactory *io_factory, const QString &url, TaskStateInfo &tsi, const QVariantMap &fs ) {
    std::auto_ptr<IOAdapter> io( io_factory->createIOAdapter() );
	
    if( !io->open( url, IOAdapterMode_Read ) ) {
        tsi.setError(Translations::errorOpeningFileRead( url ));
        return NULL;
    }

    QList<GObject*> obj_list;
    try {
        bool uni_file = false;
		QString write_lock_reason;
		
		load( io.get(), obj_list, tsi, uni_file, false );
        io->close();
		if ( !uni_file ) {
			write_lock_reason = DocumentFormat::CREATED_NOT_BY_UGENE;
		}
		return new Document( this, io_factory, url, obj_list, fs, write_lock_reason );
    }
    catch ( const StockholmBaseException& e ) {
        tsi.setError(e.msg);
    }
    catch ( ... ) {
        tsi.setError(tr( "unknown error occurred" ));
    }
    foreach( GObject* obj, obj_list ) {
        delete obj;
    }
    return NULL;
}

Document* StockholmFormat::loadExistingDocument( IOAdapter* io, TaskStateInfo& ti, const QVariantMap& fs ) {
    if( NULL == io || !io->isOpen() ) {
        ti.setError(Translations::badArgument("IO adapter"));
        return NULL;
    }
    
    QList<GObject*> obj_list;
    try {
        bool uniFile = false;
        QString write_lock_reason;
        load( io, obj_list, ti, uniFile, true );
        if ( !uniFile ) {
            write_lock_reason = DocumentFormat::CREATED_NOT_BY_UGENE;
        }
        return new Document( this, io->getFactory(), io->getUrl(), obj_list, fs, write_lock_reason );
    }
    catch ( const StockholmBaseException& e ) {
        ti.setError(e.msg);
    }
    catch ( ... ) {
        ti.setError(tr( "unknown error occurred" ));
    }
    return NULL;
}

void StockholmFormat::storeDocument( Document* doc, TaskStateInfo& tsi, IOAdapterFactory* io_factory, const QString& newDocURL) {
    assert( NULL != doc );
    io_factory = ( io_factory )? io_factory: doc->getIOAdapterFactory();
    std::auto_ptr<IOAdapter> io( io_factory->createIOAdapter() );
    QString url = ( newDocURL.isEmpty() )? doc->getURL(): newDocURL;
	
    if ( !io->open( url, IOAdapterMode_Write ) ) {
        tsi.setError(Translations::errorOpeningFileWrite(url));
        return;
    }
    try {
        foreach( GObject* p_obj, doc->getObjects() ) {
			const MAlignmentObject* aln_obj = qobject_cast<const MAlignmentObject*>( p_obj );
			assert( NULL != aln_obj );
			save( io.get(), aln_obj->getMAlignment(), aln_obj->getGObjectName() );
            if( tsi.cancelFlag ) {
                return;
            }
        }
		io->close();
    }
    catch ( const StockholmBaseException& ex ) {
        tsi.setError(ex.msg);
    }
    catch ( ... ) {
        tsi.setError(tr( "unknown error occurred" ));
    }
}

void StockholmFormat::storeDocument( Document* doc, TaskStateInfo& ti, IOAdapter* io ) {
    assert( NULL != doc );
    assert( NULL != io );
    if( !io->isOpen() ) {
        ti.setError(Translations::badArgument("IO adapter"));
        return;
    }
    QString url = doc->getURL();
    assert( !url.isEmpty() );
    try {
        foreach( GObject* p_obj, doc->getObjects() ) {
            const MAlignmentObject* aln_obj = qobject_cast<const MAlignmentObject*>( p_obj );
            assert( NULL != aln_obj );
            save( io, aln_obj->getMAlignment(), aln_obj->getGObjectName() );
            if( ti.cancelFlag ) {
                return;
            }
        }
    } catch( const StockholmBaseException& ex ) {
        ti.setError( ex.msg );
    } catch(...) {
        ti.setError(tr( "unknown error occurred" ));
    }
}

bool StockholmFormat::isDataFormatSupported( const char* data, int size ) const {
    return checkHeader( data, size );
}

bool StockholmFormat::isObjectOpSupported( const Document *doc, DocObjectOp op, GObjectType t ) const {
    Q_UNUSED( op ); Q_UNUSED( doc );
    assert( NULL != doc );
    if ( GObjectTypes::MULTIPLE_ALIGNMENT != t  ) {
		return false;
    }
	/*if (op == DocumentFormat::DocObjectOp_Add) {
		return doc->getObjects().isEmpty();
	}
	return false;*/
    return true;
}

bool StockholmFormat::checkConstraints( const DocumentFormatConstraints &c ) const {
    bool ret = true;
    foreach ( GObjectType t, c.supportedObjectTypes ) {
        ret = ret && ( GObjectTypes::MULTIPLE_ALIGNMENT == t );
    }
    if( !ret ) {
        return false;
    }
    if ( c.checkRawData ) {
        ret = ret && isDataFormatSupported( c.rawData.constData(), c.rawData.size() );
    }
    if( c.supportsStreamingRead ) {
        ret = ret && true;
    }
    return ret;
}

} //GB2
