/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "GObjectUtils.h"

#include "DNASequenceObject.h"
#include "AnnotationTableObject.h"

#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/DocumentModel.h>
#include <core_api/DNATranslation.h>
#include <core_api/GHints.h>

#include <gobjects/GObjectRelationRoles.h>
#include <gobjects/UnloadedObject.h>
#include <gobjects/GObjectTypes.h>

namespace GB2 {


QList<GObject*> GObjectUtils::select(const QList<GObject*>& objs, GObjectType t, UnloadedObjectFilter f) {
    QList<GObject*> res;
    foreach(GObject* o, objs) {
        bool isUnloaded = o->getGObjectType() == GObjectTypes::UNLOADED;
        if ((t.isEmpty() && (f == UOF_LoadedAndUnloaded || !isUnloaded)) || o->getGObjectType() == t) {
            res.append(o);
        } else if (f == UOF_LoadedAndUnloaded && isUnloaded) {
            UnloadedObject* uo = qobject_cast<UnloadedObject*>(o);
            if (uo->getLoadedObjectType() == t) {
                res.append(o);
            }
        }
    }
    return res;
}

GObject* GObjectUtils::selectOne(const QList<GObject*>& objects, GObjectType type, UnloadedObjectFilter f) {
    QList<GObject*> res = select(objects, type, f);
    return res.isEmpty() ? NULL : res.first();
}


QList<GObject*> GObjectUtils::findAllObjects(UnloadedObjectFilter f, GObjectType t) {
    assert(AppContext::getProject()!=NULL);
    QList<GObject*> res;
    foreach(Document* doc, AppContext::getProject()->getDocuments()) {
        if (t.isEmpty()) {
            if (doc->isLoaded() || f == UOF_LoadedAndUnloaded) {
                res+=doc->getObjects();
            }
        } else {
            res+=doc->findGObjectByType(t, f);
        }
    }
    return res;
}


QSet<GObject*> GObjectUtils::selectRelations(GObject* obj, GObjectType type, const QString& relationRole, 
                                             const QList<GObject*>& fromObjects, UnloadedObjectFilter f)
{
    QSet<GObject*> res;
    QList<GObjectRelation> relations = obj->getObjectRelations();
    foreach(const GObjectRelation& r, relations) {
        if (r.role != relationRole || (!type.isEmpty() && r.ref.objType!=type)) {
            continue;
        }
        GObject* obj = selectObjectByReference(r.ref, fromObjects, f);
        if (obj!=NULL) {
            res.insert(obj);
        }
    }
    return res;
}

QList<GObject*> GObjectUtils::findObjectsRelatedToObjectByRole(const GObject* obj, GObjectType resultObjType, 
                                                               const QString& role, const QList<GObject*>& fromObjects, 
                                                               UnloadedObjectFilter f) 
{
    QList<GObject*> res;
    QList<GObject*>	objects  = select(fromObjects, resultObjType, f);
    foreach(GObject* o, objects) {
        if (o->hasObjectRelation(obj, role)) {
            res.append(o);
        }
    }
    return res;
}

QList<GObject*> GObjectUtils::selectObjectsWithRelation(const QList<GObject*>& objs, GObjectType type, 
                                                        const QString& relationRole, UnloadedObjectFilter f, bool availableObjectsOnly) 
{
    QSet<GObject*> availableObjects;
    if (availableObjectsOnly) {
        availableObjects = findAllObjects(f).toSet();
    }
    QList<GObject*> res;
    foreach(GObject* obj, objs) {
        QList<GObjectRelation> relations = obj->getObjectRelations();
        foreach(const GObjectRelation& r, relations) {
            if (r.role != relationRole || (!type.isEmpty() && r.ref.objType!=type)) {
                continue;
            }
            if (availableObjectsOnly) {
                Document* doc = AppContext::getProject()->findDocumentByURL(r.ref.docUrl);
                GObject* refObj = doc == NULL ? NULL : doc->findGObjectByName(r.ref.objName);
                if (refObj == NULL || (f == UOF_LoadedOnly && refObj->getGObjectType() == GObjectTypes::UNLOADED)) {
                    continue;
                }
            }
            res.append(obj);
        }
    }
    return res;
}

GObject* GObjectUtils::selectObjectByReference(const GObjectReference& r, UnloadedObjectFilter f) {
    return selectObjectByReference(r, findAllObjects(f, r.objType), f);
}

GObject* GObjectUtils::selectObjectByReference(const GObjectReference& r, const QList<GObject*>& fromObjects, UnloadedObjectFilter f) {
    foreach(GObject* o, fromObjects) {
        if (o->getGObjectName() != r.objName) {
            continue;
        }
        if ((o->getDocument() == NULL && !r.docUrl.isEmpty())
            || (o->getDocument()->getURL() != r.docUrl)) 
        {
            continue;
        }
        if (r.objType != o->getGObjectType()) {
            if (f != UOF_LoadedAndUnloaded) {
                continue;
            }
            if (o->getGObjectType()!=GObjectTypes::UNLOADED || r.objType != qobject_cast<UnloadedObject*>(o)->getLoadedObjectType()) {
                continue;
            }
        }
        return o;
    }
    return NULL;
}

DNATranslation* GObjectUtils::findComplementTT(DNASequenceObject* so) {
    if (!so->getAlphabet()->isNucleic()) {
        return NULL;
    }
    return AppContext::getDNATranslationRegistry()->lookupComplementTranslation(so->getAlphabet());
}

DNATranslation* GObjectUtils::findAminoTT(DNASequenceObject* so, bool fromHintsOnly) {
    if (!so->getAlphabet()->isNucleic()) {
        return NULL;
    }
    DNATranslationRegistry* tr = AppContext::getDNATranslationRegistry();
    QString tid = so->getGHints()->get(AMINO_TT_GOBJECT_HINT).toString();
    DNATranslation* res = tr->lookupTranslation(so->getAlphabet(), DNATranslationType_NUCL_2_AMINO, tid);
    if (res != NULL || fromHintsOnly) {
        return res;
    }
    QList<DNATranslation*> aminoTs = tr->lookupTranslation(so->getAlphabet(), DNATranslationType_NUCL_2_AMINO);
    if (!aminoTs.empty()) {
        res = aminoTs.first();
    }
    return res;
}

bool GObjectUtils::hasType(GObject* obj, const GObjectType& type) {
    if (obj->getGObjectType() == type) {
        return true;
    }
    if (obj->getGObjectType() != GObjectTypes::UNLOADED) {
        return false;
    }
    UnloadedObject* uo = qobject_cast<UnloadedObject*>(obj);
    return uo->getLoadedObjectType() == type;
}


void GObjectUtils::updateRelationsURL(GObject* o, const QString& fromURL, const QString& toURL) {
    QList<GObjectRelation> relations = o->getObjectRelations();
    bool changed = false;
    for(int i=0; i<relations.size(); i++) {
        GObjectRelation& r = relations[i];
        if (r.ref.docUrl == fromURL) {
            r.ref.docUrl = toURL;
            changed = true;
        }
    }
    if (changed) {
        o->setObjectRelations(relations);
    }
}

} //namespace

