/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "LocalFileAdapter.h"
#include "ZlibAdapter.h"

namespace GB2 {

LocalFileAdapterFactory::LocalFileAdapterFactory(QObject* o) : IOAdapterFactory(o) {
	name = tr("local_file");
}		
	
IOAdapter* LocalFileAdapterFactory::createIOAdapter() {
	return new LocalFileAdapter(this);
}

GzippedLocalFileAdapterFactory::GzippedLocalFileAdapterFactory(QObject* o) 
: LocalFileAdapterFactory(o) {
	name = tr("local_file_gzip");
}		

IOAdapter* GzippedLocalFileAdapterFactory::createIOAdapter() {
	return new ZlibAdapter(new LocalFileAdapter(this));
}

LocalFileAdapter::LocalFileAdapter(LocalFileAdapterFactory* factory, QObject* o) : IOAdapter(factory, o), f(NULL) {
}		


bool LocalFileAdapter::open(const QString& url, IOAdapterMode m) {
    assert(!isOpen());
    assert(f==NULL);
    if (url.isEmpty()) {
        return false;
    }
    f = new QFile(url);
    QIODevice::OpenMode iomode;
    switch (m) {
        case IOAdapterMode_Read: iomode = QIODevice::ReadOnly; break;
        case IOAdapterMode_Write: iomode = QIODevice::WriteOnly | QIODevice::Truncate; break;
        case IOAdapterMode_Append: iomode = QIODevice::WriteOnly | QIODevice::Append; break;
    }
    bool res = f->open(iomode);
    if (!res) {
        delete f;
        f = NULL;
        return false;
    }
    return true;
}

void LocalFileAdapter::close() {
	assert(isOpen());
	if (!isOpen()) {
		return;
	}
	f->close();
	delete f;
	f = NULL;
}

qint64 LocalFileAdapter::readBlock(char* data, qint64 size) {
	qint64 l = f->read(data, size);
	return l;
}

qint64 LocalFileAdapter::writeBlock(const char* data, qint64 size) {
	qint64 l = f->write(data, size);
	return l;
}

bool LocalFileAdapter::skip(qint64 nBytes) {
	assert(isOpen());
	if (!isOpen()) {
		return false;
	}
	qint64 p = f->pos();
	return f->seek(p+nBytes);
}

qint64 LocalFileAdapter::left() const {
    assert(isOpen());
    qint64 p = f->pos();
    qint64 len = f->size();
    return len - p;
}

int LocalFileAdapter::getProgress() const {
	assert(isOpen());
	return int(100 * float(f->pos()) / f->size());
}

qint64 LocalFileAdapter::bytesRead() const {
    return f->pos();
}

QString LocalFileAdapter::getUrl() const {
    return f->fileName();
}

};//namespace
