/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ADVSyncViewManager.h"
#include "AnnotatedDNAView.h"
#include "ADVSingleSequenceWidget.h"
#include "ADVSequenceObjectContext.h"
#include "PanView.h"

#include <selection/DNASequenceSelection.h>
#include <gobjects/AnnotationTableObject.h>

namespace GB2 {

ADVSyncViewManager::ADVSyncViewManager(AnnotatedDNAView* v) : QObject(v), adv(v) 
{
    assert(v->getSequenceContexts().isEmpty());

    recursion = false;

    lockByStartPosAction = new QAction(tr("Lock scales: visible range start"), this);
    connect(lockByStartPosAction, SIGNAL(triggered()), SLOT(sl_lock()));
    
    lockBySeqSelAction = new QAction(tr("Lock scales: selected sequence"), this);
    connect(lockBySeqSelAction, SIGNAL(triggered()), SLOT(sl_lock()));

    lockByAnnSelAction = new QAction(tr("Lock scales: selected annotation"), this);
    connect(lockByAnnSelAction, SIGNAL(triggered()), SLOT(sl_lock()));

    syncByStartPosAction = new QAction(tr("Adjust scales: visible range start"), this);
    connect(syncByStartPosAction, SIGNAL(triggered()), SLOT(sl_sync()));

    syncBySeqSelAction = new QAction(tr("Adjust scales: selected sequence"), this);
    connect(syncBySeqSelAction, SIGNAL(triggered()), SLOT(sl_sync()));

    syncByAnnSelAction = new QAction(tr("Adjust scales: selected annotation"), this);
    connect(syncByAnnSelAction, SIGNAL(triggered()), SLOT(sl_sync()));

    lockMenu = new QMenu(tr("Lock scales"));
    lockMenu->setIcon(QIcon(":core/images/lock_scales.png"));
    lockMenu->addAction(lockByStartPosAction);
    lockMenu->addAction(lockBySeqSelAction);
    lockMenu->addAction(lockByAnnSelAction);
        
    syncMenu = new QMenu(tr("Adjust scales"));
    syncMenu->setIcon(QIcon(":core/images/sync_scales.png"));
    syncMenu->addAction(syncByStartPosAction);
    syncMenu->addAction(syncBySeqSelAction);
    syncMenu->addAction(syncByAnnSelAction);
    
    lockButton = new QToolButton();
    lockButton->setCheckable(true);
    connect(lockButton, SIGNAL(clicked()), SLOT(sl_lock()));
    lockButton->setDefaultAction(lockMenu->menuAction());
    lockButton->setCheckable(true);

    syncButton = new QToolButton();
    connect(syncButton, SIGNAL(clicked()), SLOT(sl_sync()));
    syncButton->setDefaultAction(syncMenu->menuAction());

    lockButtonTBAction = NULL;
    syncButtonTBAction = NULL;
    
    // visual mode ops
    toggleAllAction = new QAction("Toggle Sequence view", this);
    connect(toggleAllAction, SIGNAL(triggered()), SLOT(sl_toggleVisualMode()));

    togglePanAction = new QAction("Toggle Overview", this);
    connect(togglePanAction, SIGNAL(triggered()), SLOT(sl_toggleVisualMode()));
    
    toggleDetAction = new QAction("Toggle Details", this);
    connect(toggleDetAction, SIGNAL(triggered()), SLOT(sl_toggleVisualMode()));

    toggleViewButtonAction = NULL;
    toggleViewButtonMenu =  new QMenu(tr("Toggle views"));
    toggleViewButtonMenu->setIcon(QIcon(":core/images/adv_widget_menu.png"));

    toggleViewButtonMenu->addAction(toggleAllAction); //-> behavior can be not clear to user
    toggleViewButtonMenu->addAction(togglePanAction);
    toggleViewButtonMenu->addAction(toggleDetAction);
    connect(toggleViewButtonMenu, SIGNAL(aboutToShow()), SLOT(updateVisualMode()));
    
    toggleViewButton = new QToolButton();
    toggleViewButton->setDefaultAction(toggleViewButtonMenu->menuAction());
    toggleViewButton->setPopupMode(QToolButton::InstantPopup);
    
    updateEnabledState();

    connect(adv, SIGNAL(si_sequenceWidgetAdded(ADVSequenceWidget*)), SLOT(sl_sequenceWidgetAdded(ADVSequenceWidget*)));
    connect(adv, SIGNAL(si_sequenceWidgetRemoved(ADVSequenceWidget*)), SLOT(sl_sequenceWidgetRemoved(ADVSequenceWidget*)));
}

ADVSyncViewManager::~ADVSyncViewManager() {
    delete lockButton;
    delete syncButton;
    delete syncMenu;
    delete lockMenu;
    
    delete toggleViewButton;
    delete toggleViewButtonMenu;
}

void ADVSyncViewManager::updateToolbar1(QToolBar* tb) {
    if (lockButtonTBAction == NULL) {
        lockButtonTBAction = tb->addWidget(lockButton);
        syncButtonTBAction = tb->addWidget(syncButton);
    } else {
        tb->addAction(lockButtonTBAction);
        tb->addAction(syncButtonTBAction);
    }
}

void ADVSyncViewManager::updateToolbar2(QToolBar* tb) {
    if (toggleViewButtonAction == NULL) {
        toggleViewButtonAction = tb->addWidget(toggleViewButton);
    } else {
        tb->addAction(toggleViewButtonAction);
    }
}

void ADVSyncViewManager::updateEnabledState() {
    bool enabled = getViewsFromADV().size() > 1;
    syncButton->setEnabled(enabled);
    lockButton->setEnabled(enabled);
}

void ADVSyncViewManager::sl_sequenceWidgetAdded(ADVSequenceWidget* w) {
    ADVSingleSequenceWidget* sw = qobject_cast<ADVSingleSequenceWidget*>(w);
    if (sw == NULL) {
        return;
    }
    unlock();
}

void ADVSyncViewManager::sl_sequenceWidgetRemoved(ADVSequenceWidget* w) {
    ADVSingleSequenceWidget* sw = qobject_cast<ADVSingleSequenceWidget*>(w);
    if (sw == NULL) {
        return;
    }
    unlock();
}

void ADVSyncViewManager::unlock() {
    lockButton->setChecked(false);
    foreach(ADVSingleSequenceWidget* sw, views) {
        sw->getPanView()->disconnect(this);
    }
    views.clear();
    updateEnabledState();
}

QList<ADVSingleSequenceWidget*> ADVSyncViewManager::getViewsFromADV() const {
    QList<ADVSingleSequenceWidget*> res;
    foreach(ADVSequenceWidget* w, adv->getSequenceWidgets()) {
        ADVSingleSequenceWidget* sw = qobject_cast<ADVSingleSequenceWidget*>(w);
        if (sw != NULL) {
            res.append(sw);
        }
    }
    return res;
}

void ADVSyncViewManager::sl_rangeChanged() {
    if (recursion) {
        return;
    }
    recursion = true;

    PanView* activePan = qobject_cast<PanView*>(sender());
    const LRegion& activeRange = activePan->getVisibleRange();
    int activeOffset = activePan->getSyncOffset();
    foreach(ADVSingleSequenceWidget* sw, views) {
        PanView* pan = sw->getPanView();
        if (pan == activePan) {
            continue;
        }
        int panOffset = pan->getSyncOffset();
        int resultOffset = panOffset - activeOffset;
        int seqLen = pan->getSeqLen();
        int newStart = qBound(0, activeRange.startPos + resultOffset, seqLen);
        int nVisible = qMin(activeRange.len, seqLen);
        if (newStart + nVisible > seqLen) {
            newStart = seqLen - nVisible;
        }
        assert(newStart >= 0 && newStart + nVisible <= seqLen);
        pan->setVisibleRange(LRegion(newStart, nVisible));
    }
    
    recursion = false;
}

void ADVSyncViewManager::sl_lock() {
    if (lockButton->isChecked()) {
        unlock();
        return;
    }
    QObject* s = sender();
    SyncMode m = SyncMode_Start;
    if (s == lockBySeqSelAction) {
        m = SyncMode_SeqSel;
    } else if (s == lockByAnnSelAction) {
        m = SyncMode_AnnSel;
    } else if (s == lockButton) {
        m = detectSyncMode();
    }
    sync(true, m);
    lockButton->setChecked(true);
}

void ADVSyncViewManager::sl_sync() {
    QObject* s = sender();
    SyncMode m = SyncMode_Start;
    if (s == syncBySeqSelAction) {
        m = SyncMode_SeqSel;
    } else if (s == syncByAnnSelAction) {
        m = SyncMode_AnnSel;
    } else if (s == syncButton) {
        m = detectSyncMode();
    }
    sync(false, m);
}

void ADVSyncViewManager::sync(bool lock, SyncMode m) {
    ADVSingleSequenceWidget* focusedW  = qobject_cast<ADVSingleSequenceWidget*>(adv->getSequenceWidgetInFocus());
    if (focusedW == NULL) {
        return;
    }

    QList<ADVSingleSequenceWidget*> seqs = getViewsFromADV();
    QVector<int> offsets(seqs.size());
    
    //offset here ==> new panview start pos
    //dOffset is used to keep focused sequence unchanged
    LRegion focusedRange;
    int dOffset = 0; 
    for (int i=0; i< seqs.size(); i++ ){
        int offset = 0;
        ADVSingleSequenceWidget* seqW = seqs[i];
        switch(m) {
            case SyncMode_Start:  offset = seqW->getVisibleRange().startPos; break;
            case SyncMode_SeqSel: offset = offsetBySeqSel(seqW);break;
            case SyncMode_AnnSel: offset = offsetByAnnSel(seqW);break;
        }
        offsets[i] = offset;
        if (seqW == focusedW) {
            focusedRange = focusedW->getVisibleRange();
            dOffset = offset - focusedRange.startPos;
        } 
    }
    assert(!focusedRange.isEmpty());
    for (int i=0; i< seqs.size(); i++ ){
        ADVSingleSequenceWidget* seqW = seqs[i];
        int offset = offsets[i] - dOffset;

        PanView* pan = seqW->getPanView();
        if (seqW != focusedW) {
            pan->setNumBasesVisible(focusedRange.len);
            pan->setStartPos(offset);
        }
        if (lock) {
            pan->setSyncOffset(offset);
            connect(pan, SIGNAL(si_visibleRangeChanged()), SLOT(sl_rangeChanged()));
            views.append(seqW);
        } 
    }
}

int ADVSyncViewManager::offsetBySeqSel(ADVSingleSequenceWidget* w) const {
    DNASequenceSelection* seqSel = w->getSequenceContext()->getSequenceSelection();
    if (seqSel->isEmpty()) {
        return w->getVisibleRange().startPos;
    }
    return seqSel->getSelectedRegions().first().startPos;
}

int ADVSyncViewManager::offsetByAnnSel(ADVSingleSequenceWidget* w) const {
    int pos = findSelectedAnnotationPos(w);
    if (pos == -1) {
        return w->getVisibleRange().startPos;
    }
    return pos;
}

int ADVSyncViewManager::findSelectedAnnotationPos(ADVSingleSequenceWidget* w) const {
    AnnotationSelection* as = w->getSequenceContext()->getAnnotationsSelection();
    const QSet<AnnotationTableObject*>& objs = w->getSequenceContext()->getAnnotationObjects();
    foreach(const AnnotationSelectionData& d , as->getSelection()) {
        AnnotationTableObject* obj = d.annotation->getGObject();
        if (objs.contains(obj)) {
            return d.annotation->isOnComplementStrand() ? d.getSelectedRegions().last().endPos() : d.getSelectedRegions().first().startPos;
        }
    }
    return -1;
}


ADVSyncViewManager::SyncMode ADVSyncViewManager::detectSyncMode() const {
    ADVSingleSequenceWidget* focusedW  = qobject_cast<ADVSingleSequenceWidget*>(adv->getSequenceWidgetInFocus());
    assert(focusedW != NULL);
    QList<ADVSingleSequenceWidget*> seqs = getViewsFromADV();

    //if current sequence + any other sequence have annotation selection -> sync by annotation
    if (findSelectedAnnotationPos(focusedW) != -1) {
        foreach(ADVSingleSequenceWidget* sw, seqs) {
            if (sw != focusedW && findSelectedAnnotationPos(sw) != -1) {
                return SyncMode_AnnSel;
            }
        }
    }

    //if current sequence + any other sequence have sequence selection -> sync by annotation
    if (!focusedW->getSequenceContext()->getSequenceSelection()->isEmpty()) {
        foreach(ADVSingleSequenceWidget* sw, seqs) {
            if (sw != focusedW && !sw->getSequenceContext()->getSequenceSelection()->isEmpty()) {
                return SyncMode_SeqSel;
            }
        }
    }
    // else sync by start pos
    return SyncMode_Start;

}

void ADVSyncViewManager::updateVisualMode() {
    //if have at least 1 visible -> hide all
    bool haveVisiblePan = false;    
    bool haveVisibleDet = false;
    bool haveVisibleView = false;
    foreach(ADVSingleSequenceWidget* sw, getViewsFromADV()) {
        haveVisiblePan = haveVisiblePan || !sw->isPanViewCollapsed();
        haveVisibleDet = haveVisibleDet || !sw->isDetViewCollapsed();
        haveVisibleView = haveVisibleView || !sw->isViewCollapsed();
    }
    toggleAllAction->setText(haveVisibleView ? tr("Hide all sequences") : tr("Show all sequences"));
    togglePanAction->setText(haveVisiblePan ? tr("Hide all overviews") : tr("Show all overviews"));
    toggleDetAction->setText(haveVisibleDet ? tr("Hide all details")  : tr("Show all details"));
}

void ADVSyncViewManager::sl_toggleVisualMode() {
    //if have at least 1 visible -> hide all
    bool haveVisiblePan = false;    
    bool haveVisibleDet = false;
    bool haveVisibleView = false;
    
    QList<ADVSingleSequenceWidget*> views = getViewsFromADV();
    foreach(ADVSingleSequenceWidget* sw, views) {
        haveVisibleDet = haveVisibleDet || !sw->isDetViewCollapsed();
        haveVisibleView = haveVisibleView || !sw->isViewCollapsed();
    }

    QObject* s = sender();
    foreach(ADVSingleSequenceWidget* sw, views) {
        if (s == togglePanAction) {
            sw->setPanViewCollapsed(haveVisiblePan);
        } else if (s == toggleDetAction) {
            sw->setDetViewCollapsed(haveVisibleDet);
        } else {
            sw->setViewCollapsed(haveVisibleView);
        }
    }
}

}//namespace
