/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_MSA_EDITOR_SEQUENCE_AREA_H_
#define _GB2_MSA_EDITOR_SEQUENCE_AREA_H_

#include <core_api/core_api.h>
#include <core_api/LRegion.h>

#include <QtGui/QWidget>
#include <QtGui/QMenu>
#include <QtGui/QToolBar>
#include <QtGui/QScrollBar>

namespace GB2 {

class MSAEditor;
class MSAEditorUI;
class GObjectView;
class MSAColorScheme;
class MAlignment;
class MAlignmentModInfo;

class GB2_COREAPI_EXPORT MSAEditorSequenceArea : public QWidget {
    Q_OBJECT
public:
    MSAEditorSequenceArea(MSAEditorUI* ui, QScrollBar* hb, QScrollBar* vb);
    ~MSAEditorSequenceArea();

    // x dimension -> positions
    int countWidthForBases(bool countClipped) const;
    
    int getFirstVisibleBase() const {return startPos;}

    int getLastVisibleBase(bool countClipped) const;

    int getNumVisibleBases(bool countClipped) const; 

    LRegion getBaseXRange(int pos, bool useVirtualCoords) const;

    void setFirstVisibleBase(int pos);


    // y dimension -> sequences
    int countHeightForSequences(bool countClipped) const;

    int getFirstVisibleSequence() const {return startSeq;} 
    
    int getLastVisibleSequence(bool countClipped) const;

    int getNumVisibleSequences(bool countClipped) const;

    LRegion getSequenceYRange(int seqNum, bool useVirtualCoords) const;

    int getSequenceNumByY(int y);

    void setFirstVisibleSequence(int seq);

    

    bool isVisible(const QPoint& p, bool countClipped) const {return isPosVisible(p.x(), countClipped) && isSeqVisible(p.y(), countClipped);}
    
    bool isPosVisible(int pos, bool countClipped) const;
    
    bool isSeqVisible(int seq, bool countClipped) const;


    const QFont& getSeqFont() {return seqFont;}
    
    int coordToPos(int x) const;

    QPoint coordToPos(const QPoint& coord) const;

    const QPoint& getCursorPos() const {assert(checkState()); return cursorPos;}

    void setCursorPos(const QPoint& p);

    void setCursorPos(int x, int y) { setCursorPos(QPoint(x, y)); };
    
    void setCursorPos(int pos) { setCursorPos(QPoint(pos, cursorPos.y())); }

    void moveCursor(int dx, int dy);

    void centerPos(const QPoint& pos);
    
    void centerPos(int pos);

    
    const QFont& getFont() const {return seqFont;}
    
    void setFont(const QFont& f);

    QScrollBar* getVBar() const {return svBar;}

    QScrollBar* getHBar() const {return shBar;}

    int getRowHeight() const {return seqCharHeight;}
    
    int getColumnWidth() const {return seqCharWidth;}

    void highlightCurrentCursorPos()  { highlightSelection = true; update();}

signals:
    void si_startChanged(const QPoint& p, const QPoint& prev);
    void si_cursorMoved(const QPoint& p, const QPoint& prev);
    void si_scaleChanged(); //emitted when either font or scale changes

protected:
    void resizeEvent(QResizeEvent *);
    void paintEvent(QPaintEvent *);
    void mousePressEvent(QMouseEvent *);
    void keyPressEvent(QKeyEvent *);
    void focusOutEvent(QFocusEvent* fe); 
    void focusInEvent(QFocusEvent* fe);

private slots:
    void sl_onHScrollMoved(int pos);
    void sl_onVScrollMoved(int pos);
    void sl_alignmentChanged(const MAlignment&, const MAlignmentModInfo&);

    void sl_buildStaticMenu(GObjectView* v, QMenu* m);
    void sl_buildStaticToolbar(GObjectView* v, QToolBar* t);
    void sl_buildContextMenu(GObjectView* v, QMenu* m);
    void sl_lockedStateChanged();

    void sl_delSym();
    void sl_delCol();
    void sl_insSym();
    void sl_insCol();
    void sl_goto();
    void sl_removeColumnsWithGaps();
    void sl_removeAllGaps();

    void sl_onPosChangeRequest(int pos);

    void sl_changeFont(); 
    void sl_resetFont(); 
    void sl_zoomIn(); 
    void sl_zoomOut(); 

    void sl_changeColorScheme();

protected:
    virtual void wheelEvent (QWheelEvent * event);

private:
    void buildMenu(QMenu* m);
    void prepareColorSchemeMenuActions();

    bool isPosInRange(int p) const;
    bool isSeqInRange(int s) const;
    bool isInRange(const QPoint& p) const {return isPosInRange(p.x()) && isSeqInRange(p.y());}
    
    void updateActions();
    
    void updateHScrollBar();
    void updateVScrollBar();
    
    void drawAll();
    void drawContent(QPainter& p);
    void drawCursor(QPainter& p);
    void drawFocus(QPainter& p);

    void ins(const QPoint& p, bool columnMode);
    void del(const QPoint& p, bool columnMode);

    bool checkState() const;
    void validateRanges();          //called on resize/refont like events
    void updateSeqFontMetrics();    //update all cached vars based on seqFont

    MSAEditor*      editor;
    MSAEditorUI*    ui;
    QScrollBar*     shBar;
    QScrollBar*     svBar;

    int             startPos; //first visible x pos 
    int             startSeq; //first visible y pos


    QFont           seqFont;
    int             seqCharWidth;
    int             seqCharHeight;
    QPoint          cursorPos;

    QAction*        delSymAction;
    QAction*        delColAction;
    QAction*        insSymAction;
    QAction*        insColAction;
    QAction*        removeGapColumnsAction;
    QAction*        removeAllGapsAction;
    QAction*        gotoAction;
    
    QAction*        takeScreenshotAction;
    QAction*        increaseFontAction;
    QAction*        decreseFontAction;
    QAction*        changeFontAction;
    QAction*        resetFontAction;
    
    QPixmap*        cachedView;
    bool            completeRedraw;

    MSAColorScheme* colorScheme;
    bool            highlightSelection;

    QList<QAction*> colorSchemeMenuActions;
};

}//namespace
#endif
