/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_RF_BASE_ALG_H_
#define _GB2_RF_BASE_ALG_H_

#include <core_api/DNAAlphabet.h>
#include <core_api/Task.h>

#include <QtCore/QVector>

namespace GB2 {

enum RFAlgorithm {
    RFAlgorithm_Auto,
    RFAlgorithm_Diagonal,
    RFAlgorithm_Suffix
};

#define PCHAR_MATCHES(x, y) (*(x) == *(y) && *(x) != unknownChar)
#define CHAR_MATCHES(x, y)  ((x) == (y) && (x) != unknownChar)

class RFResult {
public:
    RFResult() : x(0), y(0), l(0){}
    RFResult(int _x, int _y, int _len) : x(_x), y(_y), l(_len){}

    bool operator==(const RFResult& r) const {return x == r.x || y == r.y || l == r.l;}
    bool operator!=(const RFResult& r) const {return !(*this == r);}
    bool operator <(const RFResult& r) const {return (x != r.x) ? x < r.x : (y != r.y) ? y < r.y: (l < r.l);}

    int x;
    int y;
    int l;
};

class RFResultsListener {
public:
    virtual void onResult(const RFResult& r) = 0;
    virtual void onResults(const QVector<RFResult>& v) = 0;
};

class RFAlgorithmBase : public Task {
    Q_OBJECT
public:
    RFAlgorithmBase(RFResultsListener* l, 
                    const char* seqX, int sizeX,
                    const char* seqY, int sizeY, DNAAlphabetType seqType, 
                    int w, int k, TaskFlags flags = TaskFlags_NR_FOSCOE);

    void setReportReflected(bool v) {reportReflected = v;}
    
    void prepare();
    
    static RFAlgorithmBase* createTask(RFResultsListener* l, 
                                        const char *seqx, int sizeX, 
                                        const char *seqY, int sizeY,
                                        DNAAlphabetType seqType, int w, 
                                        int mismatches = 0, RFAlgorithm alg = RFAlgorithm_Auto, 
                                        int nThreads = MAX_PARALLEL_SUBTASKS_AUTO); 

protected:
    // adds single result to global results 
    void addToResults(const RFResult& r);

    // adds multiple results to global results 
    void addToResults(const QVector<RFResult>& newResults);

    //always return true. bool -> to use in assertions
    bool checkResults(const QVector<RFResult>& v);
    bool checkResult(const RFResult& v);

    const char*             seqX;
    const char*             seqY;
    const int               SIZE_X;
    const int               SIZE_Y;
    const DNAAlphabetType   SEQ_TYPE;
    const int               WINDOW_SIZE;
    const int               K;
    const int               C;

    bool                    reflective;
    char                    unknownChar;

    RFResultsListener*      resultsListener;
    bool                    reportReflected;
};

} //namespace

#endif
