/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SWAlgorithmTask.h"

#include <core_api/AppContext.h>
#include <core_api/AppSettings.h>
#include <core_api/AppResources.h>
#include <core_api/Log.h>
#include <util_smith_waterman/SmithWatermanResult.h>

#include <iostream>
#include <time.h>
#include <QtCore/QMutexLocker>

using namespace std;

namespace GB2 {

static LogCategory log(ULOG_CAT_SW);

int startT = 0;
int endT = 0;

SWAlgorithmTask::SWAlgorithmTask(const SmithWatermanSettings& s,
                                 const QString& taskName, AlgType _algType):Task (taskName, TaskFlag_NoRun), 
                                 sWatermanConfig(s)
{
	startT = clock();
	log.info("RUN constructor SWAlgorithmTask");	

	algType = _algType;
	if (algType == sse2) if(sWatermanConfig.ptrn.length() < 8) algType = classic;	

    int maxScore = calculateMaxScore(s.ptrn, s.pSm);    
	
	minScore = (maxScore * s.percentOfScore) / 100;		
	if ( (maxScore * (int)s.percentOfScore) % 100 != 0) minScore += 1;	

    setupTask(maxScore);

	log.info("FINISH constructor SWAlgorithmTask");
}

void SWAlgorithmTask::setupTask(int maxScore) {    

    SequenceWalkerConfig c;
    c.seq = sWatermanConfig.sqnc.constData();
    c.seqSize = sWatermanConfig.sqnc.size();
	c.range = sWatermanConfig.globalRegion;
    c.complTrans = sWatermanConfig.complTT;
    c.aminoTrans = sWatermanConfig.aminoTT;
	
    int matrixLength = calculateMatrixLength(sWatermanConfig.sqnc, 
		sWatermanConfig.ptrn, 
		sWatermanConfig.gapModel.scoreGapOpen, 
		sWatermanConfig.gapModel.scoreGapExtd, 
		maxScore, 
		minScore);

	// divide sequence by PARTS_NUMBER parts
	int idealThreadCount = AppContext::getAppSettings()->getAppResourcePool()->getIdealThreadCount();

	int PARTS_NUMBER = 0;	 
	if (algType == sse2) {
		PARTS_NUMBER = idealThreadCount * 2.5;
	} else if (algType == classic){
		PARTS_NUMBER = idealThreadCount;
	} else if (algType == cuda) {
		PARTS_NUMBER = 1;
	}

    if ((PARTS_NUMBER != 1) && (PARTS_NUMBER - 1) * matrixLength < sWatermanConfig.globalRegion.len) {
		c.chunkSize = (c.seqSize + matrixLength * (PARTS_NUMBER - 1)) / PARTS_NUMBER;
		if (c.chunkSize == matrixLength) c.chunkSize++;
		c.overlapSize = matrixLength;			
    }
    else {
        c.overlapSize = 0;	
        c.chunkSize = c.seqSize;
		PARTS_NUMBER = 1;
    }    

	cout <<"PARTS_NUMBER: " <<PARTS_NUMBER <<endl;

    c.lastChunkExtraLen = PARTS_NUMBER - 1;	
    c.nThreads = PARTS_NUMBER;

    SequenceWalkerTask* t = new SequenceWalkerTask(c, this, tr("Smith Waterman2 SequenceWalker"));	
    addSubTask(t);
}

QList<PairAlignSequences> &  SWAlgorithmTask::getResult() {
	
	removeResultFromOverlap(pairAlignSequences);
    SmithWatermanAlgorithm::sortByScore(pairAlignSequences);

    return pairAlignSequences;
}

void SWAlgorithmTask::onRegion(SequenceWalkerSubtask* t, TaskStateInfo& ti) {				
    Q_UNUSED(ti);

	log.info("RUN SWAlgorithmTask::onRegion(SequenceWalkerSubtask* t, TaskStateInfo& ti)");
	

    int regionLen = t->getRegionSequenceLen();
    QByteArray localSeq(t->getRegionSequence(), regionLen);

	SmithWatermanAlgorithm * sw;

	if (algType == sse2) {
#ifdef SW2_BUILD_WITH_SSE2
        sw = new SmithWatermanAlgorithmSSE2;
#else
        log.error( "SSE2 was not enabled in this build" );
        return;
#endif //SW2_BUILD_WITH_SSE2
	} else if (algType == classic) {
        sw = new SmithWatermanAlgorithm;
    }
	else if (algType == cuda) {
#ifdef SW2_BUILD_WITH_CUDA
		sw = new SmithWatermanAlgorithmCUDA;
#else
		log.error( "CUDA was not enabled in this build" );
		return;
#endif //SW2_BUILD_WITH_CUDA
	}


    sw->launch(sWatermanConfig.pSm, sWatermanConfig.ptrn, localSeq, 
		sWatermanConfig.gapModel.scoreGapOpen + sWatermanConfig.gapModel.scoreGapExtd, 
		sWatermanConfig.gapModel.scoreGapExtd, 
		minScore);				

    QList<PairAlignSequences> res = sw->getResults();

// 	SmithWatermanAlgorithmCUDA sw;
//      sw.launch(sWatermanConfig.pSm, sWatermanConfig.ptrn, localSeq, 
//  		(-1) * (sWatermanConfig.gapModel.scoreGapOpen + sWatermanConfig.gapModel.scoreGapExtd), 
//  		(-1) * (sWatermanConfig.gapModel.scoreGapExtd), 
//  		minScore);				
// 	QList<PairAlignSequences> res = sw.results;

	for (int i = 0; i < res.size(); i++) {
		res[i].isDNAComplemented = t->isDNAComplemented();
		res[i].isAminoTranslated = t->isAminoTranslated();

		if (t->isAminoTranslated()) {
			res[i].intervalSeq1.startPos *= 3;
			res[i].intervalSeq1.len *= 3;
		}

		if (t->isDNAComplemented()) {
			res[i].intervalSeq1.startPos = t->getGlobalRegion().len - res[i].intervalSeq1.endPos();
		}
		else {
			res[i].intervalSeq1.startPos += 
				(t->getGlobalRegion().startPos - sWatermanConfig.globalRegion.startPos);
		}		
	}
	
	//print result
// 	cout <<"local seq: " <<localSeq.data() <<endl;	
// 	cout <<"print result: " <<endl;
// 	for (int i = 0; i < res.size(); i++) {
// 		QByteArray alignSearchSeq;
// 		QByteArray alignPatterSeq;
// 		res[i].getAlignSequences(localSeq, sWatermanConfig.ptrn, alignSearchSeq, alignPatterSeq );
// 		cout <<alignSearchSeq.data() <<endl;
// 		cout <<alignPatterSeq.data() <<endl <<endl;
// 	}	
    addResult(res, 0, t->isDNAComplemented(), t->isAminoTranslated());

/////////////////////
	delete sw;
	log.info("FINISH SWAlgorithmTask::onRegion(SequenceWalkerSubtask* t, TaskStateInfo& ti)");
}

void SWAlgorithmTask::removeResultFromOverlap(QList<PairAlignSequences> & res) { 	
	log.info("Removing results From Overlap");	

	for (int i = 0; i < res.size() - 1; i++) {
		for (int j = i + 1; j < res.size(); j++) {
			if (res.at(i).intervalSeq1 == res.at(j).intervalSeq1 && 
				res.at(i).score == res.at(j).score) {					
				res.removeAt(j);				
				j--;
			}
		}
	}
	
}


void SWAlgorithmTask::addResult(QList<PairAlignSequences> & res, int sPos, 
								bool _isDNAComplemented, bool _isAminoTranslated) {
    QMutexLocker ml(&lock);	
    pairAlignSequences += res;	
	pairAlignSequences += res;
}

int SWAlgorithmTask::calculateMatrixLength(const QByteArray & searchSeq, const QByteArray & patternSeq, int gapOpen, int gapExtension, int maxScore, int minScore) {

	int matrixLength = 0;

    int gap = gapOpen;
    if (gapOpen < gapExtension) gap = gapExtension;

    matrixLength = patternSeq.length() + (maxScore - minScore)/gap * (-1) + 1;	

    if (searchSeq.length() + 1 < matrixLength) matrixLength = searchSeq.length() + 1;

    matrixLength += 1;

    return matrixLength;	
}

int SWAlgorithmTask::calculateMaxScore(const QByteArray & seq, SubstMatrix const * substitutionMatrix) {
    int maxScore = 0;
    int max;	
    int substValue = 0;	

    for (int i = 0; i < seq.length(); i++) {		
        max = 0;		
        for (int j = 0; j < substitutionMatrix->getAlphabet()->getNumAlphabetChars(); j++) {			

            substValue = substitutionMatrix->
                getScore(seq.at(i), substitutionMatrix->getAlphabet()->getAlphabetChars().at(j));
            if (max < substValue) max = substValue;								
        }
        maxScore += max;
    }	
    return maxScore;
}

Task::ReportResult SWAlgorithmTask::report() {
	endT = clock();

	cout <<"&&&&&&&&&&&&&&&&&&&&&&&&& spend time : " <<(endT - startT) <<" &&&&&&&&&&&&&&&&&&&&&&&&&&" <<endl;

	log.info("RUN SWAlgorithmTask::report()");

    QList<SmithWatermanResult> resultList;	
    QList<PairAlignSequences> & resPAS =  getResult();

    SmithWatermanResult r;
    for (int i = 0; i < resPAS.size(); i++) {

        r.complement = resPAS.at(i).isDNAComplemented;
		r.trans = resPAS.at(i).isAminoTranslated;
        r.region = resPAS.at(i).intervalSeq1;
        r.region.startPos += sWatermanConfig.globalRegion.startPos;
        r.score = resPAS.at(i).score;

		sWatermanConfig.resultListener->pushResult(r);
        resultList.append(r);
    }	
    
	if (0 != sWatermanConfig.resultFilter) {
		SmithWatermanResultFilter* rf = sWatermanConfig.resultFilter;
		rf->applyFilter(&resultList);
    }	
	if (0 != sWatermanConfig.resultCallback) {
		SmithWatermanReportCallback* rcb = sWatermanConfig.resultCallback;
		QString res = rcb->report(resultList);
		if (!res.isEmpty()) {
			stateInfo.setError(res);
		}
	}    	
	
	log.info("FINISH SWAlgorithmTask::report()");
    return ReportResult_Finished;
}

} //namespace
