/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "QtScriptDebugViewController.h"
#include "QtScriptDebugPlugin.h"

#include <core_api/AppContext.h>
#include <core_api/Settings.h>
#include <script/GScriptModuleRegistry.h>
#include <test_framework/GTest.h>
#include <test_framework/GTestFrameworkComponents.h>

#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>
#include <QtGui/QFormLayout>
#include <QtGui/QHBoxLayout>
#include <QtGui/QVBoxLayout>
#include <QtGui/QDialogButtonBox>
#include <QtGui/QFormLayout>
#include <QtGui/QLineEdit>
#include <QtGui/QMenu>
#include <QtGui/QToolBar>
#include <QtGui/QPushButton>

#include <QtGui> // TODO expand!
#include <QtScript>

#include <qscriptdebugger.h>
#include <qscriptenginedebuggerfrontend.h>
#include <qscriptdebuggercodewidget_p.h>
#include "editor/scriptedit.h"

/* TRANSLATOR GB2::TestViewController */

//todo: remember splitter geom

namespace GB2 {

#define SETTINGS_ROOT QString("qtscriptdebug/view/")

QScriptValue getChildren(QScriptContext *ctx, QScriptEngine *eng)
{
    QObject* object = ctx->argument(0).toQObject();
    QObjectList objCildrenList= object->children();    
    QScriptValue objCildrenListScripted=(eng->newArray());
    if(objCildrenList.size()!=0){
        
        for(int i=0;i<objCildrenList.size();i++){
            QScriptValue scriptTmp = eng->newQObject(objCildrenList.value(i));
            objCildrenListScripted.setProperty((new QString())->setNum(i),scriptTmp);
        }
    }
    eng->globalObject().setProperty(QString("childrens_of_"+object->objectName()), objCildrenListScripted);
    int result=0;
    return QScriptValue(eng, result);
}


QtScriptDebugViewController::QtScriptDebugViewController() : MWMDIWindow(tr("Script Debugger -*"))
{
    testId=0;
    scriptFile=NULL;
    engine=NULL;
    isDebuging=false;

    jsTestHelperLib=new QFile();
    jsTestHelperLib->setFileName(QApplication::applicationDirPath()+"/scripts/libTestHelper.js");
    if(!jsTestHelperLib->exists()) {
        jsTestHelperLib->setFileName(QApplication::applicationDirPath()+"/../../scripts/libTestHelper.js");
    }
    assert(jsTestHelperLib->exists());

    debugger=new QScriptDebugger(this);
    debuggerFrontend=new QScriptEngineDebuggerFrontend();    

    debugger->setFrontend(debuggerFrontend);
    engine=initScriptEngine();
    debuggerFrontend->attachTo(engine);

    setupUi(this);
    loadScriptFileAction = new QAction(tr("load_script_file"), this);
    connect(loadScriptFileAction, SIGNAL(triggered()), SLOT(sl_loadScriptFileAction()));

    runScriptFileAction = new QAction(tr("run_script_file"), this);
    QKeySequence runAllKS(Qt::CTRL+Qt::Key_R);

    saveScriptFileAction = new QAction(tr("save_script_file"), this);
    connect(saveScriptFileAction, SIGNAL(triggered()), SLOT(sl_saveScriptFileAction()));

    saveAsScriptFileAction = new QAction(tr("save_as_script_file"), this);
    connect(saveAsScriptFileAction, SIGNAL(triggered()), SLOT(sl_saveAsScriptFileAction()));

    startDebugScriptFileAction = new QAction(tr("start_debug_script_file"), this);
    connect(startDebugScriptFileAction, SIGNAL(triggered()), SLOT(sl_startDebugScriptFileAction()));
    startDebugScriptFileAction->setEnabled(false);

    stopDebugScriptFileAction = new QAction(tr("stop_debug_script_file"), this);
    connect(stopDebugScriptFileAction, SIGNAL(triggered()), SLOT(sl_stopDebugScriptFileAction()));

    runScriptFileAction->setShortcut(runAllKS);
    connect(runScriptFileAction, SIGNAL(triggered()), SLOT(sl_runScriptFileAction()));

    this->rightLayout->insertWidget(1,debugger->scriptsWidget());
    debugger->scriptsWidget()->setFixedHeight(100);
    this->rightLayout->insertWidget(3,debugger->stackWidget());
    this->rightLayout->insertWidget(5,debugger->localsWidget());
    this->rightLayout->insertWidget(7,debugger->breakpointsWidget());
    this->leftLayout->addWidget(debugger->codeWidget());
    this->layout()->addWidget(debugger->consoleWidget());

    //hard code for set normal content margins
    QVBoxLayout* vl=debugger->scriptsWidget()->findChild<QVBoxLayout*>("");
    assert(vl!=NULL);
    vl->setContentsMargins(0,0,0,0);
    vl=debugger->stackWidget()->findChild<QVBoxLayout*>("");
    assert(vl!=NULL);
    vl->setContentsMargins(0,0,0,0);
    vl=debugger->breakpointsWidget()->findChild<QVBoxLayout*>("");
    assert(vl!=NULL);
    vl->setContentsMargins(0,0,0,0);
    vl=debugger->codeWidget()->findChild<QVBoxLayout*>("");
    assert(vl!=NULL);
    vl->setContentsMargins(0,0,0,0);
    vl=debugger->consoleWidget()->findChild<QVBoxLayout*>("");
    assert(vl!=NULL);
    vl->setContentsMargins(0,0,0,0);
    //End of hard code for set normal content margins

    codeWidget=(QScriptDebuggerCodeWidget*)debugger->codeWidget();

    codeWidget->addScript(0,"new script",1,"");
    connect(codeWidget->currentEditor(),SIGNAL(textChanged()),this,SLOT(sl_textChanged())); 

    QTimer* timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(sl_checkUncaughtExceptions()));
    timer->start(100);   
    runEngine=NULL;
    updateState();
}

QScriptEngine* QtScriptDebugViewController::initScriptEngine(){
    QScriptEngine* engine=new QScriptEngine(this);

    QMainWindow* mw= AppContext::getMainWindow()->getQMainWindow();
    assert(mw!=NULL);
    QScriptValue scriptMainWindow = engine->newQObject(mw);
    engine->globalObject().setProperty("main_window", scriptMainWindow);

    QScriptValue scriptFunctionGetChildren = engine->newFunction(getChildren);
    engine->globalObject().setProperty("getChildren", scriptFunctionGetChildren);  

    AppContext::setupToEngine(engine);
    engine->importExtension( "qt.core" );
    engine->importExtension( "qt.gui" );
    engine->importExtension( "qt.xml" );    

    jsTestHelperLib->open(QIODevice::ReadOnly | QIODevice::Text);
    engine->evaluate(jsTestHelperLib->readAll());
    jsTestHelperLib->close();
    //GTestScriptWrapper::setQTest(engine);
    AppContext::getScriptModuleRegistry()->getModuleByName("QTest")->setup(engine);
    engine->setProcessEventsInterval(100);
    return engine;
 
}

bool QtScriptDebugViewController::onCloseEvent() {
    //it does not work...
    if (isDebuging) sl_stopDebugScriptFileAction();

    if (engine!=NULL){
        debuggerFrontend->detach();
        delete engine;
        engine=NULL;
    }
    if (runEngine!=NULL){
        delete runEngine;
    }

    return true;
}

void QtScriptDebugViewController::setupMDIToolbar(QToolBar* tb) {
    tb->addAction(runScriptFileAction);
    tb->addAction(loadScriptFileAction);
    tb->addAction(saveScriptFileAction);
    tb->addAction(saveAsScriptFileAction);
    tb->addSeparator();
    tb->addAction(debugger->runToCursorAction());
    tb->addAction(startDebugScriptFileAction);
    tb->addAction(debugger->continueAction());
    tb->addAction(debugger->breakAction());
    tb->addAction(stopDebugScriptFileAction);    
    tb->addSeparator();
    tb->addAction(debugger->stepOverAction());
    tb->addAction(debugger->stepIntoAction());
    tb->addAction(debugger->stepOutAction());
    
}

void QtScriptDebugViewController::updateState() {
    if (runEngine!=NULL){
        runScriptFileAction->setEnabled((!codeWidget->currentEditor()->toPlainText().isEmpty())&&(!isDebuging)&&(!runEngine->isEvaluating()));
    }else{
        runScriptFileAction->setEnabled((!codeWidget->currentEditor()->toPlainText().isEmpty())&&(!isDebuging));
    }
    loadScriptFileAction->setEnabled(!isDebuging);
    saveScriptFileAction->setEnabled((scriptFile!=NULL)&&(!isDebuging)&&(isScriptChanged));
    saveAsScriptFileAction->setEnabled((!isDebuging)&&(isScriptChanged));

    stopDebugScriptFileAction->setEnabled(isDebuging);

    debugger->stepOverAction()->setEnabled(isDebuging);
    debugger->stepIntoAction()->setEnabled(isDebuging);
    debugger->stepOutAction()->setEnabled(isDebuging);
    debugger->continueAction()->setEnabled(isDebuging);
    debugger->breakAction()->setEnabled(isDebuging);

    if (codeWidget->currentEditor()!=NULL){
        codeWidget->currentEditor()->setReadOnly(isDebuging);
        startDebugScriptFileAction->setEnabled((codeWidget->currentEditor()->toPlainText().size()!=0)&&(!isDebuging));
    }
    
    debugger->runToCursorAction()->setEnabled(false);
}


void QtScriptDebugViewController::sl_loadScriptFileAction() {
    QString dir = AppContext::getSettings()->getValue(SETTINGS_ROOT + "lastDir", QString()).toString();
    QStringList fileNames=QFileDialog::getOpenFileNames(this, tr("select_ts_caption"),dir,tr("*.qs | *.js"));
    foreach(QString file,fileNames){
        if (file.isEmpty()) {
            return;
        }
        QFileInfo fi(file);
        dir = fi.absoluteDir().absolutePath();
        AppContext::getSettings()->setValue(SETTINGS_ROOT + "lastDir", dir);
        QString url = fi.absoluteFilePath();
        scriptFile=new QFile(url);
        if (scriptFile!=NULL){
            sl_updateTitle();
            scriptFile->open(QIODevice::ReadOnly | QIODevice::Text);
            testId++;
            codeWidget->addScript(testId,scriptFile->fileName(),1,scriptFile->readAll());
            codeWidget->setCurrentScript(testId);
            scriptFile->close();
            connect(codeWidget->currentEditor(),SIGNAL(textChanged()),this,SLOT(sl_textChanged())); 
        }
    }
    updateState();
    saveScriptFileAction->setEnabled(false);
}

void QtScriptDebugViewController::sl_updateTitle() {
    setWindowName(tr("Script Debugger - %1").arg(scriptFile->fileName()));    
}

void QtScriptDebugViewController::sl_textChanged(){
    isScriptChanged=true;
    saveScriptFileAction->setEnabled(isScriptChanged);
    saveAsScriptFileAction->setEnabled(isScriptChanged);
    updateState();
}

void QtScriptDebugViewController::sl_saveScriptFileAction() {
    scriptFile->open(QIODevice::WriteOnly|QIODevice::Text);
    QTextStream out(scriptFile);
    out << codeWidget->currentEditor()->toPlainText();
    scriptFile->close();
    isScriptChanged=false;
    sl_updateTitle();
    updateState();
}
void QtScriptDebugViewController::sl_saveAsScriptFileAction() {

    QString dir = AppContext::getSettings()->getValue(SETTINGS_ROOT + "lastDir", QString()).toString();
    QString file = QFileDialog::getSaveFileName(this, tr("select_save_path"), dir,tr("*.qs | *.js"));
    if (file.isEmpty()) {
        return;
    }
    else{
        if(codeWidget->currentEditor()->toPlainText().isEmpty()) return;
        scriptFile=new QFile(file);
        if (scriptFile!=NULL){
            scriptFile->open(QIODevice::WriteOnly|QIODevice::Text);
            QTextStream out(scriptFile);
            out << codeWidget->currentEditor()->toPlainText();
            scriptFile->close();
            isScriptChanged=false;            
        }
    }
    sl_updateTitle();
    updateState();
}

void QtScriptDebugViewController::sl_checkUncaughtExceptions(){
    if (runEngine!=NULL){
        if(runEngine->isEvaluating()){
            qDebug()<<"runEngine is evaluating...";
        }
        if(runEngine->hasUncaughtException()){
            qDebug()<<"runEngine have UncaughtException";
            //runEngine->clearExceptions();
        }
    }
    if (engine!=NULL)
    if((!isDebuging)&&(engine->hasUncaughtException())){
        qDebug()<<"Debuged engine have UncaughtException";
        debuggerFrontend->scheduleContinue();
        engine->clearExceptions();
    }
    
}
void QtScriptDebugViewController::sl_runScriptFileAction() {

    if (isScriptChanged) {
        if (scriptFile!=NULL) {
            sl_saveScriptFileAction();
        } else {
            sl_saveAsScriptFileAction();
        }
    }
    if(scriptFile==NULL) {
        return;
    }
    updateState();
    
    if(runEngine!=NULL) {
        if(runEngine->isEvaluating()){
            qDebug()<<"runEngine is evaluating...";
        }
        if(runEngine->hasUncaughtException()){
            qDebug()<<"runEngine have UncaughtException";
            runEngine->clearExceptions();
        }
    } else {
        runEngine=initScriptEngine();
    }

    scriptFile->open(QIODevice::ReadOnly | QIODevice::Text);
    qDebug()<<"evaluate = "<< (runEngine->evaluate(scriptFile->readAll(),"my program")).toString();
    scriptFile->close();
    updateState();
}

void QtScriptDebugViewController::sl_startDebugScriptFileAction() {

    assert(engine!=NULL);
    isDebuging=true;
    updateState();
    debuggerFrontend->breakAtFirstStatement();
    qDebug()<<codeWidget->currentEditor()->toPlainText();
    engine->evaluate(codeWidget->currentEditor()->toPlainText(),scriptFile->fileName(),1);

    //it is no good... but it work...
    connect(codeWidget->currentEditor(),SIGNAL(textChanged()),this,SLOT(sl_textChanged())); 
}

void QtScriptDebugViewController::sl_stopDebugScriptFileAction() {

    assert(engine!=NULL);

    if(engine->hasUncaughtException()){
            qDebug()<<"engine have UncaughtException";
            engine->clearExceptions();
    }else
        engine->abortEvaluation();
    debuggerFrontend->scheduleContinue();
    isDebuging=false;    
    updateState();    
}


}//namespace
