/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_services/AppContextImpl.h>
#include <core_services/SettingsImpl.h>
#include <core_services/DocumentFormatRegistryImpl.h>
#include <core_services/IOAdapterRegistryImpl.h>

#include "plugin_support/PluginSupportImpl.h"
#include "plugin_support/ServiceRegistryImpl.h"
#include "plugin_viewer/PluginViewerImpl.h"
#include "project_support/ProjectLoaderImpl.h"
#include "main_window/MainWindowImpl.h"
#include "project_view/ProjectViewImpl.h"
#include "logview/LogCache.h"
#include "logview/LogView.h"
#include <task_scheduler/TaskSchedulerImpl.h>
#include "task_view/TaskViewController.h"
#include "script_registry/ScriptManagerView.h"
#include <core_services/AppSettingsImpl.h>
#include "app_settings/AppSettingsGUIImpl.h"

#include <core_api/Log.h>
#include <core_api/Timer.h>
#include <core_api/DNATranslation.h>
#include <core_api/ObjectViewModel.h>
#include <core_api/ResourceTracker.h>
#include <core_api/DocumentFormatConfigurators.h>
#include <core_api/ScriptRegistry.h>
#include <core_api/DBXRefRegistry.h>
#include <core_api/UserApplicationsSettings.h>
#include <core_api/SecStructPredictAlgRegistry.h>
#include <core_api/CudaGpuRegistry.h>
#include <script/GScriptRunner.h>
#include <script/GScriptModuleRegistry.h>

#include <core_api/SubstMatrixRegistry.h>
#include <core_api/SWResultFilterRegistry.h>
#include <core_api/SmithWatermanTaskFactoryRegistry.h>
#include <molecular_geometry/MolecularSurfaceFactoryRegistry.h>

#include <document_format/DNAAlphabetRegistryImpl.h>
#include <gobjects/AnnotationSettings.h>
#include <gobjects/GObjectTypes.h>
#include <test_framework/GTestFrameworkComponents.h>
#include <util_gui/BaseDocumentFormatConfigurators.h>
#include <util_ov_msaedit/MSAColorScheme.h>

#include <workflow_support/WorkflowEnvImpl.h>
#include <workflow_library/BioActorLibrary.h>
#include <util_tasks/TaskStarter.h>
#include <util_tasks/LoadRemoteDocumentTask.h>
#include <util_index_support/UIndexSupport.h>

#include <QtGui/QApplication>
#include <QtGui/QIcon>

static GB2::LogCategory logCat(ULOG_CAT_USER_INTERFACE);

/* TRANSLATOR GB2::AppContextImpl */

#if defined(Q_WS_WIN) || defined(Q_WS_MAC)
QT_BEGIN_NAMESPACE
extern Q_GUI_EXPORT bool qt_use_native_dialogs;
QT_END_NAMESPACE
#endif

namespace GB2 {

static void registerCoreServices() {
    ServiceRegistry* sr = AppContext::getServiceRegistry();
    TaskScheduler* ts = AppContext::getTaskScheduler();
    ts->registerTopLevelTask(sr->registerServiceTask(new PluginViewerImpl()));
    ts->registerTopLevelTask(sr->registerServiceTask(new ProjectViewImpl()));
    ts->registerTopLevelTask(sr->registerServiceTask(new ScriptRegistryService()));
}

}//namespace

using namespace GB2;

static QString openDocs(int argc, char **argv) {
    QList<QUrl> urls;
    QString scriptFile;
    for (int i = 0; i < argc; ++i) {
        QString str = QString::fromLocal8Bit(argv[i]);
        if (str.endsWith(".js",Qt::CaseInsensitive)){
            scriptFile=str;
        }else
            urls << BaseIOAdapters::str2url(str);
    }
    if(urls.isEmpty())
        return scriptFile;
    Task* t = AppContext::getProjectLoader()->openProjectTask(urls, false);
    // defer loading until all plugins/services loaded
    QObject::connect(AppContext::getPluginSupport(), SIGNAL(si_allStartUpPluginsLoaded()), 
        new TaskStarter(t), SLOT(registerTask()));
    return scriptFile;
}


static void updateStaticTranslations() {
    GObjectTypes::initTypeTranslations();
    GObjectTypes::initTypeIcons();
}

static void setSearchPaths() {
    //set search paths for data files
    QStringList dataSearchPaths;
#if (defined(Q_OS_LINUX) || defined(Q_OS_UNIX)) && defined( UGENE_DATA_DIR )
    //using directory which is set during installation process on linux
    QString ugene_data_dir( UGENE_DATA_DIR );
    if( QDir(ugene_data_dir).exists() ) {
        dataSearchPaths.push_back( QString(UGENE_DATA_DIR) );
    }
#endif
    const static char * RELATIVE_DATA_DIR = "/data";
    const static char * RELATIVE_DEV_DATA_DIR = "/../../data";
    //on windows data is normally located in the application directory
    QString appDirPath = QCoreApplication::applicationDirPath();
    if( QDir(appDirPath+RELATIVE_DATA_DIR).exists() ) {
        dataSearchPaths.push_back( appDirPath+RELATIVE_DATA_DIR );
    } else if( QDir(appDirPath+RELATIVE_DEV_DATA_DIR).exists() ) {          //data location for developers
        dataSearchPaths.push_back( appDirPath+RELATIVE_DEV_DATA_DIR );
    }
    if( dataSearchPaths.empty() ) {
        dataSearchPaths.push_back("/");
    }
    QDir::setSearchPaths( PATH_PREFIX_DATA, dataSearchPaths );
    //now data files may be opened using QFile( "data:some_data_file" )
} 

int main(int argc, char **argv) 
{
    QT_REQUIRE_VERSION( argc, argv, QT_VERSION_STR );

    GTIMER(c1, t1, "main()->QApp::exec");

    QApplication app(argc, argv);
    QCoreApplication::addLibraryPath(QCoreApplication::applicationDirPath());
    QString devPluginsPath = QDir(QCoreApplication::applicationDirPath()+"/../../installer/windows").absolutePath();
    QCoreApplication::addLibraryPath(devPluginsPath); //dev version

    setSearchPaths();

    AppContextImpl* appContext = AppContextImpl::getApplicationContext();

    //1 create    
    SettingsImpl* globalSettings = new SettingsImpl(QSettings::SystemScope);
    appContext->setGlobalSettings(globalSettings);

	SettingsImpl* settings = new SettingsImpl(QSettings::UserScope);
	appContext->setSettings(settings);

    AppSettings* appSettings = new AppSettingsImpl();
    appContext->setAppSettings(appSettings);

    UserAppsSettings* userAppSettings = AppContext::getAppSettings()->getUserAppsSettings();

	QTranslator translator;
    QString transFile[] = {
        userAppSettings->getTranslationFile(),
        "transl_" + QLocale::system().name(),
        "transl_en"
    };
    bool trOK = false;
    for (int i = transFile[0].isEmpty() ? 1 : 0; i < 3; ++i) {
        if (!translator.load(transFile[i], QCoreApplication::applicationDirPath())) {
            fprintf(stderr, "Translation not found: %s\n", transFile[i].toAscii().constData());
        } else {
            trOK = true;
            break;
        }
    }
    if (!trOK) {
        fprintf(stderr, "No translations found, exiting\n");
	    return 1;   
	}
	
	app.installTranslator(&translator);
    updateStaticTranslations();
    
    AppContext::getSettings()->setValue(GLOBAL_SETTINGS + "drawIcons", true);
    
    QStringList envList = QProcess::systemEnvironment();

    LogCache logsCache;
#ifdef _DEBUG
    logsCache.print_to_console=true;
#else
    logsCache.print_to_console = envList.indexOf("UGENE_PRINT_TO_CONSOLE=true") >= 0 || envList.indexOf("UGENE_PRINT_TO_CONSOLE=1") >= 0;
#endif

    logCat.details(AppContextImpl::tr("UGENE initialization started"));

#if defined(Q_WS_WIN) || defined(Q_WS_MAC)
    qt_use_native_dialogs = envList.indexOf("UGENE_USE_NATIVE_DIALOGS=false") < 0 && envList.indexOf("UGENE_USE_NATIVE_DIALOGS=0") < 0;
#endif

    QString style = userAppSettings->getVisualStyle();
    if (!style.isEmpty()) {
        QStyle* qstyle = QStyleFactory::create(style);
        if (qstyle!=NULL) {
            app.setStyle(qstyle);
        } else {
            logCat.details(AppContextImpl::tr("Style not available %1").arg(style));
        }
    } 
    
    GScriptModuleRegistry* smreg=new GScriptModuleRegistry();
    appContext->setScriptModuleRegistry(smreg);

    ResourceTracker* resTrack = new ResourceTracker();
    appContext->setResourceTracker(resTrack);

    TaskSchedulerImpl* ts = new TaskSchedulerImpl(appSettings->getAppResourcePool());
	appContext->setTaskScheduler(ts);

    AnnotationSettingsRegistry* asr = new AnnotationSettingsRegistry();
    appContext->setAnnotationSettingsRegistry(asr);

    TestFramework* tf = new TestFramework();
    appContext->setTestFramework(tf);

    MainWindowImpl* mw = new MainWindowImpl();
	appContext->setMainWindow(mw);
	mw->show();

    AppSettingsGUI* appSettingsGUI = new AppSettingsGUIImpl();
    appContext->setAppSettingsGUI(appSettingsGUI);

    AppContext::getMainWindow()->getDockManager()->registerDock(MWDockArea_Bottom, new TaskViewDockWidget(), QKeySequence(Qt::ALT | Qt::Key_2));
    AppContext::getMainWindow()->getDockManager()->registerDock(MWDockArea_Bottom, new LogViewDockWidget(&logsCache), QKeySequence(Qt::ALT | Qt::Key_3));
    
    GObjectViewFactoryRegistry* ovfr = new GObjectViewFactoryRegistry();
    appContext->setObjectViewFactoryRegistry(ovfr);

    PluginSupportImpl* psp = new PluginSupportImpl();
    appContext->setPluginSupport(psp);

    ServiceRegistryImpl* sreg = new ServiceRegistryImpl() ;
    appContext->setServiceRegistry(sreg);

	DocumentFormatRegistryImpl* dfr = new DocumentFormatRegistryImpl();
	appContext->setDocumentFormatRegistry(dfr);

    DocumentFormatConfigurators* dfc = new DocumentFormatConfigurators();
    appContext->setDocumentFormatConfigurators(dfc);
    BaseDocumentFormatConfigurators::initBuiltInConfigurators();
    

	IOAdapterRegistryImpl* io = new IOAdapterRegistryImpl();
	appContext->setIOAdapterRegistry(io);

	DNATranslationRegistry* dtr = new DNATranslationRegistry();
	appContext->setDNATranslationRegistry(dtr);

	DNAAlphabetRegistry* dal = new DNAAlphabetRegistryImpl(dtr);
	appContext->setDNAAlphabetRegistry(dal);

	ProjectLoaderImpl* pli = new ProjectLoaderImpl();
	appContext->setProjectLoader(pli);

    ScriptManagerView * smv = new ScriptManagerView();

	DBXRefRegistry* dbxr = new DBXRefRegistry();
	appContext->setDBXRefRegistry(dbxr);

    MSAColorSchemeRegistry* mcsr = new MSAColorSchemeRegistry();
    appContext->setMSAColorSchemeRegistry(mcsr);

	SubstMatrixRegistry* smr = new SubstMatrixRegistry();
	appContext->setSubstMatrixRegistry(smr);

	SmithWatermanTaskFactoryRegistry* swar = new SmithWatermanTaskFactoryRegistry();
	appContext->setSmithWatermanTaskFactoryRegistry(swar);

    MolecularSurfaceFactoryRegistry* msfr = new MolecularSurfaceFactoryRegistry();
    appContext->setMolecularSurfaceFactoryRegistry(msfr);

    SWResultFilterRegistry* swrfr = new SWResultFilterRegistry();
	appContext->setSWResultFilterRegistry(swrfr);

    SecStructPredcitAlgRegistry* sspar = new SecStructPredcitAlgRegistry();
    appContext->setSecStructPedictAlgRegistry(sspar);

    CudaGpuRegistry * cgr = new CudaGpuRegistry();
    appContext->setCudaGpuRegistry( cgr ); 

    RecentlyDownloadedCache* rdc = new RecentlyDownloadedCache();
    appContext->setRecentlyDownloadedCache( rdc );

    Workflow::WorkflowEnv::init(new Workflow::WorkflowEnvImpl());
    Workflow::BioActorLibrary::init();
    
    UIndexSupport indexSupport;
    
    QString startupScript;
    if (argc > 1) {
        startupScript = openDocs(argc - 1, argv + 1);
    } else if (AppContext::getAppSettings()->getUserAppsSettings()->openLastProjectAtStartup()) {
        QString lastProject = ProjectLoaderImpl::getLastProjectURL();
        if (!lastProject.isEmpty()) {
            Task* t = AppContext::getProjectLoader()->openProjectTask(lastProject, false);
            // defer project start until all plugins/services loaded
            QObject::connect(AppContext::getPluginSupport(), SIGNAL(si_allStartUpPluginsLoaded()), 
                new TaskStarter(t), SLOT(registerTask()));
        }
    }
    
    GScriptRunner* sr = new GScriptRunner(startupScript);
    appContext->setScriptRunner(sr);

    registerCoreServices();

    //2 run QT GUI
    t1.stop();
    logCat.info(AppContextImpl::tr("UGENE started"));
    int rc = app.exec();

    //3. deallocate resources
    Workflow::WorkflowEnv::shutdown();
    delete smv;
   
    delete rdc;
    appContext->setRecentlyDownloadedCache(NULL);

    delete swrfr;
	appContext->setSWResultFilterRegistry(NULL);

	delete swar;
	appContext->setSmithWatermanTaskFactoryRegistry(NULL);

        delete msfr;
        appContext->setMolecularSurfaceFactoryRegistry(NULL);

    delete sr;
	appContext->setScriptRunner(NULL);

	delete smr;
	appContext->setSubstMatrixRegistry(NULL);

    delete mcsr;
    appContext->setMSAColorSchemeRegistry(NULL);

	delete dbxr;
    appContext->setDBXRefRegistry(NULL);

    delete pli;
	appContext->setProjectLoader(NULL);

    delete sreg;
    appContext->setServiceRegistry(NULL);

	delete psp;
    appContext->setPluginSupport(NULL);

	delete mw;
	appContext->setMainWindow(NULL);

    delete tf;
    appContext->setTestFramework(0);

	delete ovfr;
	appContext->setObjectViewFactoryRegistry(NULL);

	delete dal;
	appContext->setDNAAlphabetRegistry(NULL);

	delete dtr;
	appContext->setDNATranslationRegistry(NULL);

	delete io;
	appContext->setIOAdapterRegistry(NULL);

    delete dfc;
    appContext->setDocumentFormatConfigurators(NULL);

    delete dfr;
	appContext->setDocumentFormatRegistry(NULL);

	delete ts;
	appContext->setTaskScheduler(NULL);

	delete asr;
    appContext->setAnnotationSettingsRegistry(NULL);

    delete resTrack;
    appContext->setResourceTracker(NULL);

    delete cgr;
    appContext->setCudaGpuRegistry(NULL);

    delete appSettingsGUI;
    appContext->setAppSettingsGUI(NULL);

    delete appSettings;
    appContext->setAppSettings(NULL);

    delete settings;
    appContext->setSettings(NULL);

	delete globalSettings;
	appContext->setGlobalSettings(NULL);
	
    delete smreg;
    appContext->setScriptModuleRegistry(NULL);
    
    delete sspar;
    appContext->setSecStructPedictAlgRegistry(NULL);
    
	return rc;
}

