/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "UserApplicationsSettingsGUIController.h"

#include <core_api/AppContext.h>
#include <core_api/AppSettings.h>
#include <core_api/UserApplicationsSettings.h>
#include <core_api/Log.h>

#include <QtCore/QFile>
#include <QtGui/QFileDialog>
#include <QtGui/QStyleFactory>

namespace GB2
{
static LogCategory log(ULOG_CAT_USER_INTERFACE);
#define TRANSMAP_FILE_NAME "translations.txt"

UserApplicationsSettingsPageController::UserApplicationsSettingsPageController(QObject* p) 
: AppSettingsGUIPageController(tr("user_apps"), APP_SETTINGS_USER_APPS, p)
{
    translations.insert(tr("Autodetection"), "");
    QFile file( QString(PATH_PREFIX_DATA)+ ":" + TRANSMAP_FILE_NAME );

    if(!file.exists() || !file.open(QIODevice::ReadOnly)){
        log.error(tr("transmap_file_not_found_%1").arg(TRANSMAP_FILE_NAME));
        translations.insert("English", "transl_en");
        return;
    } 
    QTextStream in(&file);
    in.setCodec("UTF-8");
    in.setAutoDetectUnicode(true);
    while (!in.atEnd()) {
        QString line = in.readLine();
        if(line.length()==0 || line.startsWith("#")) {
            continue;
        }
        QStringList fields = line.split("|");
        if (fields.size()!=2) {
            log.error(tr("illegal_entry_%1").arg(line));
            continue;
        }
        translations.insert(fields[0], fields[1].trimmed());
    }
    file.close();
}


AppSettingsGUIPageState* UserApplicationsSettingsPageController::getSavedState() {
    UserApplicationsSettingsPageState* state = new UserApplicationsSettingsPageState();
    UserAppsSettings* s = AppContext::getAppSettings()->getUserAppsSettings();
    state->webBrowserUrl = s->getWebBrowserURL();
    state->translFile = s->getTranslationFile();
    state->useDefaultWebBrowser = s->useDefaultWebBrowser();
    state->openLastProjectFlag = s->openLastProjectAtStartup();
    state->style = s->getVisualStyle();
    state->downloadsDirPath = s->getDownloadDirPath();
    return state;
}

void UserApplicationsSettingsPageController::saveState(AppSettingsGUIPageState* s) {
    UserApplicationsSettingsPageState* state = qobject_cast<UserApplicationsSettingsPageState*>(s);
    UserAppsSettings* st = AppContext::getAppSettings()->getUserAppsSettings();
    st->setWebBrowserURL(state->webBrowserUrl);
    st->setTranslationFile(state->translFile);
    st->setUseDefaultWebBrowser(state->useDefaultWebBrowser);
    st->setOpenLastProjectAtStartup(state->openLastProjectFlag);
    st->setDownloadDirPath(state->downloadsDirPath);
    
    QStyle* style = QStyleFactory::create(state->style);
    if (style!=NULL) {
        QApplication::setStyle(style);
        st->setVisualStyle(state->style);
    }
}

AppSettingsGUIPageWidget* UserApplicationsSettingsPageController::createWidget(AppSettingsGUIPageState* state) {
    UserApplicationsSettingsPageWidget* r = new UserApplicationsSettingsPageWidget(this);
    r->setState(state);
    return r;
}

UserApplicationsSettingsPageWidget::UserApplicationsSettingsPageWidget(UserApplicationsSettingsPageController* ctrl) {
    setupUi(this);
    connect(webBrowserButton, SIGNAL(clicked()), SLOT(sl_wbURLClicked()));
    connect(langButton, SIGNAL(clicked()), SLOT(sl_transFileClicked()));
    connect(browseDownloadDirButton, SIGNAL(clicked()), SLOT(sl_browseButtonClicked()));
    
    QMapIterator<QString, QString> it(ctrl->translations);
    while (it.hasNext()) {
        it.next();
        langCombo->addItem(it.key(), it.value());
    }
    styleCombo->addItems(QStyleFactory::keys());
}

void UserApplicationsSettingsPageWidget::setState(AppSettingsGUIPageState* s) {
    UserApplicationsSettingsPageState* state = qobject_cast<UserApplicationsSettingsPageState*>(s);
    defaultWebBrowser->setChecked(state->useDefaultWebBrowser);
    customWebBrowser->setChecked(!state->useDefaultWebBrowser);
    webBrowserButton->setEnabled(!state->useDefaultWebBrowser);
    webBrowserEdit->setEnabled(!state->useDefaultWebBrowser);
    webBrowserEdit->setText(state->webBrowserUrl);
    downloadsDirPathEdit->setText(state->downloadsDirPath);
    int idx = langCombo->findData(state->translFile);
    if (idx < 0) {
        idx = langCombo->count();
        langCombo->addItem(tr("Custom translation [%1]").arg(state->translFile), state->translFile);
    }
    langCombo->setCurrentIndex(idx);
    
    int styleIdx = styleCombo->findText(state->style, Qt::MatchFixedString); //case insensitive
    if (styleIdx!=-1) {
        styleCombo->setCurrentIndex(styleIdx);
    }

    autoOpenProjectBox->setChecked(state->openLastProjectFlag);
    
}

AppSettingsGUIPageState* UserApplicationsSettingsPageWidget::getState(QString& err) const {
    UserApplicationsSettingsPageState* state = new UserApplicationsSettingsPageState();
    if (defaultWebBrowser->isChecked()){
        state->useDefaultWebBrowser=true;
    } else {
        QString wbUrl = webBrowserEdit->text();
        QFile wbFile(wbUrl);
        if (!wbFile.exists()) {
            webBrowserEdit->setFocus();
            err = tr("file_not_exists");
            return NULL;
        }		
        state->webBrowserUrl = wbUrl;
        state->useDefaultWebBrowser=false;
    }
    state->translFile = langCombo->itemData(langCombo->currentIndex()).toString();
    state->openLastProjectFlag = autoOpenProjectBox->isChecked();
    state->style = styleCombo->currentText();
    state->downloadsDirPath = downloadsDirPathEdit->text();

    return state;
}

void UserApplicationsSettingsPageWidget::sl_wbURLClicked() {
    QString file = QFileDialog::getOpenFileName(this, tr("select_wb_file_title"), QString(), 0);
    if (!file.isEmpty()) {
        webBrowserEdit->setText(file);
    }
}

void UserApplicationsSettingsPageWidget::sl_transFileClicked() {
    QString file = QFileDialog::getOpenFileName(this, tr("select_trans_file_title"), QString(), 0);
    if (!file.isEmpty()) {
        QFileInfo fi(file);
        int idx = langCombo->findData(fi.baseName());
        if (idx < 0) {
            idx = langCombo->findData(fi.fileName());
        }
        if (idx < 0) {
            idx = langCombo->findData(fi.absoluteFilePath());
        }
        if (idx < 0) {
            idx = langCombo->count();
            langCombo->addItem(tr("Custom translation [%1]").arg(file), fi.absoluteFilePath());
        }
        langCombo->setCurrentIndex(idx);
    }
}

void UserApplicationsSettingsPageWidget::sl_browseButtonClicked()
{
    QString path = downloadsDirPathEdit->text();
    QString dir = QFileDialog::getExistingDirectory(this, tr("Choose Directory"), path,
        QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks);
    if (!dir.isEmpty()) {
        downloadsDirPathEdit->setText(dir);
    }
    
}
} //namespace
