/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DocumentFormatRegistryImpl.h"
#include <document_format/PlainTextFormat.h>
#include <document_format/FastaFormat.h>
#include <document_format/GenbankPlainTextFormat.h>
#include <document_format/EMBLPlainTextFormat.h>
#include <document_format/ABIFormat.h>
#include <document_format/SCFFormat.h>
#include <document_format/RawDNASequenceFormat.h>
#include <document_format/ClustalWAlnFormat.h>
#include <document_format/StockholmFormat.h>
#include <document_format/NewickFormat.h>
#include <document_format/PDBFormat.h>
#include <document_format/FastqFormat.h>
#include <document_format/IndexFormat.h>
#include <document_format/ASNFormat.h>
#include <document_format/MSFFormat.h>
#include <document_format/SRFastaFormat.h>
#include <document_format/GFFFormat.h>
#include <document_format/SAMFormat.h>

namespace GB2 {

bool DocumentFormatRegistryImpl::registerFormat(DocumentFormat* f) {
    assert(getFormatById(f->getFormatId())==NULL);
    formats.push_back(f);
    emit si_documentFormatRegistered(f);
    return true;
}

QList<DocumentFormatId> DocumentFormatRegistryImpl::getRegisteredFormats() const {
    QList<DocumentFormatId> ids;
    foreach(DocumentFormat* df, formats) {
        ids.append(df->getFormatId());
    }
    return ids;
}

DocumentFormat* DocumentFormatRegistryImpl::selectFormatByFileExtension(const QString& fileExt) const {
	foreach(DocumentFormat* df, formats) {
		if (df->getSupportedDocumentFileExtensions().contains(fileExt)) {
			return df;
		}
	}
	return NULL;
}

QList<DocumentFormatId> DocumentFormatRegistryImpl::selectFormats(const DocumentFormatConstraints& c) const {
    QList<DocumentFormatId> ids;
    foreach(DocumentFormat* df, formats) {
        if (df->checkConstraints(c)) {
            ids.append(df->getFormatId());
        }
    }
    return ids;
}

bool DocumentFormatRegistryImpl::unregisterFormat(DocumentFormat* f) {
    int n = formats.removeAll(f);
    bool res = n > 0;
    if (res) {
        emit si_documentFormatUnregistered(f);
    }
    return res;
}

DocumentFormat* DocumentFormatRegistryImpl::getFormatById(DocumentFormatId id) const {
    foreach (DocumentFormat* f, formats) {
        if (f->getFormatId() == id) {
            return f;
        }
    }
    return NULL;
}

static bool priorityComparator(const DocumentFormat* f1, const DocumentFormat* f2)  {
    int p1 = f1->getFormatDetectionPriority();
    int p2 = f2->getFormatDetectionPriority();
    return p2 < p1;// reverse sort -> make higher vals on the top
}

QList<DocumentFormatId> DocumentFormatRegistryImpl::sortByDetectPriority(const QList<DocumentFormatId>& ids) const {
    QList<DocumentFormat*> formats;
    foreach(const DocumentFormatId& id, ids) {
        DocumentFormat* f = getFormatById(id);
        assert(f!=NULL);
        formats.append(f);
    }
    qSort(formats.begin(), formats.end(), priorityComparator);
    
    QList<DocumentFormatId> res;
    foreach(DocumentFormat* f, formats) {
        res.append(f->getFormatId());
    }
    return res;
}

QList<DocumentFormat*> DocumentFormatRegistryImpl::sortByDetectPriority(const QList<DocumentFormat*>& formats) const {
    QList<DocumentFormat*> res = formats;
    qStableSort(res.begin(), res.end(), priorityComparator);
    return res;
}

void DocumentFormatRegistryImpl::init() {
    PlainTextFormat* text = new PlainTextFormat(this);
    registerFormat(text);   

    FastaFormat* fasta = new FastaFormat(this);
    registerFormat(fasta);  

    GenbankPlainTextFormat* gb = new GenbankPlainTextFormat(this);
    registerFormat(gb);

    EMBLPlainTextFormat* em = new EMBLPlainTextFormat(this);
    registerFormat(em);

    ABIFormat* abi = new ABIFormat(this);
    registerFormat(abi);

    SCFFormat* scf = new SCFFormat(this);
    registerFormat(scf);

    RawDNASequenceFormat* rsf = new RawDNASequenceFormat(this);
    registerFormat(rsf);

    ClustalWAlnFormat* aln = new ClustalWAlnFormat(this);
    registerFormat(aln);

    StockholmFormat* stf = new StockholmFormat(this);
    registerFormat(stf);

	NewickFormat* nwf = new NewickFormat(this);
	registerFormat(nwf);
    
    PDBFormat* pdb = new PDBFormat(this);
    registerFormat(pdb);

    FastqFormat* ftq = new FastqFormat(this);
    registerFormat(ftq);

    IndexFormat* indf = new IndexFormat(this);
    registerFormat(indf);

    ASNFormat* asn = new ASNFormat(this);
    registerFormat(asn);

    MSFFormat* msf = new MSFFormat(this);
    registerFormat(msf);

    GFFFormat *gff = new GFFFormat(this);
    registerFormat(gff);

    SRFastaFormat* srff = new SRFastaFormat(this);
    registerFormat(srff);

	SAMFormat *sam = new SAMFormat(this);
	registerFormat(sam);
}

}//namespace
