/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <cassert>

#include "RemoteTask.h"
#include "DistributedTask.h"

namespace GB2 {

DistributedTask::DistributedTask( const QString & i, const LocalTaskSettings & s, TaskDistributor * d, const QList<RemoteMachine*>& m )
: Task( "", TaskFlag_NoRun ), taskFactoryId( i ), settings( s ), distributor( d ), availableMachines( m ), result( NULL ) {
    
    if( taskFactoryId.isEmpty() ) {
        setTaskName( tr( "Distributed task" ) );
        setError( tr( "No task given to distributed task" ) );
        return;
    }
    setTaskName( tr( "Distributed '%1' task" ).arg( taskFactoryId ) );
    
    if( NULL == distributor ) {
        setError( tr( "No task distributor given to distributed task" ) );
        return;
    }
    if( availableMachines.isEmpty() ) {
        setError( tr( "No remote machines given to distributed task" ) );
        return;
    }
}

DistributedTask::~DistributedTask() {
    delete result;
    qDeleteAll( settingsToDel );
}

bool DistributedTask::scatterSettings() {
    assert( !hasErrors() );
    
    if( 1 == availableMachines.size() ) {
        subTasksSettings.append( (LocalTaskSettings*)&settings );
    } else {
        subTasksSettings = distributor->scatter( &settings );
        if( subTasksSettings.isEmpty() ) {
            setError( tr( "Cannot distribute given task" ) );
            return false;
        }
        settingsToDel = subTasksSettings; /* because they are newly allocated */
    }
    return true;
}

void DistributedTask::prepare() {
    if( hasErrors() ) {
        return;
    }
    
    if( !scatterSettings() ) {
        assert( hasErrors() );
        return;
    }
    
    while( !subTasksSettings.isEmpty() ) {
        if( availableMachines.isEmpty() ) {
            break;
        }
        RemoteMachine * machine = availableMachines.takeFirst();
        LocalTaskSettings * set = subTasksSettings.takeFirst();
        assert( NULL != machine && NULL != set );
        
        RemoteTask * remoteTask = new RemoteTask( taskFactoryId, *set, machine );
        addSubTask( remoteTask );
        runningTaskSettings.insert( remoteTask->getTaskId(), set );
    }
}

QList< Task* > DistributedTask::onSubTaskFinished( Task * subTask ) {
    assert( NULL != subTask );
    QList< Task* > res;
    if( hasErrors() ) {
        return res;
    }
    RemoteTask * remoteTask = qobject_cast< RemoteTask* >( subTask );
    if( NULL == remoteTask ) {
        setError( tr( "Undefined subtask finished in distributed task" ) );
        return res;
    }
    if( remoteTask->hasErrors() && !remoteTask->hasNetworkErrorOccurred() ) {
        setError( tr( "One of subtasks of distributed task finished with error: '%1'" ).arg( remoteTask->getError() ) );
        cancel(); /* to cancel all subtasks running on remote machines */
        return res;
    }
    
    /* if remote task is canceled by user or network problems detected -> we should give this task to another machine */
    if( remoteTask->isCanceled() || remoteTask->hasNetworkErrorOccurred() ) {
        LocalTaskSettings * set = runningTaskSettings.value( remoteTask->getTaskId(), NULL );
        assert( NULL != set );
        subTasksSettings.append( set );
    } else { /* machine is good and task not canceled and doesn't has errors*/
        assert( !remoteTask->hasErrors() );
        availableMachines.append( remoteTask->getRemoteMachine() );
        LocalTaskResult * result = remoteTask->getResult();
        if( NULL == result ) {
            setError( tr( "One of subtasks of distributed task returned null result" ) );
            return res;
        }
        subTasksResults << result;
    }
    runningTaskSettings.remove( remoteTask->getTaskId() );
    
    if( runningTaskSettings.isEmpty() && !subTasksSettings.isEmpty() && availableMachines.isEmpty() ) {
        setError( tr( "No available machines left for distributed task" ) );
        return res;
    }
    
    if( subTasksSettings.isEmpty() || availableMachines.isEmpty() ) {
        return res;
    }
    
    LocalTaskSettings * set = subTasksSettings.takeFirst();
    RemoteMachine * machine = availableMachines.takeFirst();
    assert( NULL != set && NULL != machine );
    RemoteTask * newRemoteTask = new RemoteTask( taskFactoryId, *set, machine );
    res << newRemoteTask;
    runningTaskSettings.insert( newRemoteTask->getTaskId(), set );
    
    return res;
}

Task::ReportResult DistributedTask::report() {
    if( hasErrors() ) {
        return ReportResult_Finished;
    }
    
    result = distributor->gather( subTasksResults );
    if( NULL == result ) {
        setError( tr( "Cannot gather results for distributed task" ) );
    }
    
    return ReportResult_Finished;
}

LocalTaskResult * DistributedTask::getResult() const {
    return result;
}

} // GB2
