/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AnnotationSettings.h"

#include <core_api/AppContext.h>
#include <core_api/Settings.h>
#include <datatype/BioStruct3D.h>
#include <document_format/GenbankFeatures.h>
#include <util_gui/GUIUtils.h>
#include <gobjects/AnnotationTableObject.h>

#include <QtCore/QSet>

namespace GB2 {

#define SETTINGS_ROOT QString("annotation_settings/")
#define MAX_CACHE_SIZE 1000

AnnotationSettingsRegistry::AnnotationSettingsRegistry() {
    read();
}

AnnotationSettingsRegistry::~AnnotationSettingsRegistry() {
    save();
    qDeleteAll(persistentMap);
    qDeleteAll(transientMap);
}

void AnnotationSettingsRegistry::changeSettings(const QList<AnnotationSettings*>& settings, bool saveAsPersistent) {
    if (settings.isEmpty()) {
        return;
    }
    QStringList changedNames;
    foreach(AnnotationSettings* s, settings) {
        assert(s->color.isValid());
        assert(!s->name.isEmpty());
        persistentMap.remove(s->name);
        transientMap.remove(s->name);
        if (saveAsPersistent) {
            persistentMap[s->name] = s;
        } else {
            transientMap[s->name] = s;
        }
        changedNames.append(s->name);
    }
    emit si_annotationSettingsChanged(changedNames);
}

QStringList AnnotationSettingsRegistry::getAllSettings() const {
    return (persistentMap.keys() + transientMap.keys()).toSet().toList();
}

AnnotationSettings* AnnotationSettingsRegistry::getAnnotationSettings(Annotation* a) {
    return getAnnotationSettings(a->getAnnotationName());
}

AnnotationSettings* AnnotationSettingsRegistry::getAnnotationSettings(const QString& name) {
    //Search in persistent settings:
    AnnotationSettings* s = persistentMap.value(name);
    if (s != NULL) {
        return s;
    }
    
    //search in transient cache:
    s = transientMap.value(name);
    if (s!=NULL){
        return s;
    }
    s = new AnnotationSettings();
    s->name = name;
    s->color = GUIUtils::genLightColor(name);
    s->visible = true;
    if (transientMap.size() == MAX_CACHE_SIZE) {
        //todo: mutex!?
        transientMap.erase(transientMap.begin());
    }
    transientMap[name]=s;
    return s;
}


void AnnotationSettingsRegistry::read() {
    addPredefined();

    Settings* s = AppContext::getSettings();
    QStringList keys = s->getAllKeys(SETTINGS_ROOT);
    QList<AnnotationSettings*> list;
    foreach(const QString& key, keys) {
        QString name = key.split('/').first();
        AnnotationSettings* as = transientMap.value(name);
        if (as == NULL) {
            as = new AnnotationSettings();
            as->name = name;
        }
        as->color = s->getValue(SETTINGS_ROOT + as->name + "/color", GUIUtils::genLightColor(as->name)).value<QColor>();
        as->visible = s->getValue(SETTINGS_ROOT + as->name + "/visible", true).toBool();
        as->amino = s->getValue(SETTINGS_ROOT + as->name + "/amino", true).toBool();
        QString qs = s->getValue(SETTINGS_ROOT + as->name + "/quals", "").toString();
        if (!qs.isEmpty()) {
            as->nameQuals = qs.split(',', QString::SkipEmptyParts);
        }
        list.append(as);
    }
    changeSettings(list, false);
}

void AnnotationSettingsRegistry::save() {
    Settings* s = AppContext::getSettings();
    QStringList keys = s->getAllKeys(SETTINGS_ROOT);
    foreach(const AnnotationSettings* as, persistentMap.values()) {
        s->setValue(SETTINGS_ROOT + as->name + "/color", as->color);
        s->setValue(SETTINGS_ROOT + as->name + "/visible", as->visible);
        s->setValue(SETTINGS_ROOT + as->name + "/amino", as->amino);
        s->setValue(SETTINGS_ROOT + as->name + "/quals", as->nameQuals.join(","));
    }
}

void AnnotationSettingsRegistry::addPredefined() {
    QList<AnnotationSettings*> predefined;
    foreach(GBFeatureKeyInfo fi, GBFeatureUtils::allKeys()) {
        AnnotationSettings* as = new AnnotationSettings();
        as->name = fi.text;
        as->amino = fi.showOnAminoStrand;
        as->color = fi.color;
        as->visible = as->name!="source";
        as->nameQuals = fi.namingQuals;
        predefined.append(as);
    }
    AnnotationSettings* secStructAnnotationSettings = new  AnnotationSettings(BioStruct3D::SecStructAnnotationTag, true, QColor(102,255, 0), true);
    secStructAnnotationSettings->nameQuals.append(BioStruct3D::SecStructTypeQualifierName);
    predefined.append(secStructAnnotationSettings);
    predefined.append(new AnnotationSettings(BioStruct3D::AlphaHelixAnnotationTag, true, QColor(102,255, 0), true));
    predefined.append(new AnnotationSettings(BioStruct3D::BetaStrandAnnotationTag, true, QColor(255,255,153), true));
    predefined.append(new AnnotationSettings(BioStruct3D::TurnAnnotationTag, true, QColor(255,85,127), true));
    predefined.append(new AnnotationSettings(BioStruct3D::MoleculeAnnotationTag, false, QColor(0,255,0), false));
    changeSettings(predefined, false);
}

//////////////////////////////////////////////////////////////////////////
AnnotationSettings::AnnotationSettings() {
    amino = false;
    color = Qt::black;
    visible = true;
}

AnnotationSettings::AnnotationSettings(const QString& _name, bool _amino, const QColor& _color, bool _visible) 
: name(_name), color(_color), amino(_amino), visible(_visible)
{
}

bool AnnotationSettings::equals(const AnnotationSettings* as) const {
    return name == as->name 
        && amino == as->amino
        && color == as->color
        && visible == as->visible
        && nameQuals == as->nameQuals;
}


}//namespace
