/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "PWMatrixTests.h"
#include "DocumentModelTests.h"
#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>
#include <core_api/DocumentModel.h>
#include <core_api/GObject.h>

#include <util_tasks/LoadDocumentTask.h>

#include <datatype/PFMatrix.h>

#include <gobjects/MAlignmentObject.h>
#include <gobjects/DNASequenceObject.h>

namespace GB2 {

    /* TRANSLATOR GB2::GTest */

#define OBJ_ATTR    "obj"
#define OBJ_TYPE    "filetype"
#define OBJ_SIZE    "size"
#define OBJ_LENGTH  "length"
#define PFM_TYPE	"pfmtype"
#define EXPECTED	"expected-values"

//---------------------------------------------------------------------
void GTest_PFMtoPWMConvertTest::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);
    Q_UNUSED(el);
}

Task::ReportResult GTest_PFMtoPWMConvertTest::report() {

    return ReportResult_Finished;
}

void GTest_PFMCreateTest::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    objContextName = el.attribute(OBJ_ATTR);
    if (objContextName.isEmpty()) {
        failMissingValue(OBJ_ATTR);
        return;
    }

    objType = el.attribute(OBJ_TYPE);
    if (objContextName.isEmpty()) {
        failMissingValue(OBJ_TYPE);
        return;
    }

    QString buf = el.attribute(OBJ_SIZE);
    if (buf.isEmpty()) {
        failMissingValue(OBJ_SIZE);
        return;
    } 
    bool ok = false;
    size = buf.toInt(&ok);
    if (!ok) {
        failMissingValue(OBJ_SIZE);
        return;
    }

    buf = el.attribute(OBJ_LENGTH);
    if (buf.isEmpty()) {
        failMissingValue(OBJ_LENGTH);
        return;
    } 
    ok = false;
    length = buf.toInt(&ok);
    if (!ok) {
        failMissingValue(OBJ_LENGTH);
        return;
    }

    buf = el.attribute(PFM_TYPE);
    if (buf.isEmpty()) {
        failMissingValue(PFM_TYPE);
        return;
    } 
    int table_size;
    if (buf == "mononucleotide") {
        type = PFM_MONONUCLEOTIDE;
        table_size = 4;
    } else if (buf == "dinucleotide") {
        type = PFM_DINUCLEOTIDE;
        table_size = 16;
    } else {
        failMissingValue(PFM_TYPE);
        return;
    }
    buf = el.attribute(EXPECTED);
    if (buf.isEmpty()) {
        failMissingValue(EXPECTED);
        return;
    }
    QStringList rows = buf.split(";");
    if (rows.size() != table_size) {
        failMissingValue(EXPECTED);
        return;
    }
    for (int i = 0; i < table_size; i++) {
        values[i].resize(type == PFM_MONONUCLEOTIDE ? length : length - 1);
        QStringList curr = rows[i].split(",");
        if (curr.size() != values[i].size()) {
            failMissingValue(EXPECTED);
            return;
        }
        for (int j = 0, n = curr.size(); j < n; j++) {
            ok = false;
            values[i][j] = curr[j].toInt(&ok);
            if (!ok) {
                failMissingValue(OBJ_LENGTH);
                return;
            }
        }
    }
}

Task::ReportResult GTest_PFMCreateTest::report() {
    if (objType == "aligniment") {
        Document* doc = getContext<Document>(this, objContextName);
        if (doc == NULL) {
            stateInfo.setError(GTest::tr("context not found %1").arg(objContextName));
            return ReportResult_Finished;  
        }

        QList<GObject*> list = doc->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT);
        if (list.size() == 0) {
            stateInfo.setError(GTest::tr("container of object with type \"%1\" is empty").arg(GObjectTypes::MULTIPLE_ALIGNMENT));
            return ReportResult_Finished;  
        }
        MAlignmentObject * myAlign = (MAlignmentObject*)list.first();
        MAlignment al = myAlign->getMAlignment();
        PFMatrix pfm(al, type);
        for (int i = 0, n = (type == PFM_MONONUCLEOTIDE) ? 4 : 16; i < n; i++) {
            for (int j = 0, nn = (type == PFM_MONONUCLEOTIDE) ? length : length - 1; j < nn; j++) {
                if (values[i][j] != pfm.getValue(i, j)) {
                    stateInfo.setError(  QString("Actual results not equal with expected") );
                    return ReportResult_Finished;
                }
            }
        }
    } else if (objType == "sequences") {
        Document* doc = getContext<Document>(this, objContextName);
        if (doc == NULL) {
            stateInfo.setError(GTest::tr("context not found %1").arg(objContextName));
            return ReportResult_Finished;  
        }

        QList<GObject*> list = doc->findGObjectByType(GObjectTypes::SEQUENCE);
        if (list.size() == 0 || list.size() < size) {
            stateInfo.setError(GTest::tr("container of object with type \"%1\" is empty or less than %2").arg(GObjectTypes::MULTIPLE_ALIGNMENT).arg(size));
            return ReportResult_Finished;  
        }
        QList<DNASequence*> data;
        for (int i = 0; i < size; i++) {
            DNASequenceObject* seq = (DNASequenceObject*)list[i];
            if (seq->getSequenceLen() != length) {
                stateInfo.setError(QString("wrong length of %1 sequence: %2").arg(i + 1).arg(seq->getSequenceLen()));
                return ReportResult_Finished;  
            }
            if (seq->getAlphabet()->getType() != DNAAlphabet_NUCL) {
                stateInfo.setError(QString("Wrong sequence alphabet"));
                return ReportResult_Finished;  
            }
            DNASequence* s = new DNASequence(QString("%1 sequence").arg(i + 1), seq->getSequence(), seq->getAlphabet());
            data.push_back(s);
        }
        PFMatrix pfm(data, type);
        for (int i = 0, n = (type == PFM_MONONUCLEOTIDE) ? 4 : 16; i < n; i++) {
            for (int j = 0, nn = (type == PFM_MONONUCLEOTIDE) ? length : length - 1; j < nn; j++) {
                if (values[i][j] != pfm.getValue(i, j)) {
                    stateInfo.setError(  QString("Actual results not equal with expected") );
                    return ReportResult_Finished;
                }
            }
        }
    } else {
        stateInfo.setError(QString("wrong value: %1").arg(OBJ_TYPE));
        return ReportResult_Finished;  
    }

    return ReportResult_Finished;
}

//---------------------------------------------------------------------
QList<XMLTestFactory*> PWMatrixTests::createTestFactories() {
    QList<XMLTestFactory*> res;
    res.append(GTest_PFMtoPWMConvertTest::createFactory());
    res.append(GTest_PFMCreateTest::createFactory());
    return res;
}

}
