/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DialogUtils.h"

#include <core_api/AppContext.h>
#include <core_api/DocumentModel.h>
#include <core_api/Settings.h>
#include <core_api/Log.h>
#include <core_api/Task.h>

#include <QtCore/QFileInfo>
#include <QtCore/QDir>
#include <QtCore/QCoreApplication>
#include <QtGui/QMessageBox>

namespace GB2 {

void DialogUtils::showProjectIsLockedWarning(QWidget* p) {
	QMessageBox::critical(p, tr("project_is_locked_title"), tr("project_is_locked_message"), QMessageBox::Ok, QMessageBox::NoButton);
}



QString DialogUtils::prepareFileFilter(const QString& name, const QStringList& exts, bool any, const QStringList& extra) {
    QString line = name + " (";
    foreach(QString ext, exts) {
        line+=" *."+ext;
    }
    foreach(QString ext, exts) {
        foreach(QString s, extra) {
            line+=" *."+ext+s;
        }
    }
    line+=")";
    if (any) {
        line += "\n" + tr("All files") + " (*.*)";
    }
    return line;
}

QString DialogUtils::prepareDocumentsFileFilter(const DocumentFormatId& fid, bool any, const QStringList& extra) {
    DocumentFormat* df = AppContext::getDocumentFormatRegistry()->getFormatById(fid);
    QString result = prepareFileFilter(df->getFormatName(), df->getSupportedDocumentFileExtensions(), any, extra);
    return result;
}

QString DialogUtils::prepareDocumentsFileFilter(bool any, const QStringList& extra) {
    DocumentFormatRegistry* fr = AppContext::getDocumentFormatRegistry();
    QList<DocumentFormatId> ids = fr->getRegisteredFormats();
    QStringList result;
    foreach(DocumentFormatId id , ids) {
        DocumentFormat* df = fr->getFormatById(id);
        result.append(prepareFileFilter(df->getFormatName(), df->getSupportedDocumentFileExtensions(), false, extra));
    }
    result.sort();
    if (any) { 
        result.prepend(tr("All files") + " (*.*)");
    }
    return result.join("\n");
}

QString DialogUtils::prepareDocumentsFileFilter(const DocumentFormatConstraints& c, bool any) {
    QStringList result;

    QList<DocumentFormatId> ids = AppContext::getDocumentFormatRegistry()->getRegisteredFormats();
    foreach(const DocumentFormatId& id, ids) {
        DocumentFormat* df = AppContext::getDocumentFormatRegistry()->getFormatById(id);
        if (df->checkConstraints(c)) {
            result.append(prepareDocumentsFileFilter(id, false));
        }
    }
    result.sort();
    if (any) {
        result.prepend(tr("All files") + " (*.*)");
    }
    return result.join("\n");
}

QString DialogUtils::prepareDocumentsFileFilterByObjType(const GObjectType& t, bool any) {
    DocumentFormatConstraints c;
    c.supportedObjectTypes += t;
    return prepareDocumentsFileFilter(c, any);
}


#define SETTINGS_ROOT QString("gui/")

QString DialogUtils::getLastOpenFileDir(const QString& toolType, const QString& defaultVal) {
    QString key = SETTINGS_ROOT + (toolType.isEmpty() ? "" : toolType + "/") + "lastDir";
    QString defDir = defaultVal;
    if (defDir.isEmpty() && toolType.isEmpty()) {
        defDir = QDir::searchPaths( PATH_PREFIX_DATA ).first() + "/samples";
    }
    QString res = AppContext::getSettings()->getValue(key, defDir).toString();
    return res;
}

void DialogUtils::setLastOpenFileDir(const QString& ld, const QString& toolType) {
    QString key = SETTINGS_ROOT + (toolType.isEmpty() ? "" : toolType + "/") + "lastDir";
    AppContext::getSettings()->setValue(key, ld);
}

LastOpenDirHelper::LastOpenDirHelper(const QString& d, const QString& defaultVal) {
    domain = d;
    dir = DialogUtils::getLastOpenFileDir(domain, defaultVal);
}

LastOpenDirHelper::~LastOpenDirHelper() {
    saveURLDir2LastOpenDir();
}

void LastOpenDirHelper::saveURLDir2LastOpenDir() {
    if (!url.isEmpty()) {
        QString newDir = QFileInfo(url).absoluteDir().absolutePath();
        if (dir != newDir) {
            DialogUtils::setLastOpenFileDir(newDir, domain);
        }
    }
}

} // namespace
