/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <QtGui/QCheckBox>
#include <QtGui/QLabel>
#include <QtGui/QMessageBox>
#include <QtGui/QFileDialog>

#include <core_api/AppContext.h>
#include <core_api/Log.h>
#include <util_gui/DialogUtils.h>
#include <distributed_computing/DistributedComputingUtil.h>

#include "RemoteMachineScanDialogImpl.h"
#include "RemoteMachineAddDialogImpl.h"
#include "RemoteMachineModifyDialogImpl.h"
#include "RemoteMachineMonitorDialogImpl.h"

#define REMOTE_MACHINE_MONITOR_DLG_LOG_CAT "Remote machines dialog"

namespace GB2 {

static LogCategory log( REMOTE_MACHINE_MONITOR_DLG_LOG_CAT );

const QString RemoteMachineMonitorDialogImpl::OK_BUTTON_RUN = RemoteMachineMonitorDialogImpl::tr( "Run" );
const QString RemoteMachineMonitorDialogImpl::SAVE_SETTINGS_FILE_DOMAIN = "remote_machine_settings_save_settings";

RemoteMachineMonitorDialogImpl::RemoteMachineMonitorDialogImpl( QWidget * p, const QList< RemoteMachineMonitorItem > & monitorItems, 
                                                                const QString & taskId )
: QDialog( p ), taskFactoryId( taskId ), 
PING_YES( ":core/images/remote_machine_ping_yes.png" ), PING_NO( ":core/images/remote_machine_ping_no.png" ), 
PING_WAIT_FOR_RESPONSE( ":core/images/remote_machine_ping_waiting_response.png" ),
getPublicMachinesTask( NULL ) {
    setupUi( this );
    
    int sz = monitorItems.size();
    for( int i = 0; i < sz; ++i ) {
        RemoteMachineMonitorItem item = monitorItems.at( i );
        addMachine( item.machine, MACHINE_OLD, item.selected );
    }
    
    connect( scanPushButton, SIGNAL( clicked() ), SLOT( sl_scanPushButtonClicked() ) );
    connect( okPushButton, SIGNAL( clicked() ), SLOT( sl_okPushButtonClicked() ) );
    connect( cancelPushButton, SIGNAL( clicked() ), SLOT( sl_cancelPushButtonClicked() ) );
    connect( addPushButton, SIGNAL( clicked() ), SLOT( sl_addPushButtonClicked() ) );
    connect( removePushButton, SIGNAL( clicked() ), SLOT( sl_removePushButtonClicked() ) );
    connect( modifyPushButton, SIGNAL( clicked() ), SLOT( sl_modifyPushButtonClicked() ) );
    connect( machinesTreeWidget, SIGNAL( itemSelectionChanged() ), SLOT( sl_selectionChanged() ) );
    connect( this, SIGNAL( rejected() ), SLOT( sl_rejected() ) );
    connect( pingPushButton, SIGNAL( clicked() ), SLOT( sl_pingPushButtonClicked() ) );
    connect( getPublicMachinesButton, SIGNAL( clicked() ), SLOT( sl_getPublicMachinesButtonClicked() ) );
    
    okPushButton->setDefault( true );
    machinesTreeWidget->headerItem()->setText( 0, "" );
    
    QHeaderView * header = machinesTreeWidget->header();
    header->setClickable( false );
    header->setStretchLastSection( false );
    header->setResizeMode( 1, QHeaderView::Stretch );
    
    if( !taskFactoryId.isEmpty() ) {
        okPushButton->setText( OK_BUTTON_RUN );
    }

    initMachineActionsMenu();
}

void RemoteMachineMonitorDialogImpl::initMachineActionsMenu() {
    machinesTreeWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    connect(machinesTreeWidget, SIGNAL(customContextMenuRequested(const QPoint &)), SLOT(sl_machinesTreeMenuRequested(const QPoint&)));
    
    machineActionsMenu = new QMenu(this);
    QAction * removeMachineAction = machineActionsMenu->addAction(removePushButton->text());
    connect(removeMachineAction, SIGNAL(triggered()), SLOT(sl_removePushButtonClicked()));
    QAction * modifyMachineAction = machineActionsMenu->addAction(modifyPushButton->text());
    connect(modifyMachineAction, SIGNAL(triggered()), SLOT(sl_modifyPushButtonClicked()));
    QAction * pingMachineAction = machineActionsMenu->addAction(pingPushButton->text());
    connect(pingPushButton, SIGNAL(triggered()), SLOT(sl_pingPushButtonClicked()));
    QAction * saveMachineAction = machineActionsMenu->addAction(tr("Save machine..."));;
    connect(saveMachineAction, SIGNAL(triggered()), SLOT(sl_saveMachine()));
}

bool RemoteMachineMonitorDialogImpl::addMachine( RemoteMachineSettings * settings, RemoteMachineState state, bool select ) {
    assert( NULL != settings );
    if( hasSameMachineInTheView( settings ) ) {
        log.message( LogLevel_INFO, 
            tr( "tried to add %1 machine to view. it already exists in the view" ).arg( settings->toString() ) );
        return false;
    }
    QCheckBox * checkBox = new QCheckBox();
    checkBox->setChecked( select );
    
    RemoteMachineMonitorDialogItem item( settings, state, checkBox );
    machinesItemsByOrder << item;
    QTreeWidgetItem * widgetItem = addItemToTheView( item );
    assert( NULL != widgetItem );
    pingMachine( settings, widgetItem );
    
    return true;
}

QTreeWidgetItem * RemoteMachineMonitorDialogImpl::addItemToTheView( RemoteMachineMonitorDialogItem & item ) {
    assert( NULL != item.settings );
    QStringList strings;
    strings << QString( "" ) << item.settings->toString() << item.settings->getProtocolId();
    QTreeWidgetItem * widgetItem = new QTreeWidgetItem( machinesTreeWidget );
    
    machinesTreeWidget->setItemWidget( widgetItem, 1, new QLabel( item.settings->toString() ) );
    machinesTreeWidget->setItemWidget( widgetItem, 2, new QLabel( item.settings->getProtocolId() ) );
    machinesTreeWidget->setItemWidget( widgetItem, 3, new QLabel );
    
    connect( item.cb, SIGNAL( stateChanged( int ) ), SLOT( sl_checkBoxStateChanged( int ) ) );
    enableItem( widgetItem, item.cb->isChecked() );
    
    machinesTreeWidget->addTopLevelItem( widgetItem );
    resizeTreeWidget();
    
    return widgetItem;
}

bool RemoteMachineMonitorDialogImpl::hasSameMachineInTheView( RemoteMachineSettings * suspect ) const {
    int sz = machinesItemsByOrder.size();
    int i = 0;
    for( ; i < sz; ++i ) {
        RemoteMachineMonitorDialogItem item = machinesItemsByOrder.at( i );
        if( MACHINE_DELETED != item.state && *item.settings == *suspect ) {
            return true;
        }
    }
    return false;
}

QList< RemoteMachineMonitorDialogItem > RemoteMachineMonitorDialogImpl::getModel() const {
    return machinesItemsByOrder;
}

void RemoteMachineMonitorDialogImpl::sl_scanPushButtonClicked() {
    RemoteMachineScanDialogImpl scanDlg;
    int rc = scanDlg.exec();
    if( QDialog::Rejected == rc ) {
        return;
    }
    assert( QDialog::Accepted == rc );
    
    RemoteMachineScanDialogModel scanDlgModel = scanDlg.getModel();
    foreach( RemoteMachineSettings * scannedMachine, scanDlgModel ) {
        if( !addMachine( scannedMachine, MACHINE_NEW, false ) ) {
            delete scannedMachine; /* it already exists in the monitor list */
        }
    }
}

void RemoteMachineMonitorDialogImpl::sl_okPushButtonClicked() {
    int sz = machinesItemsByOrder.size();
    for( int i = 0; i < sz; ++i  ) {
        const RemoteMachineMonitorDialogItem & item = machinesItemsByOrder.at( i );
        QTreeWidgetItem * treeItem = machinesTreeWidget->topLevelItem( i );
        if( item.cb != qobject_cast<QCheckBox*>( machinesTreeWidget->itemWidget( treeItem, 0 ) ) ) {
            item.cb->setChecked( false );
        }
    }
    machinesItemsByOrder << deletedOldMachines;
    accept();
}

void RemoteMachineMonitorDialogImpl::sl_cancelPushButtonClicked() {
    reject();
}

void RemoteMachineMonitorDialogImpl::sl_rejected() {
    int sz = machinesItemsByOrder.size();
    int i = 0;
    for( ; i < sz; ++i ) {
        RemoteMachineMonitorDialogItem item = machinesItemsByOrder.at( i );
        if( MACHINE_NEW == item.state ) {
            delete item.settings;
        }
    }
}

void RemoteMachineMonitorDialogImpl::sl_addPushButtonClicked() {
    RemoteMachineAddDialogImpl addDlg;
    int rc = addDlg.exec();
    if( QDialog::Rejected == rc ) {
        return;
    }
    assert( QDialog::Accepted == rc );
    
    RemoteMachineSettings * newMachine = addDlg.takeAddedMachine();
    if( NULL == newMachine ) {
        return;
    }
    if( !addMachine( newMachine, MACHINE_NEW, false ) ) {
        delete newMachine; /* it already exists in the list */
    }
}

void RemoteMachineMonitorDialogImpl::sl_modifyPushButtonClicked() {
    assert( 1 == topLevelItemsSelectedNum() );
    int row = getSelectedTopLevelRow();
    assert( 0 <= row && row < machinesItemsByOrder.size() );
    
    if( pingingItems.values().contains( machinesTreeWidget->topLevelItem( row ) ) ) {
        QMessageBox::critical( this, tr( "Error!" ), tr( "Cannot modify machine that is waiting for response" ) );
        return;
    }
    
    RemoteMachineModifyDialogImpl modifyDlg( machinesItemsByOrder.at( row ).settings );
    int rc = modifyDlg.exec();
    if( QDialog::Rejected == rc ) {
        return;
    }
    assert( QDialog::Accepted == rc );
    
    RemoteMachineSettings * newMachine = modifyDlg.takeNewModifiedMachine();
    if( NULL == newMachine ) {
        return;
    }
    
    removeDialogItemAt( row );
    addMachine( newMachine, MACHINE_NEW, false );
}

void RemoteMachineMonitorDialogImpl::sl_removePushButtonClicked() {
    assert( 1 == topLevelItemsSelectedNum() );
    bool ok = removeDialogItemAt( getSelectedTopLevelRow() );
    if( !ok ) {
        QMessageBox::critical( this, tr( "Error!" ), tr( "Cannot delete machine that is waiting for response" ) );
        return;
    }
    
    checkUuids();
}

bool RemoteMachineMonitorDialogImpl::removeDialogItemAt( int row ) {
    assert( 0 <= row && row < machinesItemsByOrder.size() );
    RemoteMachineMonitorDialogItem & itemToRemove = machinesItemsByOrder[row];
    QTreeWidgetItem * treeItemToRemove = machinesTreeWidget->topLevelItem( row );
    if( pingingItems.values().contains( treeItemToRemove ) ) {
        return false; /* if selected machine is pinging now -> it's dangerous to remove it */
    }
    
    machinesTreeWidget->takeTopLevelItem( row );
    delete treeItemToRemove;
    if( MACHINE_OLD == itemToRemove.state ) {
        itemToRemove.state = MACHINE_DELETED;
        deletedOldMachines << itemToRemove;
    } else if( MACHINE_NEW == itemToRemove.state ) {
        delete itemToRemove.settings;
    } else {
        assert( false );
    }
    machinesItemsByOrder.removeAt( row );
    
    return true;
}

void RemoteMachineMonitorDialogImpl::sl_selectionChanged() {
    bool activate = 1 == topLevelItemsSelectedNum();
    removePushButton->setEnabled( activate );
    modifyPushButton->setEnabled( activate );
    pingPushButton->setEnabled( activate );
}

int RemoteMachineMonitorDialogImpl::topLevelItemsSelectedNum() const {
    QList< QTreeWidgetItem* > selectedItems = machinesTreeWidget->selectedItems();
    QList< int > rows;
    foreach( QTreeWidgetItem * wi, selectedItems ) {
        assert( NULL != wi );
        int row = machinesTreeWidget->indexOfTopLevelItem( wi );
        if( !wi->parent() && !rows.contains( row ) ) {
            rows << row;
        }
    }
    return rows.size();
}

int RemoteMachineMonitorDialogImpl::getSelectedTopLevelRow() const {
    assert( 1 == topLevelItemsSelectedNum() );
    QList< QTreeWidgetItem* > selection = machinesTreeWidget->selectedItems();
    assert( !selection.isEmpty() );
    return machinesTreeWidget->indexOfTopLevelItem( selection.first() );
}

void RemoteMachineMonitorDialogImpl::sl_checkBoxStateChanged( int state ) {
    checkBoxStateChanged( qobject_cast< QCheckBox* >( sender() ), Qt::Checked == state );
}

void RemoteMachineMonitorDialogImpl::checkBoxStateChanged( QCheckBox * cb, bool enable ) {
    assert( NULL != cb );
    QTreeWidgetItem * treeItem = findTopLevelWidgetItem( cb );
    assert( NULL != treeItem );
    enableItem( treeItem, enable );
}

QTreeWidgetItem * RemoteMachineMonitorDialogImpl::findTopLevelWidgetItem( QCheckBox * suspect ) {
    assert( NULL != suspect );
    int sz = machinesItemsByOrder.size();
    int ind = -1;
    for( int i = 0; i < sz; ++i ) {
        if( machinesItemsByOrder.at( i ).cb == suspect ) {
            ind = i;
        }
    }
    assert( -1 != ind && 0 <= ind && ind < machinesTreeWidget->topLevelItemCount() );
    return machinesTreeWidget->topLevelItem( ind );
}

static void enableWidgetIfYouCan( QWidget * widget, bool enable ) {
    if( NULL != widget ) {
        widget->setEnabled( enable );
    }
}

void RemoteMachineMonitorDialogImpl::enableItem( QTreeWidgetItem * item, bool enable ) {
    assert( NULL != item );
    /* child items may not have 2 and 3 item */
    enableWidgetIfYouCan( machinesTreeWidget->itemWidget( item, 1 ), enable );
    enableWidgetIfYouCan( machinesTreeWidget->itemWidget( item, 2 ), enable );
    int childCount = item->childCount();
    for( int i = 0; i < childCount; ++i ) {
        enableItem( item->child( i ), enable );
    }
}

void RemoteMachineMonitorDialogImpl::sl_retrieveInfoTaskStateChanged() {
    RetrieveRemoteMachineInfoTask * retrieveInfoTask = qobject_cast< RetrieveRemoteMachineInfoTask* >( sender() );
    assert( NULL != retrieveInfoTask );
    if( Task::State_Finished != retrieveInfoTask->getState() ) {
        return;
    }
    
    RemoteMachine * machine = retrieveInfoTask->takeMachine();
    assert( NULL != machine );
    QTreeWidgetItem * treeItem = pingingItems.value( machine );
    
    delete machine;
    pingingItems.remove( machine );
    
    int row = machinesTreeWidget->indexOfTopLevelItem( treeItem );
    if( -1 == row ) {
        return; /* item was deleted from the table */
    }
    
    QLabel * pingLabel = qobject_cast< QLabel* >( machinesTreeWidget->itemWidget( treeItem, 3 ) );
    assert( NULL != pingLabel );
    pingLabel->setPixmap( retrieveInfoTask->isPinging() ? PING_YES : PING_NO );
    
    RemoteMachineMonitorDialogItem & item = machinesItemsByOrder[row];
    
    if( retrieveInfoTask->hasErrors() ) {
        log.error( tr( "Retrieve info task for machine %1 finished with error: '%2'" ).
            arg( item.settings->toString() ).arg( retrieveInfoTask->getError() ) );
        item.cb->setChecked( false );
        checkBoxStateChanged( item.cb, item.cb->isChecked() );
        return;
    }
    
    if( item.servicesList.isEmpty() ) {
        item.servicesList = DistributedComputingUtil::filterRemoteMachineServices( retrieveInfoTask->getServicesList() );
        item.machineUuid = retrieveInfoTask->getUuid();
        item.hostname = retrieveInfoTask->getHostName();
        processNewItemInfo( item );    
    }
    checkUuids();
}

void RemoteMachineMonitorDialogImpl::processNewItemInfo( const RemoteMachineMonitorDialogItem & item ) {
    QTreeWidgetItem * treeItem = findTopLevelWidgetItem( item.cb );
    assert( NULL != treeItem );
    
    /* add hostname to item's name */
    QLabel * machineNameLabel = qobject_cast< QLabel* >( machinesTreeWidget->itemWidget( treeItem, 1 ) );
    assert( NULL != machineNameLabel );
    machineNameLabel->setText( item.hostname + " (" + machineNameLabel->text() + ")" );
    
    /* add services as item's children */
    foreach( const QString & serviceName, item.servicesList ) {
        QTreeWidgetItem * child = new QTreeWidgetItem( treeItem );
        machinesTreeWidget->setItemWidget( child, 1, new QLabel( serviceName ) );
        treeItem->addChild( child );
    }
    
    if( !taskFactoryId.isEmpty() && !item.servicesList.contains( taskFactoryId ) ) {
        enableItem( treeItem, false );
        return;
    }
    
    machinesTreeWidget->setItemWidget( treeItem, 0, item.cb );
    resizeTreeWidget();
    
    /* we need to enable/disable children */
    checkBoxStateChanged( item.cb, item.cb->isChecked() );
}

void RemoteMachineMonitorDialogImpl::sl_pingPushButtonClicked() {
    assert( 1 == topLevelItemsSelectedNum() );
    int row = getSelectedTopLevelRow();
    pingMachine( machinesItemsByOrder.at( row ).settings, machinesTreeWidget->topLevelItem( row ) );
}

void RemoteMachineMonitorDialogImpl::pingMachine( RemoteMachineSettings * settings, QTreeWidgetItem * item ) {
    assert( NULL != settings && NULL != item );
    if( pingingItems.values().contains( item ) ) {
        log.message( LogLevel_INFO, tr( "tried to ping machine that is pinging now" ) );
        return;
    }
    
    RemoteMachine * machine = AppContext::getProtocolInfoRegistry()->
        getProtocolInfo( settings->getProtocolId() )->getRemoteMachineFactory()->createInstance( settings );
    assert( NULL != machine );
    
    pingingItems.insert( machine, item );
    QLabel * pingLabel = qobject_cast< QLabel* >( machinesTreeWidget->itemWidget( item, 3 ) );
    assert( NULL != pingLabel );
    pingLabel->setPixmap( PING_WAIT_FOR_RESPONSE );
    
    RetrieveRemoteMachineInfoTask * retrieveInfoTask = new RetrieveRemoteMachineInfoTask( machine );
    connect( retrieveInfoTask, SIGNAL( si_stateChanged() ), SLOT( sl_retrieveInfoTaskStateChanged() ) );
    AppContext::getTaskScheduler()->registerTopLevelTask( retrieveInfoTask );
}

QString RemoteMachineMonitorDialogImpl::hasEqualUuids( RemoteMachineSettings * i, RemoteMachineSettings * j ) const {
    assert( NULL != i && NULL != j );
    return i->toString() + tr( " and " ) + j->toString() + tr( " seems to be the same machines. " ) + QString( "\n" );
}

void RemoteMachineMonitorDialogImpl::checkUuids() const {
    QString res;
    
    int sz  = machinesItemsByOrder.size();
    for( int i = 0; i < sz; ++i ) {
        for( int j = i + 1; j < sz; ++j ) {
            const RemoteMachineMonitorDialogItem & iitem = machinesItemsByOrder.at( i );
            const RemoteMachineMonitorDialogItem & jitem = machinesItemsByOrder.at( j );
            const QUuid & iuuid = iitem.machineUuid;
            const QUuid & juuid = jitem.machineUuid;
            if( !iuuid.isNull() && !juuid.isNull() && iuuid == juuid ) {
                res += hasEqualUuids( iitem.settings, jitem.settings );
            }
        }
    }
    log.message( LogLevel_INFO, res );
}

void RemoteMachineMonitorDialogImpl::resizeTreeWidget() {
    QHeaderView * header = machinesTreeWidget->header();
    header->resizeSections( QHeaderView::ResizeToContents );
    header->setResizeMode( 1, QHeaderView::Stretch );
    if( machinesItemsByOrder.size() ) {
        header->resizeSection( 0, machinesItemsByOrder.first().cb->sizeHint().width() * CHECKBOX_SIZE_HINT_MAGIC_NUMBER );
    }
}

void RemoteMachineMonitorDialogImpl::sl_getPublicMachinesButtonClicked() {
    if( NULL != getPublicMachinesTask ) {
        log.trace( tr( "request to public machines already sent" ) );
        return;
    }
    
    getPublicMachinesTask = new RetrievePublicMachinesTask();
    connect( getPublicMachinesTask, SIGNAL( si_stateChanged() ), SLOT( sl_getPublicMachinesTaskStateChanged() ) );
    AppContext::getTaskScheduler()->registerTopLevelTask( getPublicMachinesTask );
}

void RemoteMachineMonitorDialogImpl::sl_getPublicMachinesTaskStateChanged() {
    assert( NULL != getPublicMachinesTask && getPublicMachinesTask == sender() );
    if( Task::State_Finished != getPublicMachinesTask->getState() ) {
        return;
    }
    
    if( getPublicMachinesTask->hasErrors() ) {
        log.trace( tr( "Retrieve public machines failed with error:%1" ).arg( getPublicMachinesTask->getError() ) );
        return;
    }
    
    QList< RemoteMachineSettings* > newMachines = getPublicMachinesTask->takePublicMachines();
    foreach( RemoteMachineSettings * machine, newMachines ) {
        if( !addMachine( machine, MACHINE_NEW, false ) ) {
            delete machine; /* it already exists in the monitor list */
        }
    }
    
    getPublicMachinesTask = NULL; // don't worry, task scheduler will delete it
}

void RemoteMachineMonitorDialogImpl::sl_machinesTreeMenuRequested(const QPoint& p) {
    QTreeWidgetItem * item = machinesTreeWidget->itemAt(p);
    if( item == NULL || item->parent() != NULL ) {
        return;
    }
    machineActionsMenu->exec(QCursor::pos());
}

void RemoteMachineMonitorDialogImpl::sl_saveMachine() {
    assert(topLevelItemsSelectedNum() == 1);
    RemoteMachineMonitorDialogItem & item = machinesItemsByOrder[getSelectedTopLevelRow()];
    QString filename;
    LastOpenDirHelper dirHelper(SAVE_SETTINGS_FILE_DOMAIN);
    dirHelper.url = filename = QFileDialog::getSaveFileName( this, tr("Select a file to save"), dirHelper.dir);
    AppContext::getTaskScheduler()->registerTopLevelTask(new SaveRemoteMachineSettings(item.settings, filename));
}

} // GB2
