/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_MSA_CONSENSUS_ALGORITHM_H_
#define _GB2_MSA_CONSENSUS_ALGORITHM_H_

#include <core_api/core_api.h>
#include <core_api/LRegion.h>

namespace GB2 {

class MAlignment;
class MSAConsensusAlgorithm;
class DNAAlphabet;

enum ConsensusAlgorithmFlag {
    ConsensusAlgorithmFlag_Nucleic = 1 << 0,
    ConsensusAlgorithmFlag_Amino = 1 << 1,
    ConsensusAlgorithmFlag_Raw = 1 << 2,
    ConsensusAlgorithmFlag_SupportThreshold = 1 << 3,
};

typedef QFlags<ConsensusAlgorithmFlag> ConsensusAlgorithmFlags;
#define ConsensusAlgorithmFlags_AllAlphabets (ConsensusAlgorithmFlags(ConsensusAlgorithmFlag_Nucleic) | ConsensusAlgorithmFlag_Amino | ConsensusAlgorithmFlag_Raw)
#define ConsensusAlgorithmFlags_NuclAmino    (ConsensusAlgorithmFlags(ConsensusAlgorithmFlag_Nucleic) | ConsensusAlgorithmFlag_Amino)

class GB2_COREAPI_EXPORT MSAConsensusAlgorithmFactory : public QObject {
    Q_OBJECT
public:
    MSAConsensusAlgorithmFactory(const QString& algoId, ConsensusAlgorithmFlags flags, QObject* p = NULL);
    
    virtual MSAConsensusAlgorithm* createAlgorithm(const MAlignment& ma, QObject* parent = NULL) = 0;
    
    QString getId() const {return algorithmId;}

    ConsensusAlgorithmFlags getFlags() const {return flags;}

    virtual QString getDescription() const = 0;

    virtual QString getName() const = 0;

    virtual bool supportsThreshold() const {return flags.testFlag(ConsensusAlgorithmFlag_SupportThreshold);}

    virtual LRegion getThresholdRange() const {return LRegion(0, 0);}

    virtual int getDefaultThreshold() const {return 0;}

    virtual QString getThresholdSuffix() const {return QString();}

    // utility method
    static ConsensusAlgorithmFlags getAphabetFlags(const DNAAlphabet* al);

private:
    QString                 algorithmId;
    ConsensusAlgorithmFlags flags;

};

class GB2_COREAPI_EXPORT MSAConsensusAlgorithm : public QObject {
	Q_OBJECT
public:
	MSAConsensusAlgorithm(MSAConsensusAlgorithmFactory* factory, QObject* p = NULL);

	/** 
		Returns consensus char and score for the given row
		Score is a number: [0, num] sequences. Usually is means count of the char in the row
		Note that consensus character may be out of the to MSA alphabet symbols range
	*/
	virtual char getConsensusCharAndScore(const MAlignment& ma, int column, int& score) const;

	virtual char getConsensusChar(const MAlignment& ma, int column) const = 0;
    
    virtual QString getDescription() const {return factory->getDescription();}

    virtual QString getName() const {return factory->getName();}

    virtual void setThreshold(int val);
    
    virtual int getThreshold() const {return threshold;}

    bool supportsThreshold() const {return factory->supportsThreshold();}

    virtual LRegion getThresholdRange() const {return factory->getThresholdRange();}

    virtual int getDefaultThreshold() const {return factory->getDefaultThreshold();}

    virtual QString getThresholdSuffix() const {return factory->getThresholdSuffix();}

    QString getId() const {return factory->getId();}

    MSAConsensusAlgorithmFactory* getFactory() const {return factory;}

signals:
    void si_thresholdChanged(int);

private:
	MSAConsensusAlgorithmFactory* factory;
    int threshold;

};

}//namespace

#endif
