/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "EditAnnotationDialogController.h"

#include <qmessagebox.h>

#include <document_format/GenbankLocationParser.h>
#include <document_format/GenbankFeatures.h>

#include <util_gui/CreateAnnotationWidgetController.h>

/* TRANSLATOR GB2::EditAnnotationDialogController */

namespace GB2 {

EditAnnotationDialogController::EditAnnotationDialogController( Annotation* a, LRegion _seqRange, QWidget* p)
: QDialog(p), seqRange(_seqRange) {

    setupUi(this);

    nameEdit->setText(a->getAnnotationName());
    locationEdit->setText(Genbank::LocationParser::buildLocationString(a->data()));
    compliment = a->isOnComplementStrand();

    QMenu* menu = EditAnnotationDialogController::createAnnotationNamesMenu(this, this);
    showNameGroupsButton->setMenu(menu);
    showNameGroupsButton->setPopupMode(QToolButton::InstantPopup);

    connect(locationEdit, SIGNAL(textChanged(const QString&)), SLOT(sl_onTextChanged(const QString&)));
    connect(complementButton, SIGNAL(clicked()), SLOT(sl_complementLocation()));
    connect(locationEdit, SIGNAL(returnPressed()), SLOT(accept()));
    connect(nameEdit, SIGNAL(returnPressed()), SLOT(accept()));
}

void EditAnnotationDialogController::sl_onTextChanged(const QString& changedText){
    currentRegionList.clear();
    QByteArray locEditText = changedText.toAscii();
    Genbank::LocationParser::parseLocation(locEditText.constData(), changedText.length(), compliment, currentRegionList);
    if (currentRegionList.isEmpty()) {
        if(changedText.isEmpty()){
            statusLabel->setText("<font color=\"#FF0000\">" + tr("Location empty!") + "</font>");
        }else{
            statusLabel->setText("<font color=\"#FF0000\">" + tr("Location invalid!") + "</font>");
        }
    }else{
        statusLabel->setText("");
    }
}

void EditAnnotationDialogController::accept(){
    currentRegionList.clear();
    QByteArray locEditText = locationEdit->text().toAscii();
    Genbank::LocationParser::parseLocation(locEditText.constData(), locationEdit->text().length(), compliment, currentRegionList);

	bool validRegions = true;
	foreach(LRegion reg, currentRegionList){
		if(!seqRange.contains(reg)){
			validRegions = false;
		}
	}
	
    if ( currentRegionList.isEmpty() || (nameEdit->text()).isEmpty() ) {
        QDialog::reject();
	}else if (!Annotation::isValidAnnotationName(nameEdit->text())) {
		QMessageBox::critical( this, tr( "Error!" ), tr( "Wrong annotation name!" ) );
	}else if(!validRegions){
		QMessageBox::critical( this, tr( "Error!" ), tr( "New annotation locations is out of sequence bounds!" ) );
	}else{
        currentName = nameEdit->text();
        QDialog::accept();
    }
}

static bool caseInsensitiveLessThan(const QString &s1, const QString &s2) {
    return s1.toLower() < s2.toLower();
}

QMenu* EditAnnotationDialogController::createAnnotationNamesMenu( QWidget* p, QObject* receiver )
{
    assert(p!=NULL && receiver!=NULL);

    QMenu* m = new QMenu(p);
    const QMultiMap<QString, GBFeatureKey>& nameGroups = GBFeatureUtils::getKeyGroups();
    QStringList groupNames = nameGroups.uniqueKeys();
    qSort(groupNames.begin(), groupNames.end(), caseInsensitiveLessThan);
    foreach(const QString& groupName, groupNames) {
        QMenu* groupMenu = m->addMenu(groupName);
        QList<GBFeatureKey> keys = nameGroups.values(groupName);
        QStringList names;
        foreach(GBFeatureKey k, keys) {
            names.append(GBFeatureUtils::getKeyInfo(k).text);
        }
        qSort(names.begin(), names.end(), caseInsensitiveLessThan);
        foreach(const QString& name, names) {
            QAction* a = new QAction(name, groupMenu);
            connect(a, SIGNAL(triggered()), receiver, SLOT(sl_setPredefinedAnnotationName()));
            groupMenu->addAction(a);
        }
    }
    return m;
}

void EditAnnotationDialogController::sl_setPredefinedAnnotationName() {
    QAction* a = qobject_cast<QAction*>(sender());
    QString text = a->text();
    nameEdit->setText(text);
}

void EditAnnotationDialogController::sl_complementLocation() {
    QString text = locationEdit->text();
    if (text.startsWith("complement(") && text.endsWith(")")) {
        locationEdit->setText(text.mid(11, text.length()-12));
    } else {
        locationEdit->setText("complement(" + text + ")");
    }
}

}// ns

