/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MSAEditorConsensusCache.h"

#include <gobjects/MAlignmentObject.h>
#include <util_msa_consensus/MSAConsensusAlgorithm.h>

namespace GB2 {

MSAEditorConsensusCache::MSAEditorConsensusCache(QObject* p, MAlignmentObject* o, MSAConsensusAlgorithmFactory* factory) 
: QObject(p), currentVersion(1), aliObj(o), algorithm(NULL)
{
    setConsensusAlgorithm(factory);
    
	connect(aliObj, SIGNAL(si_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)), 
        SLOT(sl_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)));

    cache.resize(aliObj->getMAlignment().getLength());
}

MSAEditorConsensusCache::~MSAEditorConsensusCache() {
    delete algorithm;
    algorithm = NULL;
}

void MSAEditorConsensusCache::setConsensusAlgorithm(MSAConsensusAlgorithmFactory* factory) {
	delete algorithm;
    algorithm = NULL;
	algorithm = factory->createAlgorithm(aliObj->getMAlignment());
    connect(algorithm, SIGNAL(si_thresholdChanged(int)), SLOT(sl_thresholdChanged(int)));
	currentVersion++;
}

void MSAEditorConsensusCache::sl_alignmentChanged(const MAlignment&, const MAlignmentModInfo&) {
    cache.resize(aliObj->getMAlignment().getLength());
    currentVersion++;
}

void MSAEditorConsensusCache::updateCacheItem(int pos) {
    const MAlignment& ma = aliObj->getMAlignment();
    assert(pos >= 0 && pos < ma.getLength());
    assert(pos < cache.size() && cache.size() == ma.getLength());
    const CacheItem& cci = cache[pos];
    if (cci.version == currentVersion) {
        return;
    }
    CacheItem& ci = cache[pos];
    int count = 0;
    int nSeq = ma.getNumRows();
	
	ci.topChar = algorithm->getConsensusCharAndScore(ma, pos, count);
    ci.topPercent = (char)qRound(count * 100 / nSeq);
    assert(ci.topPercent >=0 && ci.topPercent<=100);
    ci.version = currentVersion;
}

char MSAEditorConsensusCache::getConsensusChar(int pos) {
    updateCacheItem(pos);
    const CacheItem& ci = cache[pos];
    return ci.topChar;
}

int MSAEditorConsensusCache::getConsensusCharPercent(int pos) {
    updateCacheItem(pos);
    const CacheItem& ci = cache[pos];
    return ci.topPercent;
}

QByteArray MSAEditorConsensusCache::getConsensusLine() {
    QByteArray res;
    const MAlignment& ma = aliObj->getMAlignment();
    for (int i=0, n = ma.getLength(); i<n; i++) {
        char c = getConsensusChar(i);
        if (c!=MAlignment_GapChar) {
            res.append(c);
        }
    }
    return res;
}

void MSAEditorConsensusCache::sl_thresholdChanged(int newValue) {
    Q_UNUSED(newValue);
    currentVersion++;
}

}//namespace

