/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MSAEditorFactory.h"
#include "MSAEditor.h"
#include "MSAEditorTasks.h"
#include "MSAEditorState.h"

#include <core_api/Log.h>
#include <core_api/DocumentModel.h>
#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>

#include <gobjects/MAlignmentObject.h>
#include <gobjects/UnloadedObject.h>

#include <selection/SelectionUtils.h>

namespace GB2 {

/* TRANSLATOR GB2::MSAEditor */
/* TRANSLATOR GB2::ObjectViewTask */

static LogCategory log(ULOG_CAT_MSA);

const GObjectViewFactoryId MSAEditorFactory::ID("MSAEditor");

MSAEditorFactory::MSAEditorFactory() 
: GObjectViewFactory(ID, tr("Alignment editor"))
{
}

bool MSAEditorFactory::canCreateView(const MultiGSelection& multiSelection) {
	bool hasMSADocuments = !SelectionUtils::findDocumentsWithObjects(
                                GObjectTypes::MULTIPLE_ALIGNMENT, &multiSelection, UOF_LoadedAndUnloaded, true).isEmpty();
    if (hasMSADocuments) {
        return true;
    }
    return false;
}

#define MAX_VIEWS 10

Task* MSAEditorFactory::createViewTask(const MultiGSelection& multiSelection, bool single) {
	QSet<GObject*> msaObjects = SelectionUtils::findObjects(GObjectTypes::MULTIPLE_ALIGNMENT, &multiSelection, UOF_LoadedAndUnloaded);
    QSet<Document*> docsWithMSA = SelectionUtils::findDocumentsWithObjects(GObjectTypes::MULTIPLE_ALIGNMENT, 
                                                            &multiSelection, UOF_LoadedAndUnloaded, false);
	QList<OpenMSAEditorTask*> resTasks;

    foreach(Document* doc, docsWithMSA) {
        QList<GObject*> docObjs = doc->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT, UOF_LoadedAndUnloaded);
        if (!docObjs.isEmpty()) {
            msaObjects+=docObjs.toSet();
        } else {
            resTasks.append(new OpenMSAEditorTask(doc));
            if (resTasks.size() == MAX_VIEWS) {
                break;
            }
        }
    }

    if (!msaObjects.isEmpty()) {
		foreach(GObject* o, msaObjects) {
            if (resTasks.size() == MAX_VIEWS) {
                break;
            }
            if (o->getGObjectType() == GObjectTypes::UNLOADED) {
                resTasks.append(new OpenMSAEditorTask(qobject_cast<UnloadedObject*>(o)));
            } else {
			    assert(o->getGObjectType() == GObjectTypes::MULTIPLE_ALIGNMENT);
			    resTasks.append(new OpenMSAEditorTask(qobject_cast<MAlignmentObject*>(o)));
            }
		}
	}
	
	if (resTasks.isEmpty()) {
		return NULL;
	}

	if (resTasks.size() == 1 || single) {
		return resTasks.first();
	}

	Task* result = new Task(tr("open_multiple_views"), TaskFlag_NoRun);
	foreach(Task* t, resTasks) {
		result->addSubTask(t);
	}
	return result;
}

bool MSAEditorFactory::isStateInSelection(const MultiGSelection& multiSelection, const QVariantMap& stateData) {
	MSAEditorState state(stateData);
	if (!state.isValid()) {
		return false;
	}
	GObjectReference ref = state.getMSAObject();
	Document* doc = AppContext::getProject()->findDocumentByURL(ref.docUrl);
	if (doc == NULL) { //todo: accept to use invalid state removal routines of ObjectViewTask ???
		return false;
	}
	//check that document is in selection
	QList<Document*> selectedDocs = SelectionUtils::getSelectedDocs(multiSelection);
	if (selectedDocs.contains(doc)) {
		return true;
	}
	//check that object is in selection
	QList<GObject*> selectedObjects = SelectionUtils::getSelectedObjects(multiSelection);
	GObject* obj = doc->findGObjectByName(ref.objName);
	bool res = obj!=NULL && selectedObjects.contains(obj);
	return res;
}

Task* MSAEditorFactory::createViewTask(const QString& viewName, const QVariantMap& stateData) {
	return new OpenSavedMSAEditorTask(viewName, stateData);
}

}//namespace

